package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Color;
import java.awt.Component;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.*;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.agent.AgentChannelState;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.plugins.trending.TrendingService;

/**
 * 
 *
 * @author onoprien
 */
abstract public class MonitorTable extends AbstractTableModel {

// -- Fields : -----------------------------------------------------------------
    
    public static final Color COLOR_FG = Color.BLACK; // default foreground
    public static final Color COLOR_BG = Color.WHITE; // default background
    private static final Color COLOR_GOOD = new Color(160, 255, 160);
    private static final Color COLOR_WARN = new Color(255, 255, 100);
    private static final Color COLOR_ERR = new Color(255, 160, 160);
    private static final Color COLOR_OFF = new Color(160, 200, 255);
    private static final Color COLOR_POPUP = new Color(255, 255, 160);
    private static final Color COLOR_MULTI = Color.LIGHT_GRAY;
    private static final Color COLOR_NA = Color.GRAY;

    protected int nRows, nColumns;
    protected Cell[][] cells; // [row][column], covers actual table
    
    static private final List<MonitorField> AFFECTED_BY_STATE = Arrays.asList((new MonitorField[] {MonitorField.VALUE, MonitorField.HIGH, MonitorField.LOW}));

// -- Life cycle : -------------------------------------------------------------
    
    MonitorTable() {
    }
    
    MonitorTable(Cell[][] cells) {
        this.cells = cells;
        nRows = cells.length;
        nColumns = cells[0].length;
    }
    
// -- Implementing TableModel : ------------------------------------------------

    @Override
    public int getRowCount() {
        return nRows;
    }

    @Override
    public int getColumnCount() {
        return nColumns;
    }

    @Override
    public Data getValueAt(int row, int column) {
        return cells[row][column].getData();
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        Cell cell = cells[row][column];
        if (cell.getItems().size() != 1) return false;
        MonitorField field = cell.getField();
        return field != null & field.isEditable();
    }

    @Override
    public void setValueAt(Object aValue, int row, int column) {
        Cell cell = cells[row][column];
        MonitorField field = cell.getField();
        boolean isHigh = MonitorField.HIGH.equals(field);
        if ((isHigh || MonitorField.LOW.equals(field)) && cell.getItems().size() == 1) {
            try {
                double newValue = Double.parseDouble(aValue.toString());
                AgentChannel channel = cell.getItems().get(0).getChannel();
                if (channel != null) {
//                    double oldValue = (Double) (isHigh ? channel.get(MonitorField.HIGH.name()) : channel.get(MonitorField.LOW.name()));
//                    if (Double.isNaN(oldValue) || (Math.abs(oldValue - newValue) > 0.00001)) { // FIXME
                        String target = isHigh ? "limitHi" : "limitLo";
                        Console.getConsole().sendCommand(channel.getAgentName() +"/change", channel.getLocalPath(), target, newValue);
//                    }
                }
            } catch (NumberFormatException x) {
                return;
            }
            super.setValueAt(getValueAt(row, column), row, column);
        } else {
            super.setValueAt(aValue, row, column);
        }
    }
    
    
// -- Updating table model : ---------------------------------------------------

    protected void update(Item item, List<String> attributes) {
        
//        if (!(attributes != null && attributes.size() == 1 && attributes.get(0).equals("value"))) {
//            StringBuilder sb = new StringBuilder("Update ");
//            sb.append(item.getName()).append(": ");
//            if (attributes == null) {
//                sb.append("null");
//            } else {
//                attributes.forEach(a -> sb.append(a).append(" "));
//            }
//            System.out.println(sb);
//        }
        
        HashSet<MonitorField> fields = new HashSet<>();
        if (attributes == null) {
            fields.add(null);
        } else {
            attributes.forEach(att -> {
                List<MonitorField> f = getField(att);
                fields.addAll(f); 
            });
        }
        
        fields.forEach(field -> {
            List<int[]> affectedCells = getCells(item, field);
            for (int[] index : affectedCells) {
                Cell cell = cells[index[0]][index[1]];
                field = cell.getField();
                List<Item> items = cell.getItems();
                int nItems = items.size();
                if (cell.getData() == Data.MULTI) {
                    Data data = format(items.get(0), field);
                    for (int i = 1; i < nItems; i++) {
                        Data other = format(items.get(i), field);
                        if (!data.equals(other)) {
                            data = null;
                            break;
                        }
                    }
                    if (data != null) {
                        cell.setData(data);
                        fireTableCellUpdated(index[0], index[1]);
                    }
                } else {
                    Data data = format(item, field);
                    if (!data.equals(cell.getData())) {
                        cell.setData(data);
                        fireTableCellUpdated(index[0], index[1]);
                    }
                }
            }
        });
        
    }

    /**
     * Returns indices of all cells whose content is affected by the specified item and field.
     * The implementation provided by this class is universal but very inefficient - subclasses
     * handling specific types of tables are expected to override it.
     * 
     * @param item Item.
     * @param field Field. If {@code null}, all cells affected by the specified item are included.
     * @return List of cells.
     */
    protected List<int[]> getCells(Item item, MonitorField field) {
        List<int[]> out = new ArrayList<>();
        for (int row = 0; row < nRows; row++) {
            for (int col = 0; col < nColumns; col++) {
                Cell cell = cells[row][col];
                if (cell.getItems().contains(item) && (field == null || field.equals(cell.getField()))) {
                    out.add(new int[] {row, col});
                }
            }
        }
        return out;
    }
    
    protected List<MonitorField> getField(String attribute) {
        if (MonitorField.STATE_KEY.equals(attribute)) {
            return AFFECTED_BY_STATE;
        } else {
            return Collections.singletonList(MonitorField.valueOf(attribute));
        }
    }
    
// -- Formatting cells : --------------------------------------------------------

    protected Data format(Item item, MonitorField field) {

        if (MonitorField.NAME.equals(field)) {
            return new Data(item.getName(), null, null, SwingConstants.LEFT);
        }

        AgentChannel channel = item.getChannel();
        if (channel == null) {
            return Data.NA;
        }
        Object value = channel.get(field.name());
        if (value == null) {
            return Data.NA;
        }
        
        String text = null;
        Color fgColor = null;
        Color bgColor = null;
        Integer horizontalAlignment = null;

        if (MonitorField.VALUE.equals(field)) {
            if (value instanceof Double) {
                if (((Double) value).isNaN()) {
                    text = "NaN";
                } else {
                    double d = (Double) value;
                    String f = (String) channel.get(MonitorField.FORMAT_KEY);
                    text = String.format((f == null ? "%f " : f), d);
                }
            } else {
                text = value.toString();
            }
            Object o = channel.get(MonitorField.STATE_KEY);
            if (o instanceof AgentChannelState) {
                AgentChannelState state = (AgentChannelState) o;
                if (state.isOnline()) {
                    bgColor = state.isGood() ? COLOR_GOOD : COLOR_ERR;
                } else {
                    bgColor = COLOR_OFF;
                }
            }
            horizontalAlignment = SwingConstants.RIGHT;

        } else if (MonitorField.HIGH.equals(field) || MonitorField.LOW.equals(field)) {
            double d = Double.NaN;
            if (value instanceof Double) {
                d = ((Double)value);
            } else if (value instanceof Float) {
                d = ((Float)value).doubleValue();
            } else if (value instanceof String) {
                try {
                    d = Double.parseDouble((String)value);
                } catch (NumberFormatException x) {
                    text = (String)value;
                }
            }
            if (Double.isNaN(d)) {
                if (text == null) text = "NaN";
            } else {
                String f = (String) channel.get(MonitorField.FORMAT_KEY);
                text = String.format((f == null ? "%f " : f), d);
            }
            Object o = channel.get(MonitorField.STATE_KEY);
            if (o instanceof AgentChannelState) {
                AgentChannelState state = (AgentChannelState) o;
                if (MonitorField.HIGH.equals(field) ? state.isHighLimitChange() : state.isLowLimitChange()) {
                    fgColor = Color.BLUE;
                }
            }

        } else if (MonitorField.LOW_WARN.equals(field) || MonitorField.HIGH_WARN.equals(field)) {
            if (value instanceof Double) {
                if (((Double) value).isNaN()) {
                    text = "NaN";
                } else {
                    double d = (Double) value;
                    String f = (String) channel.get(MonitorField.FORMAT_KEY);
                    text = f == null ? String.valueOf(d) : String.format(f, d);
                }
            } else {
                text = value.toString();
            }

        } else if (MonitorField.ALERT_HIGH.equals(field) || MonitorField.ALERT_LOW.equals(field)) {
            text = "  \u2713";

//            } else if (MonitorField.DESCR.equals(field)) {
//                
//            } else if (MonitorField.UNITS.equals(field)) {
//                
        } else {
            text = value.toString();
        }

        return new Data(text, fgColor, bgColor, horizontalAlignment);
    }

    protected boolean format(Cell cell) {
        MonitorField field = cell.getField();
        List<Item> items = cell.getItems();
        Data newData;
        switch (items.size()) {
            case 0:
                newData = new Data(field.getTitle(), SwingConstants.CENTER);
                break;
            case 1:
                newData = format(items.get(0), field);
                break;
            default:
                newData = format(items.get(0), field);
                for (int i = 1; i < items.size(); i++) {
                    newData = format(items.get(i), field);
                    if (!newData.equals(format(items.get(i), field))) {
                        newData = Data.MULTI;
                        break;
                    }
                }                
        }
        if (newData.equals(cell.getData())) {
            return false;
        } else {
            cell.setData(newData);
            return true;
        }
    }

        
// -- Cell content classes : ---------------------------------------------------

    protected static class Cell {
        
        static protected Cell EMPTY = new Cell(Collections.emptyList(), MonitorField.NULL, Data.EMPTY);
        
        private final List<Item> items;
        private final MonitorField field;
        
        private Data data;
                
        Cell() {
            this(Collections.emptyList(), MonitorField.NULL);
        }
        
        Cell(Item item, MonitorField field) {
            this(Collections.singletonList(item), field);
        }

        Cell(List<? extends Item> items, MonitorField field) {
            this(items, field, null);
        }

        Cell(List<? extends Item> items, MonitorField field, Data data) {
            this.items = (List<Item>) items;
            this.field = field;
            this.data = data;
        }
        
        public List<Item> getItems() {
            return items;
        }
        
        public MonitorField getField() {
            return field;
        }
        
        public void setData(Data data) {
            this.data = data;
        }
        
        public Data getData() {
            return data;
        }
        
    }
    
    protected static class Data {
        
        static final Data MULTI = new Data("-", COLOR_FG, COLOR_MULTI, SwingConstants.CENTER);
        static final Data NA = new Data("", COLOR_FG, COLOR_NA, SwingConstants.CENTER);
        static final Data EMPTY = new Data("", COLOR_FG, COLOR_BG, SwingConstants.CENTER);

        final String text;
        final Color fgColor, bgColor;
        final int horizontalAlignment;
        
        Data(String text, Color fgColor, Color bgColor, Integer horizontalAlignment) {
            this.text = text == null ? "" : text;
            this.fgColor = fgColor == null ? COLOR_FG : fgColor;
            this.bgColor = bgColor == null ? COLOR_BG : bgColor;
            this.horizontalAlignment = horizontalAlignment == null ? SwingConstants.CENTER : horizontalAlignment;
        }
        
        Data() {
            this(null, null, null, null);
        }
        
        Data(String text, int horizontalAlignment) {
            this(text, null, null, horizontalAlignment);
        }

        @Override
        public boolean equals(Object obj) {
            if (!(obj instanceof Data)) return false;
            Data other = (Data) obj;
            return Objects.equals(text, other.text) && Objects.equals(fgColor, other.fgColor) && Objects.equals(bgColor, other.bgColor) && Objects.equals(horizontalAlignment, other.horizontalAlignment);
        }

        @Override
        public int hashCode() {
            final int PRIME = 31;
            int result = 1;
            result = result * PRIME + (text == null ? 0 : text.hashCode());
            result = result * PRIME + (fgColor == null ? 0 : fgColor.hashCode());
            result = result * PRIME + (bgColor == null ? 0 : bgColor.hashCode());
            result = result * PRIME + horizontalAlignment;
            return result;
        }

        @Override
        public String toString() {
            return text;
        }
        
    }
    
    
// -- Monitored channel and its display name : ---------------------------------
    
    public interface Item {

        AgentChannel getChannel();
        
        String getName();
    }
    
    
// -- Customized JTable : ------------------------------------------------------
    
    public JTable makeTable() {
        
        JTable table = new JTable(this);
        
        JTableHeader header = table.getTableHeader();
        header.setReorderingAllowed(false);
        header.setResizingAllowed(false); // no resizing to avoid mismatch between renderer and editor tables
        TableCellRenderer headerRenderer = header.getDefaultRenderer();
        if (headerRenderer instanceof DefaultTableCellRenderer) {
            ((DefaultTableCellRenderer)headerRenderer).setHorizontalAlignment(SwingConstants.CENTER);
        }

        Renderer renderer = new Renderer();
        table.setDefaultRenderer(Object.class, renderer);
        table.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
        table.setRowSelectionAllowed(false);
        table.setColumnSelectionAllowed(false);
        table.setRowHeight(table.getRowHeight() + 2);
        table.setShowGrid(true);
        table.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent evt) {
                tableMouseClicked(evt);
            }
        });

        for (int col = 0; col < nColumns; col++) {
            TableColumn column = table.getColumnModel().getColumn(col);
            Component comp = headerRenderer.getTableCellRendererComponent(null, column.getHeaderValue(), false, false, 0, 0);
            int maxWidth = comp.getPreferredSize().width;
            for (int row=0; row<nRows; row++) {
                comp = renderer.getTableCellRendererComponent(table, getValueAt(row, col), false, false, row, col);
                maxWidth = Math.max(maxWidth, comp.getPreferredSize().width);
            }
            column.setPreferredWidth(maxWidth);
        }
        return table;
    }
    
    private void tableMouseClicked(MouseEvent evt) {
        int nClick = evt.getClickCount();
        JTable table = (JTable) evt.getSource();
        Point point = evt.getPoint();
        int row = table.rowAtPoint(point);
        int column = table.columnAtPoint(point);
        if (nClick == 1) {
            // FIXME
        } else if (nClick == 2) {
            Cell cc = cells[row][column];
            if (cc.getField().equals(MonitorField.VALUE)) {
                List<Item> items = cc.getItems();
                if (items.size() == 1) {
                    Item item = items.get(0);
                    String[] path = {item.getChannel().getAgentName(), item.getChannel().getLocalPath()};
                    TrendingService trending = (TrendingService) Console.getConsole().getConsoleLookup().lookup(TrendingService.class);
                    if (trending == null) {
                        return;
                    }
                    trending.show(path);
                }
            }
        }
    }
    
    static class Renderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            Data cc = (Data) value;
            super.getTableCellRendererComponent(table, cc.text, false, false, row, column);
            setBackground(cc.bgColor);
            setForeground(cc.fgColor);
            setHorizontalAlignment(cc.horizontalAlignment);
            return this;
        }
    }
    
}
