 package org.lsst.ccs.gconsole.plugins.monitor;

import org.lsst.ccs.subsystem.monitor.ui.tree.MonitorExport;
import org.lsst.ccs.subsystem.monitor.ui.tree.MonitorDisplay;
import org.lsst.ccs.subsystem.monitor.ui.tree.Field;
import org.lsst.ccs.subsystem.monitor.ui.tree.Channel;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.*;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.SwingUtilities;
import org.freehep.application.Application;
import org.freehep.application.mdi.PageContext;
import org.freehep.application.mdi.PageEvent;
import org.freehep.application.mdi.PageListener;
import org.freehep.application.mdi.PageManager;
import org.freehep.application.studio.Studio;
import org.freehep.graphicsbase.util.export.ExportDialog;
import org.freehep.util.commanddispatcher.CommandProcessor;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.openide.util.Lookup.Template;

/**
 * Graphical console plugin that supports display of monitoring data.
 * Unless otherwise specified, methods of this class can be called on any thread.
 *
 * @author onoprien
 */
@Plugin(name="LSST Monitoring Plugin",
        id="monitor",
        description="LSST CCS monitoring data display service.")
public class LsstMonitorPlugin extends ConsolePlugin {

// -- Fields : -----------------------------------------------------------------
    
    static public final String DISPLAY_AS_TREE = "displayAsTree";
    static public final String DISPLAY_DOT_SEPARATOR = "displayDotSeparator";
    
    private final Studio app = (Studio) Application.getApplication();
    private final Action newView;
    private final ArrayList<MonitorView> currentViews = new ArrayList<>(0);
    private final ArrayDeque<MonitorPageDialog.Descriptor> recentViews = new ArrayDeque<>(0);

// -- Life cycle : -------------------------------------------------------------
    
    public LsstMonitorPlugin() {
        newView = new AbstractAction("Configure...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                openNewView(null);
            }
        };
    }

    @Override
    public void initialize() {
        
        // Preference to control appearence of legacy monitoring tables:
        
        getServices().addProperty(DISPLAY_AS_TREE, false);
        getServices().addPreference(new String[] {"LSST","Monitor"}, "Data Display", "${"+ DISPLAY_AS_TREE +"}Display tables as trees.");

        // Monitoring menu:
        
        SwingUtilities.invokeLater(this::resetMenu);
        
        // Enabling dumping monitoring data into files:
        
        PageManager pm = app.getPageManager();
        pm.addPageListener(new Saver());
        
    }

    @Override
    public void shutdown() {
    }


// -- Opening monitoring pages : -----------------------------------------------
    
    /**
     * Opens a new graphical console page that displays the specified view.
     * This method should be called on the AWT Event Dispatch Thread.
     * 
     * @param view View to display.
     */
    public void openPage(MonitorView view) {
        PanelManager pm = getConsole().getPanelManager();
        JComponent panel = view.getPanel();
        pm.open(panel, view.getName());
        pm.addListener(e -> {
            if (e.hasKey(Panel.OPEN) && (Boolean) e.getNewValue() == false) {
                view.uninstall(getConsole().getStatusAggregator());
            }
        }, panel);
        view.install(getConsole().getStatusAggregator());
    }
    
    /**
     * Closes the console page that displays the specified view.
     * This method should be called on the AWT Event Dispatch Thread.
     * 
     * @param view View to close.
     */
    public void closePage(MonitorView view) {
        PanelManager pm = getConsole().getPanelManager();
        JComponent panel = view.getPanel();
        if (panel != null) pm.close(panel);
        view.uninstall(getConsole().getStatusAggregator());
    }
    
    /**
     * Registers the specified view for updates.
     * @param view View to update.
     */
    public void addView(MonitorView view) {
        view.install(getConsole().getStatusAggregator());
    }
    
    /**
     * Removes the specified view so it is no longer updated.
     * @param view View to remove.
     */
    public void removeView(MonitorView view) {
        view.uninstall(getConsole().getStatusAggregator());
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void openNewView(MonitorPageDialog.Descriptor desc) {
        MonitorView view;
        if (desc == null) {
            MonitorPageDialog d = MonitorPageDialog.show(LsstMonitorPlugin.this.getConsole().getWindow());
            view = d.getView();
            desc = d.getDescriptor();
        } else {
            view = MonitorPageDialog.getView(desc);
        }
        if (view != null) {
            JComponent panel = view.getPanel();
            if (panel != null) {
                panel.putClientProperty(MonitorPageDialog.Descriptor.class.getName(), desc);
                TreeMap<Object, Object> par = new TreeMap<>();
                par.put(Panel.TITLE, view.getName());
                Consumer<JComponent> onClose = c -> onClose(c);
                par.put(Panel.ON_CLOSE, onClose);
                Console.getConsole().getPanelManager().open(panel, par);
                view.install(getConsole().getStatusAggregator());
                currentViews.add(view);
            }
        }
    }
    
    private void onClose(Component component) {
        
        if (!(component instanceof JComponent)) return;
        JComponent panel = (JComponent) component;
        
        MonitorView view = null;
        Iterator<MonitorView> it = currentViews.iterator();
        while (it.hasNext()) {
            MonitorView v = it.next();
            if (panel == v.getPanel()) {
                view = v;
                it.remove();
                break;
            }
        }
        if (view == null) return;
        view.uninstall(getConsole().getStatusAggregator());
        
        Object o = panel.getClientProperty(MonitorPageDialog.Descriptor.class.getName());
        if (o instanceof MonitorPageDialog.Descriptor) {
            MonitorPageDialog.Descriptor desc = (MonitorPageDialog.Descriptor) o;
            Iterator<MonitorPageDialog.Descriptor> itt = recentViews.iterator();
            while (itt.hasNext()) {
                if (itt.next().getName().equals(desc.getName())) {
                    itt.remove();
                    break;
                }
            }
            if (recentViews.size() > 9) {
                recentViews.pollLast();
            }
            recentViews.addFirst(desc);
            resetMenu();
        }
    }
    
    private void resetMenu() {
        Console console = getConsole();
        console.removeMenu(" CCS Tools ", "Monitoring...");
        getServices().addMenu(newView, "400: CCS Tools :-1:15", "Monitoring...:1");
        int i = 0;
        for (MonitorPageDialog.Descriptor desc : recentViews) {
            Action act = new AbstractAction(desc.getName()) {
                @Override
                public void actionPerformed(ActionEvent e) {
                    openNewView(desc);
                }
            };
            getServices().addMenu(act, "400: CCS Tools :-1:15", "Monitoring...:1:" + i++);
        }
    }
    
    
// -- Saving and restoring sessions : ------------------------------------------

    @Override
    public ComponentDescriptor save() {
        Descriptor desc = new Descriptor(getServices().getDescriptor());
        
        ArrayList<MonitorPageDialog.Descriptor> current = new ArrayList<>(currentViews.size());
        for (MonitorView view : currentViews) {
            JComponent panel = view.getPanel();
            if (panel != null) {
                Object o = panel.getClientProperty(MonitorPageDialog.Descriptor.class.getName());
                if (o instanceof MonitorPageDialog.Descriptor) {
                    current.add((MonitorPageDialog.Descriptor)o);
                }
            }
        }
        if (!current.isEmpty()) {
            desc.setCurrentViews(current.toArray(new MonitorPageDialog.Descriptor[current.size()]));
        }
        
        if (!recentViews.isEmpty()) {
            desc.setRecentViews(recentViews.toArray(new MonitorPageDialog.Descriptor[recentViews.size()]));
        }
        
        return desc;
    }

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        for (MonitorView view : new ArrayList<>(currentViews)) {
            closePage(view);
        }
        currentViews.clear();
        recentViews.clear();
        if (storageBean instanceof Descriptor) {
            Descriptor desc = (Descriptor) storageBean;
            if (desc.getCurrentViews() != null) {
                for (MonitorPageDialog.Descriptor d : desc.getCurrentViews()) {
                    openNewView(d);
                }
            }
            if (desc.getRecentViews() != null) {
                recentViews.addAll(Arrays.asList(desc.getRecentViews()));
            }
        }
        resetMenu();
        return true;
    }

    static public class Descriptor extends ComponentDescriptor {

        private MonitorPageDialog.Descriptor[] currentViews;
        private MonitorPageDialog.Descriptor[] recentViews;
        
        public Descriptor() {
        }
        
        public Descriptor(ComponentDescriptor seed) {
            super(seed);
        }

        public MonitorPageDialog.Descriptor[] getCurrentViews() {
            return currentViews;
        }

        public void setCurrentViews(MonitorPageDialog.Descriptor[] currentViews) {
            this.currentViews = currentViews;
        }

        public MonitorPageDialog.Descriptor getCurrentViews(int index) {
            return this.currentViews[index];
        }

        public void setCurrentViews(int index, MonitorPageDialog.Descriptor currentViews) {
            this.currentViews[index] = currentViews;
        }

        public MonitorPageDialog.Descriptor[] getRecentViews() {
            return recentViews;
        }

        public void setRecentViews(MonitorPageDialog.Descriptor[] recentViews) {
            this.recentViews = recentViews;
        }

        public MonitorPageDialog.Descriptor getRecentViews(int index) {
            return this.recentViews[index];
        }

        public void setRecentViews(int index, MonitorPageDialog.Descriptor recentViews) {
            this.recentViews[index] = recentViews;
        }
        
    }
    
    
// -- Temporary code for saving monitoring pages : -----------------------------
    
    private class Saver extends CommandProcessor implements PageListener {
        
        private boolean on;

        @Override
        public void pageChanged(PageEvent pe) {
            PageContext context = pe.getPageContext();
            if (context == null) return;
            JComponent panel = (JComponent) context.getPage();
            if (panel == null) return;
            switch (pe.getID()) {
                case PageEvent.PAGESELECTED:
                    MonitorDisplay md = findMonitorDisplay(panel, false);
                    if (md == null) {
                        if (on) {
                            app.getCommandTargetManager().remove(this);
                            on = false;
                        }
                    } else {
                        if (!on) {
                            app.getCommandTargetManager().add(this);
                            on = true;
                        }
                    }
                    break;
                case PageEvent.PAGEDESELECTED:
//                    if (on) {
//                        app.getCommandTargetManager().remove(this);
//                        on = false;
//                    }
                    break;
            }
        }
        
        private MonitorDisplay findMonitorDisplay(JComponent component, boolean visibleOnly) {
            if (visibleOnly && !component.isShowing()) {
                return null;
            }
            if (component instanceof MonitorDisplay) {
                return (MonitorDisplay) component;
            }
            Component[] cc = component.getComponents();
            for (Component c : cc) {
                if (c instanceof JComponent) {
                    MonitorDisplay md = findMonitorDisplay((JComponent)c, visibleOnly);
                    if (md != null) return md;
                }
            }
            return null;
        } 
        
        public void onSaveAs() {
            try {
                PageManager pm = app.getPageManager();
                PageContext context = pm.getSelectedPage();
                JComponent component = (JComponent) context.getPage();
                MonitorDisplay md = findMonitorDisplay(component, true);

                Properties user = app.getUserProperties();
                String creator = "creator";
                ExportDialog dlg = new ExportDialog(creator, false);
                dlg.addExportFileType(new MonitorExport());
                dlg.addExportFileType(new MonitorExport()); // working around the bug in ExportDialog line 160
                dlg.setUserProperties(user);
                dlg.showExportDialog(app, "Save As...", (Component)md, "monitor");
            } catch (NullPointerException|ClassCastException x) {
                app.error("No selected monitoring table", x);
            }
        }
        
    }
    
    static public void saveData(OutputStream out, String mimeType, List<AgentChannel> channels, List<MonitorField> fields) {
        
        PrintStream ps = new PrintStream(out);
        int n = Field.values().length;
        
        StringBuilder sb = new StringBuilder();
        for (MonitorField field : fields) {
            sb.append(field.getTitle()).append(",");
        }      
        ps.println(sb.substring(0, sb.length()-1));
        
        for (AgentChannel channel : channels) {
            sb = new StringBuilder();
            for (MonitorField field : fields) {
                sb.append(channel.get(field.name())).append(",");
            }
            ps.println(sb.substring(0, sb.length()-1));
        }
        
    }

    static public void saveData(OutputStream out, String mimeType, List<Channel> channels) {
        
        PrintStream ps = new PrintStream(out);
        int n = Field.values().length;
        
        StringBuilder sb = new StringBuilder();
        for (Field field : Field.values()) {
            sb.append(field).append(",");
        }      
        ps.println(sb.substring(0, sb.length()-1));
        
        for (Channel channel : channels) {
            sb = new StringBuilder();
            for (Field field : Field.values()) {
                sb.append(channel.get(field)).append(",");
            }
            ps.println(sb.substring(0, sb.length()-1));
        }
        
    }

}
