package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;
import javax.swing.JComponent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.agent.AgentStatusAggregator;
import org.lsst.ccs.gconsole.agent.AgentStatusEvent;

/**
 * A monitoring data view that manages several dependent views.
 * 
 * Depending view modes:<ul>
 * <li>Listens to the status aggregator independently, with its own filter.
 * <li>Receives events filtered by the parent view filter.
 * <li>Has its AbstractMonitorView methods called by the parent view.
 * </ul>
 * Calls to setFilter(...) are forwarded to children as well. That means children added
 * before and after this call are treated differently.
 *
 * @author onoprien
 */
public class CompoundView extends AbstractMonitorView {
    
//    public enum Mode {
//        INDEPENDENT, FORWARD, CHAIN, IGNORE
//    }

// -- Fields : -----------------------------------------------------------------
    
    private final JComponent panel;
    private final ArrayList<MonitorView> children1 = new ArrayList<>(0); // Listen to the status aggregator independently, with its own filter.
    private final ArrayList<MonitorView> children2 = new ArrayList<>(0); // Receive events filtered by the parent view filter.
    private final ArrayList<AbstractMonitorView> children3 = new ArrayList<>(0); // Has its AbstractMonitorView methods called by the parent view.

// -- Life cycle : -------------------------------------------------------------
    
    public CompoundView(JComponent panel) {
        this.panel = panel;
    }
    

// -- Getters and setters : ----------------------------------------------------

    /**
     * Returns the graphical component maintained by this view.
     * @return Graphical component to be displayed by the GUI.
     */
    @Override
    public JComponent getPanel() {
        return panel;
    }

    @Override
    public void setFilter(AgentChannelsFilter filter) {
        super.setFilter(filter);
        children1.forEach(child -> child.setFilter(filter));
        children2.forEach(child -> child.setFilter(filter));
    }
    
    
// -- Adding views : -----------------------------------------------------------
    
    public void addView(MonitorView view, boolean chainFilters) {
        if (chainFilters) {
            children2.add(view);
        } else {
            children1.add(view);
        }
    }
    
    public void addView(AbstractMonitorView view) {
        children3.add(view);
    }
    
    
// -- Listening to the aggregator : --------------------------------------------

    @Override
    public void connect(AgentStatusEvent event) {
        super.connect(event);
        children2.forEach(view -> view.connect(event.filter(view.getFilter())));
    }

    @Override
    public void configure(AgentStatusEvent event) {
        super.configure(event);
        children2.forEach(view -> view.configure(event.filter(view.getFilter())));
    }

    @Override
    public void statusChanged(AgentStatusEvent event) {
        super.statusChanged(event);
        children2.forEach(view -> view.statusChanged(event.filter(view.getFilter())));
    }

    @Override
    public void disconnect(AgentStatusEvent event) {
        super.disconnect(event);
        children2.forEach(view -> view.disconnect(event.filter(view.getFilter())));
    }
   
// -- Calling AbstractMonitorView hooks : --------------------------------------    

    @Override
    protected void connect(AgentInfo agent) {
        children3.forEach(view -> view.connect(agent));
    }

    @Override
    protected void disconnect(AgentInfo agent) {
        children3.forEach(view -> view.disconnect(agent));
    }

    @Override
    protected void addChannels(AgentInfo agent, Map<String, AgentChannel> channels) {
        children3.forEach(view -> view.addChannels(agent, channels));
    }

    @Override
    protected void removeChannels(AgentInfo agent, List<String> paths) {
        children3.forEach(view -> view.removeChannels(agent, paths));
    }

    @Override
    protected void updateChannels(AgentInfo agent, Map<String, Map.Entry<AgentChannel, List<String>>> channels) {
        children3.forEach(view -> view.updateChannels(agent, channels));
    }
    
    
// -- Installation : -----------------------------------------------------------

    @Override
    public void install(AgentStatusAggregator aggregator) {
        super.install(aggregator);
        children1.forEach(view -> view.install(aggregator));
    }

    @Override
    public void uninstall(AgentStatusAggregator aggregator) {
        super.uninstall(aggregator);
        children1.forEach(view -> view.uninstall(aggregator));
    }    

}
