package org.lsst.ccs.gconsole.plugins.monitor;

import java.time.Duration;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.JComponent;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.agent.AgentStatusEvent;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;

/**
 * 
 *
 * @author onoprien
 */
abstract public class AbstractMonitorView implements MonitorView {

// -- Fields : -----------------------------------------------------------------
    
    protected AgentChannelsFilter filter;
    
    private boolean fixedDisplayList;
    private Set<String> displayChannels;

// -- Life cycle : -------------------------------------------------------------
    
// -- Getters and setters : ----------------------------------------------------

    @Override
    public JComponent getPanel() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public AgentChannelsFilter getFilter() {
        return filter;
    }

    @Override
    public void setFilter(AgentChannelsFilter filter) {
        this.filter = filter;
        fixedDisplayList = filter != null && filter.getDisplayChannels() != null;
    }
    
    protected boolean isChannelListFixed() {
        return fixedDisplayList;
    }
    
    
// -- Default agent status listener implementation : ---------------------------

    @Override
    public void connect(AgentStatusEvent event) {
        SwingUtilities.invokeLater(() -> {
            connect(event.getSource());
        });
    }

    @Override
    public void configure(AgentStatusEvent event) {
        if (displayChannels == null) {
            List<String> fixed = filter.getDisplayChannels();
            if (fixed == null) {
                fixedDisplayList = false;
                displayChannels = Collections.emptySet();
            } else {
                fixedDisplayList = true;
                displayChannels = new HashSet<>(fixed);
                SwingUtilities.invokeLater(() -> {
                    addChannels(event.getSource(), fixed.stream().collect(Collectors.toMap(dp -> dp, dp -> null)));
                });
            }
        }
        update(event);
    }

    @Override
    public void statusChanged(AgentStatusEvent event) {
        update(event);
    }

    @Override
    public void disconnect(AgentStatusEvent event) {
        update(event);
        SwingUtilities.invokeLater(() -> {
            disconnect(event.getSource());
        });
    }
    
    private void update(AgentStatusEvent event) {
        
        if (fixedDisplayList) {
            
            Map<String,Map.Entry<AgentChannel,List<String>>> changedChannels = new LinkedHashMap<>();
            
            for (AgentChannel channel : event.getAddedChannels()) {
                for (String displayPath : filter.getDisplayPath(channel)) {
                    if (displayChannels.contains(displayPath)) {
                        changedChannels.put(displayPath, new AbstractMap.SimpleEntry<>(channel, null));
                    }
                }
            }
            
            for (AgentChannel channel : event.getRemovedChannels()) {
                for (String displayPath : filter.getDisplayPath(channel)) {
                    if (displayChannels.contains(displayPath)) {
                        changedChannels.put(displayPath, null);
                    }
                }
            }
            
            for (Map.Entry<AgentChannel, List<String>> change : event.getStatusChanges().entrySet()) {
                for (String displayPath : filter.getDisplayPath(change.getKey())) {
                    if (displayChannels.contains(displayPath)) {
                        changedChannels.put(displayPath, change);
                    }
                }
            }
            
            SwingUtilities.invokeLater(() -> {
                updateChannels(event.getSource(), changedChannels);
            });
            
        } else {
            
            Map<String,AgentChannel> addedChannels = null;
            for (AgentChannel channel : event.getAddedChannels()) {
                for (String displayPath : filter.getDisplayPath(channel)) {
                    if (addedChannels == null) addedChannels = new LinkedHashMap<>();
                    addedChannels.put(displayPath, channel);
                }
            }
            
            List<String> removedChannels = null;
            for (AgentChannel channel : event.getRemovedChannels()) {
                for (String displayPath : filter.getDisplayPath(channel)) {
                    if (removedChannels == null) removedChannels = new ArrayList<>(event.getRemovedChannels().size());
                    removedChannels.add(displayPath);
                }
            }
            
            Map<String,Map.Entry<AgentChannel,List<String>>> changedChannels = null;
            for (Map.Entry<AgentChannel,List<String>> e : event.getStatusChanges().entrySet()) {
                for (String displayPath : filter.getDisplayPath(e.getKey())) {
                    if (changedChannels == null) changedChannels = new LinkedHashMap<>();
                    changedChannels.put(displayPath, e);
                }
            }
            
            Map<String, AgentChannel> added = addedChannels;
            List<String> removed = removedChannels;
            Map<String,Map.Entry<AgentChannel,List<String>>> changed = changedChannels;
            SwingUtilities.invokeLater(() -> {
                if (added != null) addChannels(event.getSource(), added);
                if (removed != null) removeChannels(event.getSource(), removed);
                if (changed != null) updateChannels(event.getSource(), changed);
            });
            
        }
        
    }
    
    
// -- EDT hooks for updating the graphical component : -------------------------
    
    protected void connect(AgentInfo agent) {}
    
    protected void disconnect(AgentInfo agent) {}

    /**
     * Adds channels to display.
     * @param agent Source agent.
     * @param channels Map of paths to channels.
     */
    protected void addChannels(AgentInfo agent, Map<String,AgentChannel> channels) {}

    /**
     * Removes a channels from this view.
     * @param agent Source agent.
     * @param paths Paths to remove.
     */
    protected void removeChannels(AgentInfo agent, List<String> paths) {}
    
    /**
     * Called to notify this view that the specified channels have been updated.
     * @param agent Source agent.
     * @param channels Map of paths to key-value pairs, where the key is as {@code AgentChannel} object
     *                 and the value is a list of changed attributes.
     */
    protected void updateChannels(AgentInfo agent, Map<String,Map.Entry<AgentChannel,List<String>>> channels) {}
    
    
// -- Sending commands : -------------------------------------------------------
    
    protected final Object sendCommand(Duration timeout, String dest, String target, String cmnd, Object... args) throws Exception {
        ConcurrentMessagingUtils cmu = new ConcurrentMessagingUtils(Console.getConsole().getMessagingAccess());
        String dst = dest + (target == null ? "" : "/" + target);
        CommandRequest cmd = new CommandRequest(dst, cmnd, args);
        return cmu.sendSynchronousCommand(cmd, timeout);
    }
    
    protected final void sendCommand(String dest, String target, String cmnd, Object... args) {
        Thread t = new Thread(() -> {
            try {
                sendCommand(Duration.ofSeconds(1L), dest, target, cmnd, args);
            } catch (Exception x) {}
        }, "Temporary thread for sending a command");
        t.start();
    }

}
