package org.lsst.ccs.gconsole.plugins.alert;

import java.awt.event.ActionEvent;
import java.lang.reflect.InvocationTargetException;
import java.util.Map;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.StatusClearedAlert;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.plugins.tracer.LsstTracerPlugin;
import org.lsst.ccs.gconsole.plugins.tracer.Tracer;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.CommandOriginator;

/**
 * Graphical console plugin that handles alerts received from CCS subsystems.
 *
 * @author onoprien
 */
@Plugin(name="LSST Alert Notification Plugin",
        id="alert",
        description="LSST CCS alert notification service.")
public class LsstAlertPlugin extends ConsolePlugin {
    
// -- Fields : -----------------------------------------------------------------
    
    static private final String OPT_AUTO_START_VIEWER = "autoStartViewer";
    static private final String OPT_AUTO_START_TRACER = "autoStartTracer";
    static private final String OPT_SHOW_TESTER = "showTester";
    static final String OPT_HISTORY = "history";
    static final String OPT_MUTE = "mute";
    static final String OPT_TOFRONT = "toFront";
    static final String OPT_SELECT = "selectLast";
    
    private AlertHandler alertHandler; // protected by this
    
    private Action viewerAction;
    private volatile AlertViewer viewer; // modified on EDT only
    
    private JFrame emitter;

    
// -- Lifecycle : --------------------------------------------------------------

    @Override
    synchronized public void initialize() {
        
        // Define plugin settings and preferences
        
        getServices().addProperty(OPT_AUTO_START_VIEWER, false);
        getServices().addPreference(new String[] {"LSST","Alerts"}, "Startup", "${"+ OPT_AUTO_START_VIEWER +"} Start viewer on console sturtup.");
        getServices().addProperty(OPT_AUTO_START_TRACER, false);
        getServices().addPreference(new String[] {"LSST","Alerts"}, "Startup", "${"+ OPT_AUTO_START_TRACER +"} Start tracer on console sturtup.");
        
        getServices().addProperty(OPT_SHOW_TESTER, false);
        getServices().addPreference(new String[] {"LSST","Alerts"}, null, "${"+ OPT_SHOW_TESTER +"} Show tester controls.");
        
        getServices().addProperty(OPT_HISTORY, false);
        getServices().addProperty(OPT_MUTE, false);
        getServices().addProperty(OPT_TOFRONT, true);
        getServices().addProperty(OPT_SELECT, true);
        
        // Create menus for launching alert viewer and tracer
        
        viewerAction = new AbstractAction("Viewer") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if ((Boolean) getValue(Action.SELECTED_KEY)) {
                    onNewAlertViewer(null);
                } else {
                    stopAlertViewer();
                }
                
            }
        };
        viewerAction.putValue(Action.SELECTED_KEY, false);
        getServices().addMenu(viewerAction, "400: CCS Tools :-1:4", "Alerts:1");
        
        Action act  = new AbstractAction("Tracer") {
            @Override
            public void actionPerformed(ActionEvent e) {
                onNewAlertTracer();
            }
        };
        getServices().addMenu(act, "400: CCS Tools :-1:4", "Alerts:2");
        
        // Creare alert service and register it with the lookup
        
        alertHandler = new AlertHandler(this);
        getConsole().getConsoleLookup().add(alertHandler);
    }

    @Override
    synchronized public void start() {
        if ((boolean) getServices().getProperty(OPT_AUTO_START_VIEWER)) {
            onNewAlertViewer(null);
        }
        if ((boolean) getServices().getProperty(OPT_AUTO_START_TRACER)) {
            onNewAlertTracer();
        }
        if ((boolean) getServices().getProperty(OPT_SHOW_TESTER)) {
            showTester();
        }
    }
  
    @Override
    synchronized public void shutdown() {
        if (alertHandler != null) {
            getConsole().getConsoleLookup().remove(alertHandler);
            alertHandler = null;
        }
    }
    
    
// -- Operations : -------------------------------------------------------------
    
    public void onNewAlertViewer(AlertViewer.Descriptor config) {
        if (SwingUtilities.isEventDispatchThread()) {
            if (viewer == null) {
                viewerAction.putValue(Action.SELECTED_KEY, true);
                viewer = new AlertViewer(this, config);
                alertHandler.addListener(viewer);
            }
        } else {
            SwingUtilities.invokeLater(() -> onNewAlertViewer(config));
        }
    }
    
    public void stopAlertViewer() {
        if (SwingUtilities.isEventDispatchThread()) {
            if (viewer != null) {
                viewerAction.putValue(Action.SELECTED_KEY, false);
                alertHandler.removeListener(viewer);
                viewer.stop();
                viewer = null;
            }
        } else {
            SwingUtilities.invokeLater(this::stopAlertViewer);
        }
    }
    
    public void onNewAlertTracer() {
        LsstTracerPlugin tracerPlugin = (LsstTracerPlugin) getConsole().getConsoleLookup().lookup(LsstTracerPlugin.class);
        Tracer tracer = tracerPlugin.createTracer("Alerts", "BuiltIn/Alerts");
        tracer.setFilter(filteredMessage -> {
            BusMessage bm = filteredMessage.getBusMessage();
            if (bm instanceof StatusRaisedAlert) {
                StatusRaisedAlert am = (StatusRaisedAlert) bm;
                Alert alert = am.getRaisedAlert();
                String id = alert.getAlertId();
                StringBuilder sb = new StringBuilder();
                sb.append(AlertViewer.formatTimeStamp(am.getTimeStamp())).append(" : ");
                sb.append(am.getRaisedAlertSummary().getRaisedAlert(id).getLatestAlertState());
                sb.append(" from ").append(am.getOriginAgentInfo().getName()).append(".\n");
                sb.append("ID: ").append(id).append(".  Description:\n");
                sb.append(alert.getDescription()).append("\n");
                filteredMessage.setMessage(sb.toString());
                return filteredMessage;
            } else if (bm instanceof StatusClearedAlert) {
                StatusClearedAlert am = (StatusClearedAlert) bm;
                StringBuilder sb = new StringBuilder();
                sb.append(AlertViewer.formatTimeStamp(am.getTimeStamp())).append(" : ");
                sb.append(am.getOriginAgentInfo().getName()).append(" cleared alerts ");
                for (String id : am.getClearAlertIds()) {
                    sb.append(id).append(", ");
                }
                sb.delete(sb.length()-2, sb.length()).append(".\n");
                filteredMessage.setMessage(sb.toString());
                return filteredMessage;
            } else {
                return null;
            }
        });
    }
    
    public void clearAlerts(Map<String,String[]> alarms) {
        AgentMessagingLayer messenger = getConsole().getMessagingAccess();
        CommandOriginator originator = new CommandOriginator() {
            public void processAck(CommandAck ack) {}
            public void processResult(CommandResult result) {}
            public void processNack(CommandNack nack) {}
        };
        alarms.forEach((source, ids) -> {
            if (source.startsWith("Local")) {
                for (String id : ids) {
                    alertHandler.submitAlert(source, new Alert(id, ""), AlertState.NOMINAL, null);
                }
            } else {
                CommandRequest request;
                if (ids == null) {
                    request = new CommandRequest(source, "clearAllAlerts");
                } else {
                    request = new CommandRequest(source, "clearAlerts", (Object) ids);
                }
                messenger.sendCommandRequest(request, originator);
            }
        });
    }

// -- Saving/restoring : -------------------------------------------------------
    
    @Override
    public ComponentDescriptor save() {
        Descriptor desc = new Descriptor(getServices().getDescriptor());
        desc.setHistory((boolean) getServices().getProperty(OPT_HISTORY));
        desc.setMute((boolean) getServices().getProperty(OPT_MUTE));
        desc.setToFront((boolean) getServices().getProperty(OPT_TOFRONT));
        desc.setSelectLast((boolean) getServices().getProperty(OPT_SELECT));
        if (viewer != null) {
            if (SwingUtilities.isEventDispatchThread()) {
                desc.setViewerConfig(viewer.save());
            } else {
                try {
                    SwingUtilities.invokeAndWait(() -> {
                        if (viewer != null) {
                            desc.setViewerConfig(viewer.save());
                        }
                    });
                } catch (InterruptedException|InvocationTargetException x) {
                }
            }
        }
        return desc;
    }

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        
        if (!(storageBean instanceof Descriptor)) {
            throw new IllegalArgumentException("Illegal descriptor type: "+ storageBean.getClassName());
        }
        Descriptor desc = (Descriptor) storageBean;
        
        getServices().setProperty(OPT_HISTORY, desc.isHistory());
        getServices().setProperty(OPT_MUTE, desc.isMute());
        getServices().setProperty(OPT_TOFRONT, desc.isToFront());
        getServices().setProperty(OPT_SELECT, desc.isSelectLast());
        
        AlertViewer.Descriptor viewerConfig = desc.getViewerConfig();
        if (viewerConfig == null) {
            if (viewer != null) {
                stopAlertViewer();
            }
        } else {
            stopAlertViewer();
            onNewAlertViewer(viewerConfig);
        }
        return true;
    }
    
    static public class Descriptor extends ComponentDescriptor {

        private boolean selectLast;
        private boolean toFront;
        private boolean mute;
        private boolean history;
        private AlertViewer.Descriptor viewerConfig;
        
        public Descriptor() {
        }
        
        public Descriptor(ComponentDescriptor seed) {
            super(seed);
        }

        public boolean isSelectLast() {
            return selectLast;
        }

        public void setSelectLast(boolean selectLast) {
            this.selectLast = selectLast;
        }

        public boolean isToFront() {
            return toFront;
        }

        public void setToFront(boolean toFront) {
            this.toFront = toFront;
        }

        public boolean isMute() {
            return mute;
        }

        public void setMute(boolean mute) {
            this.mute = mute;
        }

        public boolean isHistory() {
            return history;
        }

        public void setHistory(boolean history) {
            this.history = history;
        }

        public AlertViewer.Descriptor getViewerConfig() {
            return viewerConfig;
        }

        public void setViewerConfig(AlertViewer.Descriptor viewerConfig) {
            this.viewerConfig = viewerConfig;
        }
        
    }

// -- Testing : ----------------------------------------------------------------
    @Override
    public void propertiesChanged(Object source, Map<String, Object> changes) {
        Boolean tester = (Boolean) changes.get(OPT_SHOW_TESTER);
        if (tester != null && tester) {
            showTester();
        }
    }
    
    private void showTester() {
        Action act = new AbstractAction("Tester") {
            @Override
            public void actionPerformed(ActionEvent ee) {
                setEnabled(false);
                SwingUtilities.invokeLater(() -> {
                    Box treeButtonsPanel = Box.createHorizontalBox();

                    JButton emitAlert = new JButton("Bus Alarm");
                    emitAlert.addActionListener(e -> {
                        String command = JOptionPane.showInputDialog(getConsole().getWindow(), "Enter ID:A|W|N:cause", "ID:A:cause");
                        String[] tokens = command.split(":");
                        AlertState level = tokens[1].equalsIgnoreCase("A") ? AlertState.ALARM : (tokens[1].equalsIgnoreCase("W") ? AlertState.WARNING : (tokens[1].equalsIgnoreCase("N") ? AlertState.NOMINAL : null));
                        if (level == null) {
                            if (tokens[0].isEmpty()) {
                                getConsole().clearAllAlerts();
                            } else {
                                getConsole().clearAlerts(tokens[0]);
                            }
                        } else {
                            getConsole().raiseAlert(new Alert(tokens[0], "description"), level, tokens[2]);
                        }
                    });
                    treeButtonsPanel.add(emitAlert);

                    emitAlert = new JButton("Local Alarm");
                    emitAlert.addActionListener(e -> {
                        String command = JOptionPane.showInputDialog(emitter, "Enter Source:ID:A|W|N:cause", "Source:ID:W:cause");
                        String[] tokens = command.split(":");
                        AlertState level = tokens[2].equalsIgnoreCase("A") ? AlertState.ALARM : (tokens[2].equalsIgnoreCase("W") ? AlertState.WARNING : AlertState.NOMINAL);
                        alertHandler.submitAlert(tokens[0], new Alert(tokens[1], "description"), level, tokens[3]);
                    });
                    treeButtonsPanel.add(emitAlert);

                    emitter = new JFrame("Alert emitter: "+ LsstAlertPlugin.this.getConsole().getName());
                    emitter.add(treeButtonsPanel);
                    emitter.pack();
                    emitter.setVisible(true);
                });
            }
        };
        getServices().addMenu(act, "400: CCS Tools :-1:4", "Alerts:1:1");
    }
    
}
