package org.lsst.ccs.gconsole.plugins.alert;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;
import java.util.*;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JEditorPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.data.RaisedAlertInstance;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.panel.PanelEvent;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelListener;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.gconsole.base.panel.PanelType;

/**
 * Display of CCS alert summary.
 * All methods except {@link #onAlert} should be called on EDT.
 *
 * @author onoprien
 */
public class AlertViewer implements AlertListener, PanelListener {

// -- Fields : -----------------------------------------------------------------
    
    private final LsstAlertPlugin plugin;
    private final String PANEL_GROUP = "AlertsViewer";
    private Descriptor config;
    
    private AlertTree tree;
    private JEditorPane infoPanel;
    private JPanel rightPanel;
    private JScrollPane leftPanel;
    
    private Action clearAction;
    
    private JCheckBox freezeBox;
    private JCheckBox historyBox;
    private JCheckBox muteBox;
    private JCheckBox toFrontBox;
    private JCheckBox selectNewBox;
    
    static private final int HSPACE = 10;
    static private final int VSPACE = 5;
    static private final EnumMap<AlertState,String> ALCOLOR = new EnumMap<>(AlertState.class);
    static {
        ALCOLOR.put(AlertState.NOMINAL, "#00AA00");
        ALCOLOR.put(AlertState.WARNING, "#0000AA");
        ALCOLOR.put(AlertState.ALARM, "#AA0000");
    }

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Creates an instance of {@code AlertViewer}.
     * No GUI panels are constructed and no resources are allocated at this point.
     * @param plugin Alert plugin instance that created this viewer.
     * @param configuration
     */
    public AlertViewer(LsstAlertPlugin plugin, Descriptor configuration) {
        this.plugin = plugin;
        config = configuration;
    }
    
    /**
     * Wake up and get ready to display alerts.
     * This method is called if an alert event is received while this viewer is asleep,
     * before the event is handled.
     */
    private void start() {
                
        clearAction = new AbstractAction("Clear") {
            @Override public void actionPerformed(ActionEvent e) {
                AlertNode node = tree.getSelectedNode();
                if (node == null) return;
                HashMap<String, String[]> alerts = getAlertMap(node);
                new SwingWorker<Void, Void>() {
                    @Override
                    protected Void doInBackground() throws Exception {
                        plugin.clearAlerts(alerts);
                        return null;
                    }
                }.execute();
            }
        };
        clearAction.putValue(Action.SHORT_DESCRIPTION, "Clear selected alerts.");
        
        Box buttonPanel = Box.createHorizontalBox();
        buttonPanel.add(new JButton(clearAction));
        buttonPanel.add(Box.createRigidArea(new Dimension(2*HSPACE, VSPACE)));

        freezeBox = new JCheckBox("Freeze");
        freezeBox.setSelected(false);
        freezeBox.addActionListener(e -> {
            if (!freezeBox.isSelected()) {
                infoPanel.setText(tree.getSelectedNode().getInfo());
            }
        });
        freezeBox.setToolTipText("Freeze information panel content");
        buttonPanel.add(freezeBox);
        buttonPanel.add(Box.createRigidArea(new Dimension(2*HSPACE, VSPACE)));
        
        historyBox = new JCheckBox("History");
        historyBox.setSelected((boolean) plugin.getServices().getProperty(LsstAlertPlugin.OPT_HISTORY));
        historyBox.addActionListener(e -> {
            plugin.getServices().setProperty(LsstAlertPlugin.OPT_HISTORY, historyBox.isSelected());
            infoPanel.setText(tree.getSelectedNode().getInfo());
        });
        historyBox.setToolTipText("Show history of alerts");
        buttonPanel.add(historyBox);
        buttonPanel.add(Box.createRigidArea(new Dimension(2*HSPACE, VSPACE)));
        
        muteBox = new JCheckBox("Mute");
        muteBox.setSelected((boolean) plugin.getServices().getProperty(LsstAlertPlugin.OPT_MUTE));
        muteBox.addActionListener(e -> plugin.getServices().setProperty(LsstAlertPlugin.OPT_MUTE, muteBox.isSelected()));
        muteBox.setToolTipText("Disable audio alerts");
        buttonPanel.add(muteBox);
        buttonPanel.add(Box.createRigidArea(new Dimension(2*HSPACE, VSPACE)));
        
        selectNewBox = new JCheckBox("Select new alerts");
        selectNewBox.setSelected((boolean) plugin.getServices().getProperty(LsstAlertPlugin.OPT_SELECT));
        selectNewBox.addActionListener(e -> plugin.getServices().setProperty(LsstAlertPlugin.OPT_SELECT, selectNewBox.isSelected()));
        selectNewBox.setToolTipText("Automatically select new alerts");
        buttonPanel.add(selectNewBox);
        buttonPanel.add(Box.createRigidArea(new Dimension(2*HSPACE, VSPACE)));
        
        toFrontBox = new JCheckBox("Show viewer on alert");
        toFrontBox.setSelected((boolean) plugin.getServices().getProperty(LsstAlertPlugin.OPT_TOFRONT));
        toFrontBox.addActionListener(e -> plugin.getServices().setProperty(LsstAlertPlugin.OPT_TOFRONT, toFrontBox.isSelected()));
        toFrontBox.setToolTipText("Display alert viewer when new alerts are received");
        buttonPanel.add(toFrontBox);
        buttonPanel.add(Box.createHorizontalGlue());
        buttonPanel.setBorder(BorderFactory.createEmptyBorder(VSPACE, VSPACE, VSPACE, VSPACE));
    
        tree = new AlertTree();
        tree.addTreeSelectionListener(e -> {
            AlertNode node = tree.getSelectedNode();
            if (node == null) { // nothing selected
                clearAction.setEnabled(false);
                historyBox.setEnabled(false);
                if (!freezeBox.isSelected()) {
                    infoPanel.setText("");
                }
            } else {
                clearAction.setEnabled(true);
                historyBox.setEnabled(node.isLeaf());
                if (!freezeBox.isSelected()) {
                    infoPanel.setText(node.getInfo());
                }
            }
        });
        tree.setBorder(BorderFactory.createEmptyBorder(VSPACE, VSPACE, VSPACE, VSPACE));
        tree.addMouseListener(new MouseAdapter() {
            @Override public void mouseClicked(MouseEvent e) {
                if (freezeBox.isSelected()) freezeBox.doClick();
            }
        });
    
        infoPanel = new JEditorPane();
        infoPanel.setEditable(false);
        infoPanel.setContentType("text/html");
        infoPanel.setBorder(BorderFactory.createEmptyBorder(VSPACE, VSPACE, VSPACE, VSPACE));
        infoPanel.setText("");
        
        rightPanel = new JPanel(new BorderLayout());
        rightPanel.add(new JScrollPane(infoPanel), BorderLayout.CENTER);
        rightPanel.add(buttonPanel, BorderLayout.SOUTH);
        
        leftPanel = new JScrollPane(tree);
        PanelManager panMan = plugin.getConsole().getPanelManager();
        Map<Panel,Object> state = new EnumMap<>(Panel.class);
        state.put(Panel.TYPE, PanelType.CONTROL);
        state.put(Panel.TITLE, "Alerts");
        state.put(Panel.GROUP, PANEL_GROUP);
        panMan.open(leftPanel, state);
        state.put(Panel.TYPE, PanelType.DATA);
        if (config != null && config.isUndocked()) {
            state.put(Panel.DOCKED, false);
            state.put(Panel.ICONIZED, config.isIconized());
            state.put(Panel.SIZE, config.getSize());
            state.put(Panel.LOCATION, config.getLocation());
            state.put(Panel.DEVICE, config.getDevice());
            state.put(Panel.MAXIMIZED, config.isMaximized());
        }
        panMan.open(rightPanel, state);
        panMan.addListener(this, leftPanel);
        panMan.addListener(this, rightPanel);
    }

    /** Release resources and go to sleep until the next event is received. */
    void stop() {
        if (tree == null) return;
        config = save();
        PanelManager panMan = plugin.getConsole().getPanelManager();
        panMan.removeListener(this);
        panMan.close(Collections.singletonMap(Panel.GROUP, PANEL_GROUP));
        clearAction = null;
        historyBox = null;
        muteBox = null;
        toFrontBox = null;
        tree = null;
        infoPanel = null;
        rightPanel = null;
        leftPanel = null;
    }


// -- Processing alert events --------------------------------------------------
    
    @Override
    public void onAlert(AlertEvent event) {
        if (SwingUtilities.isEventDispatchThread()) {
            update(event);
        } else {
            SwingUtilities.invokeLater(() -> update(event));
        }
    }

    private void update(AlertEvent event) {
        
        String source = event.getSource();
        RaisedAlertSummary summary = event.getSummary();
        
        // Process agent disconnects
        
        if (summary == null) {
            if (tree != null) {
                 AlertNode sourceNode = getSourceNode(source);
                if (sourceNode != null) {
                    sourceNode.removeFromParent();
                    tree.getModel().reload();
                    tree.clearSelection();
                }
            }
            return;
        }
        
        // Start viewer if necessary
        
        Set<RaisedAlertHistory> currentAlerts = summary.getAllRaisedAlertHistories();
        if (tree == null) {
            if (currentAlerts.isEmpty()) {
                return;
            } else {
                start();
            }
        }
        
        // Update tree model

        TreePath selPath = tree.getSelectionPath();
        
        AlertNode sourceNode = getSourceNode(source);        
        if (sourceNode == null) { // tree has no alerts from this source
            if (currentAlerts.isEmpty()) {
                return; // and the event does not have any alerts either
            }
            sourceNode = new AlertNode(source);
            for (RaisedAlertHistory h : currentAlerts) {
                updateAlert(sourceNode, h);
            }
            tree.getRoot().add(sourceNode);
        } else {
            HashMap<String,RaisedAlertHistory> id2history = new HashMap<>(currentAlerts.size()*2);
            currentAlerts.forEach(h -> id2history.put(h.getLatestAlert().getAlertId(), h));
            for (IDNode idNode : sourceNode.getLeaves()) {
                    String id = idNode.getID();
                    RaisedAlertHistory history = id2history.remove(id);
                    if (history == null) {
                        removeAlert(idNode);
                    } else {
                        if (getLatestAlertTimestamp(history).compareTo(idNode.getTimeStamp()) > 0) {
                            updateAlert(sourceNode, history);
                        }
                    }
            }
            for (RaisedAlertHistory h : id2history.values()) {
                updateAlert(sourceNode, h);
            }
        }
        
        // Update tree
        
        tree.getModel().reload();
        TreePath path;
        Alert alert = event.getAlert();
        if (alert == null) {
            path = new TreePath(new Object[] {tree.getRoot(), sourceNode});
        } else {
            String id = alert.getAlertId();
            IDNode idNode = tree.findID(sourceNode, id);
            if (idNode == null) {
                path = new TreePath(new Object[] {tree.getRoot(), sourceNode});
            } else {
                path = new TreePath(idNode.getPath());
            }
        }
        if (!selectNewBox.isSelected()) {
            path = selPath;
        }
        if (isValidPath(path)) {
            tree.setSelectionPath(path);
            tree.expandPath(path);
        } else {
            tree.clearSelection();
        }        
        if (!muteBox.isSelected()) {
            Toolkit.getDefaultToolkit().beep();
        }
        if (toFrontBox.isSelected()) {
            PanelManager panMan = plugin.getConsole().getPanelManager();
            panMan.set(leftPanel, Panel.SELECTED, true);
            panMan.set(rightPanel, Panel.SELECTED, true);
        }
        
    }

    @Override
    public void process(PanelEvent event) { // panel closed
        if (event.hasKey(Panel.OPEN) && !((Boolean)event.getNewValue())) {
            stop();
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private AlertNode getSourceNode(String source) {
        AlertNode root = tree.getRoot();
        for (int i=0; i<root.getChildCount(); i++) {
            AlertNode node = root.getChildAt(i);
            if (node.getUserObject().equals(source)) return node;
        }
        return null;
    }
    
    private HashMap<String,String[]> getAlertMap(AlertNode node) {
        HashMap<String, String[]> alerts = new HashMap<>();
        if (node instanceof IDNode) {
            IDNode idNode = (IDNode) node;
            alerts.put(node.getSource(), new String[]{idNode.getID()});
        } else if (node.isRoot()) {
            Enumeration e = node.children();
            while (e.hasMoreElements()) {
                AlertNode sourceNode = (AlertNode) e.nextElement();
                alerts.put(sourceNode.getSource(), null);
            }
        } else if (node.getLevel() == 1) {
            alerts.put(node.getSource(), null);
        } else {
            String source = node.getSource();
            ArrayList<String> ids = new ArrayList<>();
            Enumeration e = node.depthFirstEnumeration();
            while (e.hasMoreElements()) {
                Object o = e.nextElement();
                if (o instanceof IDNode) {
                    IDNode idNode = (IDNode) o;
                    ids.add(idNode.getID());
                }
            }
            alerts.put(source, ids.toArray(new String[0]));
        }
        return alerts;
    }
    
    static public String formatTimeStamp(long millis) {
        Instant instant = Instant.ofEpochMilli(millis);
        return formatTimeStamp(instant);
    }
    
    static public String formatTimeStamp(Instant instant) {
        DateTimeFormatter formatter = DateTimeFormatter.ofLocalizedDateTime(FormatStyle.SHORT).withZone(ZoneId.systemDefault());
        return formatter.format(instant);
    }
        
    static private String toColoredString(String text, AlertState level) {
        if (text == null) text = level.toString();
        String color = ALCOLOR.get(level);
        return "<font color="+ color +">"+ text +"</font>";
    }
    
    private void updateAlert(AlertNode sourceNode, RaisedAlertHistory history) {
        
        String id = history.getLatestAlert().getAlertId();
        String[] ss = id.split("/");
        ArrayList<String> tokens = new ArrayList<>(ss.length);
        for (String s : ss) {
            s = s.trim();
            if (!s.isEmpty()) tokens.add(s);
        }
        if (tokens.isEmpty()) return;
        
        for (int i=0; i<tokens.size()-1; i++) {
            String s = tokens.get(i);
            AlertNode tokenNode = null;
            Enumeration e = sourceNode.children();
            while (e.hasMoreElements()) {
                AlertNode node = (AlertNode) e.nextElement();
                if (!node.isLeaf()) {
                    if (node.getUserObject().equals(s)) {
                        tokenNode = node;
                        break;
                    }
                }
            }
            if (tokenNode == null) {
                tokenNode = new AlertNode(s);
                sourceNode.add(tokenNode);
            }
            sourceNode = tokenNode;
        }
        
        IDNode idNode = null;
        Enumeration e = sourceNode.children();
        while (e.hasMoreElements()) {
            Object o = e.nextElement();
            if (o instanceof IDNode) {
                IDNode node = (IDNode) o;
                if (node.getID().equals(id)) {
                    idNode = node;
                    break;
                }
            }
        }
        
        if (idNode == null) {
            idNode = new IDNode(history);
            sourceNode.add(idNode);
        } else {
            idNode.setHistory(history);
        }
        
    }
    
    private void removeAlert(IDNode idNode) {
        AlertNode node = idNode;
        while (true) {
            AlertNode parent = node.getParent();
            if (parent == null) return;
            if (parent.isRoot() || parent.getChildCount() > 1) {
                node.removeFromParent();
                return;
            } else {
                node = parent;
            }
        }
    }
    
    private boolean isValidPath(TreePath path) {
        int n = path.getPathCount();
        if (n == 0) {
            return false;
        } else {
            AlertNode child = (AlertNode) path.getLastPathComponent();
            for (int i=n-2; i>=0; i--) {
                AlertNode parentFromPath = (AlertNode) path.getPathComponent(i);
                AlertNode parentFromChild = child.getParent();
                if (parentFromPath != null && parentFromChild != null && parentFromPath.equals(parentFromChild)) {
                    child = parentFromPath;
                } else {
                    return false;
                }
            }
            return child.equals(tree.getRoot());
        }
    }
    
    // FIXME: remove once history uses Instant time stamps.
    static private Instant getLatestAlertTimestamp(RaisedAlertHistory history) {
        return Instant.ofEpochMilli(history.getLatestAlertTimestamp());
    }


// -- Alert tree utilities : ---------------------------------------------------
    
    static private enum WatchState {ACTIVE, MUTE, IGNORE}
    
    private class AlertTree extends JTree {
        
        AlertTree() {
            super(new AlertTreeModel());
            getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
            setCellRenderer(new AlertTreeRenderer());
        }
        
        public AlertNode getSelectedNode() {
            return (AlertNode) getLastSelectedPathComponent();
        }
        
        @Override
        public AlertTreeModel getModel() {
            return (AlertTreeModel) super.getModel();
        }
        
        AlertNode getRoot() {
            return getModel().getRoot();
        }
                
        IDNode findID(AlertNode sourceNode, String id) {
            if (sourceNode == null) return null;
            Enumeration e = sourceNode.depthFirstEnumeration();
            while (e.hasMoreElements()) {
                Object node = e.nextElement();
                if (node instanceof IDNode && ((IDNode)node).getID().equals(id))  {
                    return (IDNode)node;
                }
            }
            return null;
        }

    }
    
    private class AlertTreeModel extends DefaultTreeModel {
        
        AlertTreeModel() {
            super(new AlertNode("CCS Alerts") {
                
                @Override
                WatchState getWatchState() {
                    return isIgnore() ? WatchState.IGNORE : (isMute() ? WatchState.MUTE : WatchState.ACTIVE);
                }
                
                @Override
                String getInfo() {
                    StringBuilder sb = new StringBuilder();
                    sb.append("<html>CCS Control System</b><p>");
                    int activeAlerts = 0;
                    if (children != null) {
                        for (Object child : children) {
                            if (((AlertNode)child).getAlertLevel() != AlertState.NOMINAL) {
                                activeAlerts++;
                            }
                        }
                    }
                    sb.append("Subsystems with active alerts: <b>").append(activeAlerts).append("</b><br>");
                    sb.append("Overall alert level: <b>").append(getAlertLevel()).append("</b><br>");
                    sb.append("Last changed at <b>").append(formatTimeStamp(getTimeStamp())).append("</b></html>");
                    return sb.toString();
                }

                @Override
                String getSource() {
                    throw new UnsupportedOperationException("Trying to look up source subsystem for the root node.");
                }
                
            });
        }
        
        @Override
        public AlertNode getRoot() {
            return (AlertNode) super.getRoot();
        }
        
    }
    
    /** Mutable tree node that contains an instance of AlertNodeContent as its user object. */
    static class AlertNode extends DefaultMutableTreeNode implements Comparable<AlertNode> {
        
        private boolean mute;
        private boolean ignore;
        
        AlertNode(String userObject) {
            super(userObject);
        }

        @Override
        public String getUserObject() {
            return (String) super.getUserObject();
        }

        @Override
        public int compareTo(AlertNode other) {
            return getUserObject().compareTo(other.getUserObject());
        }
        
        @Override
        public void add(MutableTreeNode newChildNode) {
            AlertNode newChild = (AlertNode) newChildNode;
            Enumeration<TreeNode> e = children();
            int i = 0;
            boolean replace = false;
            while (e.hasMoreElements()) {
                AlertNode child = (AlertNode) e.nextElement();
                int c = newChild.compareTo(child);
                if (c > 0) {
                    i++;
                } else {
                    if (c == 0) replace = true;
                    break;
                }
            }
            if (replace) remove(i);
            insert(newChild, i);
        }

        @Override
        public AlertNode getParent() {
            return (AlertNode) super.getParent();
        }

        @Override
        public AlertNode getChildAt(int index) {
            return (AlertNode) super.getChildAt(index);
        }
        
        public void forEachChild(Consumer<? super AlertNode> action) {
            if (children == null) return;
            children.forEach(action);
        }
        
        public ArrayList<IDNode> getLeaves() {
            ArrayList<IDNode> out = new ArrayList<>(this.getLeafCount());
            Enumeration e = this.depthFirstEnumeration();
            while (e.hasMoreElements()) {
                Object node = e.nextElement();
                if (node instanceof IDNode) {
                    out.add((IDNode)node);
                }
            }
            return out;
        }
        
        String getInfo() {
            StringBuilder sb = new StringBuilder();
            sb.append("<html>Source: <b>").append(getSource()).append("</b><br>");
            sb.append("Alerts selected: <b>").append(getLeafCount()).append("</b><br>");
            sb.append("Highest alert level: <b>").append(toColoredString(null, getAlertLevel())).append("</b><br>");
            sb.append("Last changed at <b>").append(formatTimeStamp(getTimeStamp())).append("</b></html>");
            return sb.toString();
        }

        String getSource() {
            int level = getLevel();
            if (level > 1) {
                return getParent().getSource();
            } else {
                return getUserObject();
            }
        }
        
        Instant getTimeStamp() {
            Instant out = Instant.EPOCH;
            if (children != null) {
                for (Object child : children) {
                    Instant time = ((AlertNode) child).getTimeStamp();
                    if (time.compareTo(out) > 0) out = time;
                }
            }
            return out;
        }
        
        AlertState getAlertLevel() {
            AlertState alertState = AlertState.NOMINAL;
            Enumeration e = this.children();
            while (e.hasMoreElements()) {
                AlertState childState = ((AlertNode) e.nextElement()).getAlertLevel();
                if (childState.compareTo(alertState) > 0) {
                    alertState = childState;
                    if (alertState.equals(AlertState.ALARM)) break;
                }
            }
            return alertState;
        }
        
        WatchState getWatchState() {
            WatchState thisState = isIgnore() ? WatchState.IGNORE : (isMute() ? WatchState.MUTE : WatchState.ACTIVE);
            if (thisState == WatchState.IGNORE) return WatchState.IGNORE;
            WatchState parentState = this.getParent().getWatchState();
            return thisState.compareTo(parentState) == -1 ? parentState : thisState;
        }
        
        boolean isMute() {
            return mute;
        }
        
        void setMute(boolean isMute) {
            mute = isMute;
        }
        
        boolean isIgnore() {
            return ignore;
        }
        
        void setIgnore(boolean isIgnored) {
            ignore = isIgnored;
        }

        @Override
        public String toString() {
            return "<html>"+ toColoredString(super.toString(), getAlertLevel()) +"</html>";
        }
    }
    
    final class IDNode extends AlertNode {
        
        private RaisedAlertHistory history;
        
        IDNode(RaisedAlertHistory history) {
            super(history.getLatestAlert().getAlertId());
            setHistory(history);
        }

        void setHistory(RaisedAlertHistory history) {
            this.history = history;
        }
        
        RaisedAlertHistory getHistory() {
            return history;
        }
        
        String getID() {
            return getUserObject();
        }

        @Override
        public String toString() {
            return "<html>"+ toColoredString(getLastIdComponent() +" ("+ history.getLatestAlert().getDescription(), getAlertLevel()) +")</html>";
        }
        
        String getLastIdComponent() {
            String[] ss = getID().split("/");
            int i = ss.length;
            while (i > 0) {
                String s = ss[--i].trim();
                if (!s.isEmpty()) return s;
            }
            return getID();
        }
        
        @Override
        String getInfo() {
            Alert alert = history.getLatestAlert();
            StringBuilder sb = new StringBuilder();
            sb.append("<html>Source: <b>").append(getSource()).append("</b><br>");
            sb.append("Alert ID: <b>").append(alert.getAlertId()).append("</b><br>");
            sb.append("Alert description: <b>").append(alert.getDescription()).append("</b><br>");
            sb.append("Highest level: ").append(toColoredString(null, history.getHighestAlertState()));
            if (historyBox.isSelected()) {
                sb.append("<p><b>History:</b><p>");
                List<RaisedAlertInstance> alerts = history.getRaisedAlertInstancesList();
                int n = alerts.size();
                int nShow = Math.min(n, 50);
                for (int i = 0; i < nShow; i++) {
                    RaisedAlertInstance ra = alerts.get(i);
                    sb.append(toColoredString(formatTimeStamp(ra.getTimestamp()) +" "+ ra.getAlertState(), ra.getAlertState())).append(":<br>");
                    sb.append(ra.getCause()).append("<p>");
                }
                if (nShow < n) {
                    if (nShow < n - 1) {
                        sb.append("<p>...<p>").append(n - nShow - 1).append(" more").append("<p>...<p>");
                    }
                    RaisedAlertInstance ra = alerts.get(n - 1);
                    sb.append(toColoredString(formatTimeStamp(ra.getTimestamp()) +" "+ ra.getAlertState(), ra.getAlertState())).append(":<br>");
                    sb.append(ra.getCause()).append("<p>");
                }

            } else {
                sb.append("<br>Level at <b>").append(formatTimeStamp(history.getLatestAlertTimestamp())).append("</b> : <b>").append(toColoredString(null, history.getLatestAlertState())).append("</b><br>");
                List<RaisedAlertInstance> alerts = history.getRaisedAlertInstancesList();
                int n = alerts.size();
                if (n>0) {
                    sb.append("<p><b>Last cause:</b><p>").append(alerts.get(n-1).getCause()).append("</html>");
                }
            }
            return sb.toString();
        }
        
        @Override
        Instant getTimeStamp() {
            return getLatestAlertTimestamp(history);
        }
        
        @Override
        AlertState getAlertLevel() {
            return history.getHighestAlertState();
        }
        
    }
    
    private class AlertTreeRenderer extends DefaultTreeCellRenderer {

//        @Override
//        public Component getTreeCellRendererComponent(JTree tree, Object value, boolean sel, boolean expanded, boolean leaf, int row, boolean hasFocus) {
//            return super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);
//        }
        
    }
    
    
// -- Saving session : ---------------------------------------------------------
    
    static public class Descriptor extends ComponentDescriptor {
        
        private boolean iconized;
        private boolean undocked;
        private Dimension size;
        private Point location;
        private String device;
        private boolean maximized;

        public Descriptor() {
        }

        public boolean isIconized() {
            return iconized;
        }

        public void setIconized(boolean iconized) {
            this.iconized = iconized;
        }

        public boolean isUndocked() {
            return undocked;
        }

        public void setUndocked(boolean undocked) {
            this.undocked = undocked;
        }
 
        public Dimension getSize() {
            return size;
        }

        public void setSize(Dimension size) {
            this.size = size;
        }

        public Point getLocation() {
            return location;
        }

        public void setLocation(Point location) {
            this.location = location;
        }
 
        public String getDevice() {
            return device;
        }

        public void setDevice(String device) {
            this.device = device;
        }

        public boolean isMaximized() {
            return maximized;
        }

        public void setMaximized(boolean maximized) {
            this.maximized = maximized;
        }
      
    }
    
    Descriptor save() {
        if (rightPanel == null) {
            return config;
        } else {
            PanelManager panMan = plugin.getConsole().getPanelManager();
            Descriptor config = new Descriptor();
            config.setUndocked(! (Boolean) panMan.get(rightPanel, Panel.DOCKED));
            if (config.isUndocked()) {
                config.setIconized((Boolean) panMan.get(rightPanel, Panel.ICONIZED));
                config.setSize((Dimension) panMan.get(rightPanel, Panel.SIZE));
                config.setLocation((Point) panMan.get(rightPanel, Panel.LOCATION));
                Object device = panMan.get(rightPanel, Panel.DEVICE);
                if (device != null) {
                    config.setDevice(device.toString());
                }
                config.setMaximized((Boolean) panMan.get(rightPanel, Panel.MAXIMIZED));
            }
            return config;
        }
    }
    
}
