package org.lsst.ccs.gconsole.agent;

import java.util.*;

/**
 * A filter that selects a set of data channels and defines their presentation.
 * Original channels are identified by their unique string trending paths (slash-separated,
 * the first segment is the subsystem name).  This filter maps some of the original
 * channels to one or more display paths in "[page//]path1/.../pathN/name" format.
 * <p>
 * The exact interpretation of filter output depends on the tool using it. 
 * See {@link AgentStatusAggregator} for more details.
 *
 * @author onoprien
 */
public interface AgentChannelsFilter {
    
    /**
     * Trivial filter that accepts all channels and does not modify paths.
     * This filter uses default implementations for all methods.
     */
    static AgentChannelsFilter ALL = new AgentChannelsFilter() {
        @Override
        public String getName() {return "All channels";}
    };
    
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns the name of this filter.
     * 
     * @return Name of this filter.
     */
    default String getName() {
        return null;
    }
    
    /**
     * Returns a list of agent names accepted by this filter.
     * 
     * @return Agent names accepted by this filter, or {@code null} if any agent might be accepted.
     */
    default List<String> getAgents() {
        return null;
    }
    
    /**
     * Returns a list of original channel paths accepted by this filter.
     * 
     * @return Original channel paths accepted by this filter, or {@code null} if any
     * original path for which {@link #getDisplayPath} returns a non-empty list is accepted.
     */
    default List<String> getOriginChannels() {
        return null;
    }
    
    /**
     * Returns a list of display paths produced by this filter.
     * Typically, if this method returns a non-null set, components downstream from
     * this filter will display this fixed set of channels, whether or not the data
     * is available, regardless of what paths are returned by {@link #getDisplayPath}
     * applied to accepted original channels. Otherwise, any display channels for
     * which the data is available will be displayed.
     * 
     * @return Display paths produced by this filter, or {@code null} if the set is not fixed.
     */
    default List<String> getDisplayChannels() {
        return null;
    }
    
    /**
     * Returns the original path corresponding to the specified display channel.
     * 
     * @param displayPath Display channel path.
     * @return Original path, or {@code null} if the specified path does not correspond to any original channel.
     */
    default String getOriginPath(String displayPath) {
        return displayPath;
    }
    
    /**
     * Returns a list of display channels for the specified original channel.
     * The default implementation forwards the call to {@link #getDisplayPath(String)}
     * method. Subclasses may override it to take into account channel attributes and metadata.
     * <p>
     * Not that if the specified original channel is not accepted by this filter,
     * this method should return an empty list. {@code null} should never be returned.
     * 
     * @param channel Original data channel.
     * @return List of display channels.
     */
    default List<String> getDisplayPath(AgentChannel channel) {
        return getDisplayPath(channel.getPath());
    }
    
    /**
     * Returns a list of display channels for the specified original path.
     * <p>
     * Not that if the specified original channel is not accepted by this filter,
     * this method should return an empty list. {@code null} should never be returned.
     * 
     * @param originPath Original path.
     * @return List of display channels.
     */
    default List<String> getDisplayPath(String originPath) {
        return Collections.singletonList(originPath);
    }
    
    /**
     * Returns additional attributes associated with the specified path.
     *
     * @param path Full or partial display path.
     * @return Map of attribute names to their values.
     */
    default Map<String,Object> getAttributes(String path) {
        return Collections.emptyMap();
    }
    
    /**
     * Returns a list of attributes to display for each channel.
     * 
     * @param compact If {@code true}, returns a list of columns in a compact view.
     * @return Display fields, or {@code null} if this filter does not provides any hints on what attributes should be displayed.
     */
    default List<String> getFields(boolean compact) {
        return null;
    }

    
// -- Handling listeners : -----------------------------------------------------
    
    /**
     * Registers a listener that will be notified of any changes in this filter.
     * 
     * @param listener Listener to be notified.
     */
    default void addListener(Listener listener) {}
    
    /**
     * Removes a listener.
     * 
     * @param listener Listener to be removed.
     */
    default void removeListener(Listener listener) {}
    
    /**
     * Interface to be implemented by classes that should be notified of changes in the {@link AgentChannelsFilter}.
     */
    interface Listener {
        void filterChanged(Event event);
    }
    
    public static class Event extends EventObject {
        public Event(AgentChannelsFilter filter) {
            super(filter);
        }
        @Override
        public AgentChannelsFilter getSource() {
            return (AgentChannelsFilter) super.getSource();
        }
    }
    
}
