package org.lsst.ccs.subsystem.monitor.ui;

import java.io.OutputStream;
import java.util.*;
import javax.swing.BoxLayout;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.SwingUtilities;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.plugins.monitor.Channel;
import org.lsst.ccs.gconsole.plugins.monitor.DataTree;
import org.lsst.ccs.gconsole.plugins.monitor.Field;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorDisplay;
import org.lsst.ccs.subsystem.monitor.data.MonitorChan;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.monitor.data.MonitorState;

/*
 * Drop-in replacement for MonitorTrendingTable.
 * This class exist to support legacy code.
 *
 * @author onoprien
 */
public class MonitorTrendingTree extends JPanel implements MonitorTrendingView {
    
// -- Fields : -----------------------------------------------------------------
    
    private final CommandSender sender;
    private final AgentChannelsFilter filter;
    
    private final Map<String, SubsysDesc> subsysMap = new HashMap<>();
    private final LinkedHashMap<String,Channel> data = new LinkedHashMap<>();  // maps origin path to current data
    private final LinkedHashMap<String,DataTree> displayPath2view = new LinkedHashMap<>();  // maps display path to view


// -- Life cycle : -------------------------------------------------------------

    public MonitorTrendingTree(CommandSender sender, AgentChannelsFilter filter) {
        this.sender = sender;
        this.filter = filter == null ? AgentChannelsFilter.PASS : filter;
        Set<String> subsystems = this.filter.getAgents();
        if (subsystems != null) {
            for (String sName : subsystems) {
                SubsysDesc sDesc = new SubsysDesc(sName);
                subsysMap.put(sName, sDesc);
            }
        }
        add(new EmptyPanel());
    }
    
    
// -- Updating : ---------------------------------------------------------------

    public void setSubsystems(String... sNames) {
        if (subsysMap.isEmpty()) {
            for (String sName : sNames) {
                SubsysDesc sDesc = new SubsysDesc(sName);
                subsysMap.put(sName, sDesc);
            }
        } else {
            Set<String> subs = new HashSet<>(Arrays.asList(sNames));
            Iterator<Map.Entry<String, SubsysDesc>> it = subsysMap.entrySet().iterator();
            while (it.hasNext()) {
                String sName = it.next().getKey();
                if (!subs.contains(sName)) {
                    it.remove();
                }
            }
        }
    }

    public void updateTableModel(String sName, MonitorFullState status) {
        SubsysDesc sDesc = subsysMap.get(sName);
        if (sDesc == null) return;
        SwingUtilities.invokeLater(new UpdateTableModel(sDesc, status));
    }

    class UpdateTableModel implements Runnable {

        final SubsysDesc sDesc;
        final MonitorFullState s;

        UpdateTableModel(SubsysDesc sDesc, MonitorFullState s) {
            this.sDesc = sDesc;
            this.s = s;
        }

        @Override
        public void run() {
            
            // update subsystem description
            
            sDesc.enabled = true;
            sDesc.channels.clear();
            sDesc.goodChans.clear();
            sDesc.onlineChans.clear();
            sDesc.lowLimitChange.clear();
            sDesc.highLimitChange.clear();
            sDesc.name2channel.clear();
            MonitorState state = s.getMonitorState();
            sDesc.goodChans.or(state.getGoodChans());
            sDesc.onlineChans.or(state.getOnlineChans());
            sDesc.lowLimitChange.or(state.getLowLimitChange());
            sDesc.highLimitChange.or(state.getHighLimitChange());
            
            Set<String> originChannels = filter.getOriginChannels();
            Set<String> displayChannels = filter.getDisplayChannels();
            int i=0;
            for (MonitorChan mc : s.getChannels()) {
                Channel chan = new Channel(sDesc.name, mc);
                chan.setOnline(sDesc.onlineChans.get(i));
                chan.setGood(sDesc.goodChans.get(i));
                chan.setLowLimitChanged(sDesc.lowLimitChange.get(i));
                chan.setHighLimitChanged(sDesc.highLimitChange.get(i++));
                String originPath = chan.getPath();
                if (originChannels == null || originChannels.contains(originPath)) {
                    List<String> dpList = filter.getDisplayPath(originPath);
                    boolean needed = false;
                    for (String dp : dpList) {
                        if (displayChannels == null || displayChannels.contains(dp)) {
                            needed = true;
                        }
                    }
                    sDesc.channels.add(needed ? chan : null);
                } else {
                    sDesc.channels.add(null);
                }
            }
            
            // update data
            
            Iterator<Map.Entry<String,Channel>> it = data.entrySet().iterator();
            while (it.hasNext()) {
                if (it.next().getValue().getSubsystemName().equals(sDesc.name)) {
                    it.remove();
                }
            }
            for (Channel c : sDesc.channels) {
                if (c != null) {
                    sDesc.name2channel.put(c.getName(), c);
                    data.put(c.getPath(), c);
                }
            }
            
            // build MonitorView panels
            
            Map<String,List<String>> page2channels = new HashMap<>();
            for (SubsysDesc sub : subsysMap.values()) {
                for (Channel c : sub.channels) {
                    if (c != null) {
                        for (String displayPath : filter.getDisplayPath(c.getPath())) {
                            if (displayChannels == null || displayChannels.contains(displayPath)) {
                                String page = s.getPages().get(c.getPage());
                                DisplayPath dp = new DisplayPath(displayPath, page);
                                List<String> channels = page2channels.get(dp.page);
                                if (channels == null) {
                                    channels = new ArrayList<>();
                                    page2channels.put(dp.page, channels);
                                }
                                channels.add(displayPath);
                            }
                        }
                    }
                }
            }
            displayPath2view.clear();
            List<DataTree> views = new ArrayList<>();
            for (Map.Entry<String,List<String>> e : page2channels.entrySet()) {
                DataTree view = new DataTree(sender);
                views.add(view);
                view.setName(e.getKey());
                LinkedHashMap<String,Channel> cc = new LinkedHashMap<>();
                for (String dp : e.getValue()) {
                    displayPath2view.put(dp, view);
                    String originChannel = filter.getOriginPath(dp);
                    cc.put(dp, data.get(originChannel));
                }
                view.setChannels(cc);
            }
            
            // update GUI
            
            removeAll();
            setLayout(new BoxLayout(MonitorTrendingTree.this, BoxLayout.LINE_AXIS));
            if (views.size() == 1) {
                add(new JScrollPane(views.get(0).getView()));
            } else if (views.size() > 1) {
                JTabbedPane tabs = new JTabbedPane();
                for (DataTree view : views) {
                    tabs.add(view.getName(), new JScrollPane(view.getView()));
                }
                add(tabs);
            }
            MonitorTrendingTree.this.validate();
        }

    }

    public void updateTableValue(String sName, String cName, double value) {
        Target target = makeTarget(sName, cName);
        if (target == null) return;
        SwingUtilities.invokeLater(new UpdateTableValue(target, value));
    }

    class UpdateTableValue implements Runnable {

        Target target;
        double value;

        UpdateTableValue(Target target, double value) {
            this.target = target;
            this.value = value;
        }

        @Override
        public void run() {
            target.channel.setValue(value);
            for (String dp : target.displayPaths) {
                DataTree view = displayPath2view.get(dp);
                if (view != null) {
                    view.update(dp);
                }
            }
        }
    }

    public void updateTableLimit(String sName, String cName, String limName, String value) {
        Target target = makeTarget(sName, cName);
        if (target == null) return;
        try {
            double limitValue = Double.parseDouble(value);
            Field field = limName.equals("alarmHigh") ? Field.HIGH : Field.LOW;
            SwingUtilities.invokeLater(new UpdateLimitValue(target, field, limitValue));
        } catch (NumberFormatException|NullPointerException x) {
        }
    }

    class UpdateLimitValue implements Runnable {

        final Target target;
        final Field field;
        final double value;

        UpdateLimitValue(Target target, Field field, double value) {
            this.target = target;
            this.field = field;
            this.value = value;
        }

        @Override
        public void run() {
            target.channel.set(field, value);
            for (String dp : target.displayPaths) {
                DataTree view = displayPath2view.get(dp);
                if (view != null) {
                    view.update(dp, field);
                }
            }
        }

    }

    public void updateTableState(String sName, MonitorState status) {
        SubsysDesc sDesc = subsysMap.get(sName);
        if (sDesc == null) return;
        SwingUtilities.invokeLater(new UpdateState(sDesc, status));
    }

    class UpdateState implements Runnable {

        SubsysDesc sDesc;
        MonitorState s;

        UpdateState(SubsysDesc sDesc, MonitorState s) {
            this.sDesc = sDesc;
            this.s = s;
        }

        @Override
        public void run() {
            BitSet updateValue = (BitSet) s.getGoodChans().clone();
            updateValue.xor(sDesc.goodChans);
            sDesc.goodChans.xor(updateValue);
            BitSet updateOnline = (BitSet) s.getOnlineChans().clone();
            updateOnline.xor(sDesc.onlineChans);
            sDesc.onlineChans.xor(updateOnline);
            if (!sDesc.channels.isEmpty()) {
                updateValue.or(updateOnline);
                for (int i = updateValue.nextSetBit(0); i >= 0; i = updateValue.nextSetBit(i+1)) {
                    if (i < sDesc.channels.size()) {
                        Channel channel = sDesc.channels.get(i);
                        if (channel != null) {
                            channel.setGood(sDesc.goodChans.get(i));
                            channel.setOnline(sDesc.onlineChans.get(i));
                            for (String dp : filter.getDisplayPath(channel.getPath())) {
                                DataTree view = displayPath2view.get(dp);
                                if (view != null) {
                                    view.update(dp, Field.VALUE);
                                }
                            }
                        }
                    }
                }
            }
            BitSet updateHigh = (BitSet) s.getHighLimitChange().clone();
            updateHigh.xor(sDesc.highLimitChange);
            sDesc.highLimitChange.xor(updateHigh);
            for (int i = updateHigh.nextSetBit(0); i >= 0; i = updateHigh.nextSetBit(i + 1)) {
                if (i < sDesc.channels.size()) {
                    Channel channel = sDesc.channels.get(i);
                    if (channel != null) {
                        channel.setHighLimitChanged(sDesc.highLimitChange.get(i));
                        for (String dp : filter.getDisplayPath(channel.getPath())) {
                            DataTree view = displayPath2view.get(dp);
                            if (view != null) {
                                view.update(dp, Field.HIGH);
                            }
                        }
                    }
                }
            }
            BitSet updateLow = (BitSet) s.getLowLimitChange().clone();
            updateLow.xor(sDesc.lowLimitChange);
            sDesc.lowLimitChange.xor(updateLow);
            for (int i = updateLow.nextSetBit(0); i >= 0; i = updateLow.nextSetBit(i + 1)) {
                if (i < sDesc.channels.size()) {
                    Channel channel = sDesc.channels.get(i);
                    if (channel != null) {
                    channel.setLowLimitChanged(sDesc.lowLimitChange.get(i));
                        for (String dp : filter.getDisplayPath(channel.getPath())) {
                            DataTree view = displayPath2view.get(dp);
                            if (view != null) {
                                view.update(dp, Field.LOW);
                            }
                        }
                    }
                }
            }
        }

    }

    public void disableSystem(String sName) {
        SubsysDesc sDesc = subsysMap.get(sName);
        if (sDesc == null) return;
        SwingUtilities.invokeLater(new DisableSystem(sDesc));
    }

    class DisableSystem implements Runnable {
        
        final SubsysDesc sDesc;

        DisableSystem(SubsysDesc sDesc) {
            this.sDesc = sDesc;
        }

        @Override
        public void run() {
            sDesc.enabled = false;
            for (Channel chan : sDesc.channels) {
                for (String dp : filter.getDisplayPath(chan.getPath())) {
                    DataTree view = displayPath2view.get(dp);
                    if (view != null) {
                        view.update(dp, Field.VALUE);
                    }
                }
            }
        }
    }
    
    
// -- Local classes : ----------------------------------------------------------

    static class SubsysDesc {
        final String name;
        boolean enabled;
        final List<Channel> channels = new ArrayList<>();        // Ordered list of channels
        final BitSet goodChans = new BitSet();                        // Ordered set of good channels
        final BitSet onlineChans = new BitSet();                      // Ordered set of online channels
        final BitSet lowLimitChange = new BitSet();                   // Ordered set of channels with changed low limit
        final BitSet highLimitChange = new BitSet();                  // Ordered set of channels with changed high limit
        final HashMap<String,Channel> name2channel = new HashMap<>();   // Map of channel names to channels
        SubsysDesc(String name) {
            this.name = name;
        }
    }
    
    static class Target {
        Target(SubsysDesc sDesc, Channel channel, List<String> displayPaths) {
            this.sDesc = sDesc;
            this.channel = channel;
            this.displayPaths = displayPaths;
        }
        final SubsysDesc sDesc;
        final Channel channel;
        final List<String> displayPaths;        
    }
    
    static class DisplayPath {
        final String page;
        final String node;
        DisplayPath(String displayPath, String page) {
            String[] ss = displayPath.split("//");
            if (ss.length == 1) {
                this.page = page == null ? "Monitor" : page;
                node = displayPath;
            } else {
                this.page = ss[0];
                node = ss[1];
            }
        }
    }

    /** Empty panel added before the list of channels is set to enable "Save As..." */
    static class EmptyPanel extends JPanel implements MonitorDisplay {
        @Override
        public void saveData(OutputStream out, String mimeType) {}        
    }
    
    
// -- Local methods : ----------------------------------------------------------

    private Target makeTarget(String sName, String cName) {
        SubsysDesc sDesc = subsysMap.get(sName);
        if (sDesc == null) return null;
        Channel channel = sDesc.name2channel.get(cName);
        if (channel == null) return null;
//        if (originChannels != null && !originChannels.contains(originChannel)) return null;
        List<String> displayPaths = filter.getDisplayPath(channel.getPath());
        if (displayPaths.isEmpty()) return null;
        return new Target(sDesc, channel, displayPaths);
    }
}
