package org.lsst.ccs.gconsole.plugins.states;

import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.List;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JScrollPane;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.Plugin;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;

/**
 * Graphical console plugin that supports display of subsystem states.
 *
 * @author The LSST CCS Team
 */
@Plugin(name="LSST States Plugin",
        id="state",
        description="LSST CCS subsystem states service.")
public class LsstStatesPlugin extends ConsolePlugin implements AgentPresenceListener {

    
// -- Life cycle : -------------------------------------------------------------

    private final SubsystemStatesTableModel model = new SubsystemStatesTableModel();
    private final List<SubsystemStatesChangeListener> listeners = new ArrayList<>();
    private final JComponent statesPage = new JScrollPane(new SubsystemStatesTable(model));

    private final Action statesAction;

    public LsstStatesPlugin() {
        statesAction = new AbstractAction("States") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if ((Boolean) getValue(Action.SELECTED_KEY)) {
                    openStatesPage();
                } else {
                    closeStatesPage();
                }
            }
        };
        statesAction.putValue(Action.SELECTED_KEY, false);
    }
    
    @Override
    synchronized public void initialize() {
        getServices().addMenu(statesAction, "400: CCS Tools :-1:15");
        
    }

    synchronized private void openStatesPage() {
        statesAction.putValue(Action.SELECTED_KEY, true);
        getConsole().getPanelManager().open(statesPage, "Subsystem States");        
        getConsole().getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);
        
    }

    synchronized private void closeStatesPage() {
        statesAction.putValue(Action.SELECTED_KEY, false);
        getConsole().getPanelManager().close(statesPage);        
        getConsole().getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(this);
        for (SubsystemStatesChangeListener listener : listeners) {
            getConsole().getMessagingAccess().removeStatusMessageListener(listener);
        }
        listeners.clear();
        model.clear();
    }
    
    
    
    @Override
    public void connecting(AgentInfo agent) {
        SubsystemStatesChangeListener listener = new SubsystemStatesChangeListener(agent.getName(), model);
        listeners.add(listener);
        getConsole().getMessagingAccess().addStatusMessageListener(listener, BusMessageFilterFactory.messageOrigin(agent.getName()));
    }

    @Override
    public void disconnecting(AgentInfo agent) {
        for ( SubsystemStatesChangeListener listener : listeners ) {
            if ( listener.getSubsystem().equals(agent.getName()) ) {
                getConsole().getMessagingAccess().removeStatusMessageListener(listener);
                listener.disable();
            }
        }
    }

    
}
