package org.lsst.ccs.gconsole.base.panel;

import java.awt.Component;
import java.io.Serializable;
import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import javax.swing.JComponent;

/**
 * Management of panels in the graphical console.
 * All access to {@code PanelManager} except for listener registration and removal should happen on the AWT Event Dispatching Thread.
 *
 * @author onoprien
 */
public interface PanelManager {

// -- Opening and closing pages : ----------------------------------------------
    
   void open(Component panel, Map<?,?> properties);
   
   default void open(Component panel, String title) {
        open(panel, Collections.singletonMap(Panel.TITLE, title));
    }
   
   void close(Component panel);
   
   default void close(Predicate<Component> selector) {
        getPanels(selector).forEach(panel -> close(panel));
    }
   
   default void close(Map<?,?> properties) {
        getPanels(properties).forEach(panel -> close(panel));
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns the list of panels managed by this {@code PanelManager}.
     * @return The list of panels managed by this {@code PanelManager}.
     */
    List<? extends Component> getPanels();
    
    default List<? extends Component> getPanels(Predicate<Component> selector) {
        return getPanels().stream().filter(selector).collect(Collectors.toList());
    }
    
    default List<? extends Component> getPanels(Map<?,?> properties) {
        return getPanels(panel -> isInState(panel, properties));
    }
    
    default List<? extends Component> getPanels(Object key, Object value) {
        return getPanels(Collections.singletonMap(key, value));
    }
    
    default Component getSelectedPanel(PanelType type) {
        for (Component panel : getPanels()) {
            Boolean selected = (Boolean) get(panel, Panel.SELECTED);
            if (type == null) type = PanelType.DATA;
            if (selected != null && selected && type.equals(get(panel, Panel.TYPE))) {
                return panel;
            }
        }
        return null;
    }
    
    /**
     * Returns the value of the panel property identified by the specified key.
     * 
     * @param panel
     * @param key The key whose associated value is to be returned.
     * @return The value to which the specified key is mapped, or {@code null} if the state contains no mapping for the key.
     */
    Object get(Component panel, Object key);
    
    /**
     * Returns the value of the panel state attribute identified by the specified key, or {@code defaultValue} 
     * if either the panel state contains no mapping for the key, or the key is mapped to {@code null} .
     * 
     * @param panel
     * @param key The key whose associated value is to be returned.
     * @param defaultValue The default value.
     * @return The value to which the specified key is mapped.
     */
    default Object getOrDefault(Component panel, Object key, Object defaultValue) {
        Object value = get(panel, key);
        return value == null ? defaultValue : value;
    }
    
    default HashMap<String,Serializable> get(Component panel, Collection<?> keys) {
        HashMap<String, Serializable> out = new HashMap<>(keys.size()*2);
        keys.forEach(key -> {
            Object value = get(panel,key);
            if (value instanceof Serializable) {
                out.put(key.toString(), (Serializable)value);
            }
        });
        return out;
    }
    
    boolean isInState(Component panel, Map<?,?> properties);
    
    Component getPanel(Component component);
    
    
// -- Setters : ----------------------------------------------------------------
    
    void set(Component panel, Object key, Object value);
    
    
// -- Event and listener handling : --------------------------------------------
    
    void addListener(PanelListener listener);
    
    void addListener(PanelListener listener, Component panel);
    
    void removeListener(PanelListener listener);
    
    void removeListener(PanelListener listener, Component panel);

}
