package org.lsst.ccs.gconsole.plugins.trending.timeselection;

import java.awt.Dimension;
import java.awt.Window;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.EventObject;
import javafx.application.Platform;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;

/**
 * GUI panel for entering a custom time window.
 *
 * @author onoprien
 */
public class TimeWindowDialog extends JPanel {

// -- Private parts : ----------------------------------------------------------
    
    private final int VSPACE = 5;
    private final int HSPACE = 10;
    
    private final JPanel startPanel, endPanel, durationPanel;
    private final JLabel agoLabel;
    private final JTimeDateChooser startChooser, endChooser;
    private final JDurationChooser startDuraChooser, durationChooser;
    private final ButtonGroup startGroup, endGroup;
    private final JRadioButton startFixedButton, endFixedButton;
    private final JRadioButton startRelButton, endRelButton;
    private final JTextField nameField;
    private final JCheckBox saveCheck;
    
    static private TimeWindow range;
    static private JButton okButton;
    
    private final Object[] lastChoosers = new Object[2]; // most recently used choosers

    
// -- Construction : -----------------------------------------------------------
    
    private TimeWindowDialog() {
        
        setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
        
        // Start panel
        
        startPanel = new JPanel();    
        startPanel.setLayout(new BoxLayout(startPanel, BoxLayout.Y_AXIS));
        startPanel.setAlignmentX(LEFT_ALIGNMENT);
        startPanel.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createTitledBorder(" Starts "),
                BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));

        Box box = Box.createHorizontalBox();
        startGroup = new ButtonGroup();
        startFixedButton = new JRadioButton("Fixed: ");
        startGroup.add(startFixedButton);
        box.add(startFixedButton);
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        startChooser = JTimeDateChooser.getInstance();
        box.add(startChooser);
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        box.add(Box.createHorizontalGlue());
        startPanel.add(box);

        startPanel.add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));

        box = Box.createHorizontalBox();
        add(box);
        startRelButton = new JRadioButton("Relative: ");
        startGroup.add(startRelButton);
        box.add(startRelButton);
        startDuraChooser = new JDurationChooser();
        box.add(startDuraChooser);
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        box.add(agoLabel = new JLabel("ago."));
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        box.add(Box.createHorizontalGlue());
        startPanel.add(box);

        startPanel.add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));
        add(startPanel);
       
        add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));
        
        // End panel
        
        endPanel = new JPanel();    
        endPanel.setLayout(new BoxLayout(endPanel, BoxLayout.Y_AXIS));
        endPanel.setAlignmentX(LEFT_ALIGNMENT);
        endPanel.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createTitledBorder(" Ends "),
                BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));

        box = Box.createHorizontalBox();
        endGroup = new ButtonGroup();
        endFixedButton = new JRadioButton("Fixed: ");
        endGroup.add(endFixedButton);
        box.add(endFixedButton);
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        endChooser = JTimeDateChooser.getInstance();
        box.add(endChooser);
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        box.add(Box.createHorizontalGlue());
        endPanel.add(box);

        endPanel.add(Box.createRigidArea(new Dimension(0, VSPACE)));

        box = Box.createHorizontalBox();
        add(box);
        endRelButton = new JRadioButton("Relative:  Now");
        endGroup.add(endRelButton);
        box.add(endRelButton);
        box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        box.add(Box.createHorizontalGlue());
        endPanel.add(box);

        endPanel.add(Box.createRigidArea(new Dimension(0, VSPACE)));
        add(endPanel);
       
        add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));
        
        // Duration panel
        
        durationPanel = new JPanel();
        durationPanel.setLayout(new BoxLayout(durationPanel, BoxLayout.Y_AXIS));
        durationPanel.setBorder(BorderFactory.createCompoundBorder(
                    BorderFactory.createTitledBorder(" Duration "), 
                    BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));
        durationChooser = new JDurationChooser(1, ChronoUnit.DAYS);
        durationChooser.setAmountLimits(1, Integer.MAX_VALUE);
        durationPanel.add(durationChooser);
        durationPanel.add(Box.createHorizontalGlue());
        durationPanel.setAlignmentX(LEFT_ALIGNMENT);
        add(durationPanel);
         
        add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));
        
        // Remember As panel
       
        Box saveBox = Box.createVerticalBox();
        saveBox.setAlignmentX(LEFT_ALIGNMENT);
        saveBox.setBorder(BorderFactory.createCompoundBorder(
                    BorderFactory.createTitledBorder(" Remember As "), 
                    BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));
        nameField = new JTextField();
        nameField.setColumns(30);
        nameField.addCaretListener(new CaretListener() {
            public void caretUpdate(CaretEvent e) {
                saveCheck.setEnabled(!nameField.getText().trim().isEmpty());
            }
        });
        saveBox.add(nameField);
        saveBox.add(Box.createRigidArea(new Dimension(0, VSPACE)));
        saveCheck = new JCheckBox("Remember between sessions");
        saveBox.add(saveCheck);
        add(saveBox);
        
        set(range);
        
        startFixedButton.addActionListener(this::onUserAction);
        startRelButton.addActionListener(this::onUserAction);
        endFixedButton.addActionListener(this::onUserAction);
        endRelButton.addActionListener(this::onUserAction);
        startChooser.addListener(this::onUserAction);
        startDuraChooser.addListener(this::onUserAction);
        endChooser.addListener(this::onUserAction);
        durationChooser.addListener(this::onUserAction);
    }


// -- Displaying the GUI : -----------------------------------------------------
    
    /**
     * Opens trending time window creation/editing dialog.
     * 
     * @param parent Parent component.
     * @param seed Time window to be edited, or {@code null} if a new time window should be created.
     * @return Modified or created time window, or {@code null} if the dialog is canceled.
     */
    static public TimeWindow editTimeWindow(JComponent parent, TimeWindow seed) {
        range = seed;
        okButton = new JButton("  OK  ");
        final TimeWindowDialog tsp = new TimeWindowDialog();
        okButton.addActionListener(e -> {
            tsp.getRange();
            Window w = SwingUtilities.getWindowAncestor(tsp);
            if (w != null) w.setVisible(false);
        });
        JButton cancelButton = new JButton("Cancel");
        cancelButton.addActionListener(e -> {
            range = null;
            Window w = SwingUtilities.getWindowAncestor(tsp);
            if (w != null) w.setVisible(false);
        });
        JOptionPane.showOptionDialog(parent, tsp, "Select time range", JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE, null, new Object[] {okButton,cancelButton}, null);
        okButton = null;
        return range;
    }

    
// -- Local methods : ----------------------------------------------------------
    
    private void checkValidy() {
        boolean ok = false;
        okButton.setEnabled(ok);
    }
    
    private TimeWindow getRange() {
        String name = nameField.getText().trim();
        boolean persistent = !name.isEmpty() && saveCheck.isSelected();
        long start =  startFixedButton.isSelected() ? startChooser.get().getEpochSecond() : -startDuraChooser.get();
        long end = endFixedButton.isSelected() ? endChooser.get().getEpochSecond() : 0L;
        if (range == null) {
            range = new TimeWindow(name, start, end, persistent);
        } else {
            range.set(name, start, end, persistent);
        }
        return range;
    }
    
    private void set(TimeWindow window) {
    
        if (window == null) {
            window = new TimeWindow("", "now - 86400", "now", false);
        }
        
        Instant startTime = null;
        Instant endTime;
        boolean startFixed = window.isLowerEdgeFixed();
        boolean endFixed = window.isUpperEdgeFixed();
        
        if (startFixed) {
            startFixedButton.setSelected(true);
            startTime = Instant.ofEpochMilli(window.getLowerEdge(0L));
            startChooser.set(startTime);
        } else {
            startRelButton.setSelected(true);
            long dura = window.getLowerEdgeDelay();
            startDuraChooser.set(dura/1000L);
        }
        
        if (endFixed) {
            endFixedButton.setSelected(true);
            endTime = Instant.ofEpochMilli(window.getUpperEdge(0L));
            endChooser.set(endTime);
            durationChooser.set(Duration.between(startTime, endTime).getSeconds());
        } else {
            endRelButton.setSelected(true);
        }
        
        onUserAction(null);
        
        saveCheck.setSelected(window.isPersistent());
        String name = window.getName();
        if (name != null && !name.isEmpty()) {
            nameField.setText(name);
            saveCheck.setEnabled(true);
        } else {
            saveCheck.setEnabled(false);
        }
    }
    
    private void onUserAction(Object source) {
        
        if (source instanceof EventObject) {
            source = ((EventObject)source).getSource();
        }
        
        if (source == startChooser) {
            
            Instant start = startChooser.get();
            if (endFixedButton.isSelected()) {
                Object last = lastChooser(source);
                Instant end = endChooser.get();
                long duration = durationChooser.get();
                long gap = Duration.between(start, end).getSeconds();
                if ((last == null || last == endChooser) && (gap > 0)) {
                    durationChooser.set(gap);
                } else {
                    endChooser.set(start.plusSeconds(duration));
                }
            } else {
                long beforeNow = Duration.between(start, Instant.now()).getSeconds();
                if (beforeNow <= 0) {
                    endFixedButton.setSelected(true);
                    endChooser.set(start.plus(1, ChronoUnit.DAYS));
                    durationChooser.set(1, ChronoUnit.DAYS);
                }
            }
            
        } else if (source == endChooser) {
            
            Instant start = startChooser.get();
            Instant end = endChooser.get();
            long gap = Duration.between(start, end).getSeconds();
            long duration = durationChooser.get();
            Object last = lastChooser(source);
            if ((last == null || last == startChooser) && (gap > 0)) {
                durationChooser.set(gap);
            } else {
                startChooser.set(end.minusSeconds(duration));
            }
            
        } else if (source == durationChooser) {
            
            long duration = durationChooser.get();
            if (endFixedButton.isSelected()) {
                Object last = lastChooser(source);
                if (last == null || last == endChooser) {
                    Instant end = endChooser.get();
                    Instant start = end.minusSeconds(duration);
                    startChooser.set(start);
                } else {
                    Instant start = startChooser.get();
                    Instant end = start.plusSeconds(duration);
                    endChooser.set(end);
                }
            }
            
        } else if (source == startDuraChooser) {
            
//            durationChooser.set(startDuraChooser.get());
            
        } else if (source == startFixedButton) {
            
            ZonedDateTime end = ZonedDateTime.now().truncatedTo(ChronoUnit.DAYS);
            endChooser.set(end);
            ZonedDateTime start = end.minusDays(1);
            startChooser.set(start);
            durationChooser.set(1, ChronoUnit.DAYS);
            
        } else if (source == startRelButton) {
            
            if (endFixedButton.isSelected()) {
                endRelButton.setSelected(true);
                onUserAction(endRelButton);
                return;
            } else {
                Instant start = startChooser.get();
                long dura = Duration.between(start, Instant.now()).getSeconds();
                startDuraChooser.setRounded(dura);
            }
            
        } else if (source == endFixedButton) {
            
            if (startFixedButton.isSelected()) {
                Instant start = startChooser.get();
                Instant end = Instant.now();
                long dura = Duration.between(start, end).getSeconds();
                durationChooser.set(dura);
            } else {
                startFixedButton.setSelected(true);
                onUserAction(startFixedButton);
                return;
            }
            
        } else if (source == endRelButton) {
                        
        } else {
                        
        }
        
        updateEnable();
        
        // update disabled choosers
        
        if (startRelButton.isSelected()) {
            startChooser.set(Instant.now().minusSeconds(startDuraChooser.get()));
        } else {
            startDuraChooser.setRounded(Duration.between(startChooser.get(), Instant.now()).getSeconds());
        }
        
        if (endRelButton.isSelected()) {
            endChooser.set(Instant.now());
            if (startRelButton.isSelected()) {
                durationChooser.set(startDuraChooser.get());
            } else {
                durationChooser.setRounded(Duration.between(startChooser.get(), Instant.now()).getSeconds());
            }
        }
        
    }
    
    private void updateEnable() {
        
        // enable/disable choosers
        
        if (startRelButton.isSelected()) { // all relative
            startChooser.setEnabled(false);
            startDuraChooser.setEnabled(true);
            agoLabel.setEnabled(true);
            endChooser.setEnabled(false);
            durationChooser.setEnabled(false);
            durationPanel.setEnabled(false);
        } else if (endRelButton.isSelected()) { // start fixed, end relative
            startChooser.setEnabled(true);
            startDuraChooser.setEnabled(false);
            agoLabel.setEnabled(false);
            endChooser.setEnabled(false);
            durationChooser.setEnabled(false);
            durationPanel.setEnabled(false);
        } else { // all fixed
            startChooser.setEnabled(true);
            startDuraChooser.setEnabled(false);
            agoLabel.setEnabled(false);
            endChooser.setEnabled(true);
            durationChooser.setEnabled(true);
            durationPanel.setEnabled(true);
        }
        
    }

     
    Object lastChooser(Object chooser) {
        Object last = lastChoosers[0];
        if (last == chooser) {
            return lastChoosers[1];
        } else {
            lastChoosers[1] = last;
            lastChoosers[0] = chooser;
            return last;
        }
    }

    
// -- Testing : ----------------------------------------------------------------
    
    static public void main(String... args) {
        
        SwingUtilities.invokeLater(() -> {

            TimeWindow tw = null;
            while (true) {
                tw = TimeWindowDialog.editTimeWindow(null, tw);
                System.out.println(tw);
                if (tw == null) {
                    Platform.exit();
                    break;
                }
            }
        });

                
    }
                
}
