package org.lsst.ccs.gconsole.plugins.trending;

import java.util.Map;

/**
 * Time history data for a specific channel and a specific time window.
 * The data includes all metadata available for .
 *
 * @author onoprien
 */
public class TrendData {
    
// -- Fields : -----------------------------------------------------------------
    
    private final Map<String, long[]> times;
    private final Map<String, double[]> values;
    private final long[] timeRange;
    
// -- Life cycle : -------------------------------------------------------------
    
    public TrendData(Map<String, long[]> times, Map<String, double[]> values) {
        this(times, values, null);
    }
    
    public TrendData(Map<String, long[]> times, Map<String, double[]> values, long[] timeRange) {
        this.times = times;
        this.values = values;
        this.timeRange = timeRange;
    }
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns an array of time values for points in this history.
     * @return An array of times, in milliseconds.
     */
    public long[] getTime() {
        return times.get("value");
    }
    
    /**
     * Returns an array of values for points in this history.
     * @return An array of values.
     */
    public double[] getValue() {
        return values.get("value");
    }

    /**
     * Returns an array of time points available in this time history for the specified metadata type.
     * 
     * @param key String that identifies the metadata.
     * @return An array of times, in milliseconds.
     */
    public long[] getTime(String key) {
        return times.get(key);
    }
    
    /**
     * Returns an array of values contained in this time history for the specified metadata type.
     * 
     * @param key String that identifies the metadata.
     * @return An array of values for the specified data.
     */
    public double[] getValue(String key) {
        return values.get(key);
    }
    
    /**
     * Returns earliest time for data points contained in this dataset.
     * @return Lowest time for data points contained in this dataset.
     */
    public long getLowT() {
        long[] time = getTime();
        if (time == null || time.length == 0) {
            return System.currentTimeMillis() - 3600000L;
        } else {
            return time[0];
        }
    }
    
    /**
     * Returns latest time for data points contained in this dataset.
     * @return Highest time for data points contained in this dataset. 
     */
    public long getHighT() {
        long[] time = getTime();
        if (time == null || time.length == 0) {
            return System.currentTimeMillis();
        } else {
            return time[time.length - 1];
        }
    }
    
    /**
     * Returns the time range of this data set.
     * @return two-element array containing the bounds of the time range of this data set.
     */
    public long[] getTimeRange() {
        return timeRange == null ? new long[] {getLowT(), getHighT()} : timeRange;
    }
    
}
