package org.lsst.ccs.gconsole.agent;

import java.util.*;

/**
 * A filter that selects a set of data channels and defines their presentation.
 * Original channels are identified by their unique string trending paths (slash-separated,
 * the first segment is the subsystem name).  This filter maps some of the original
 * channels to one or more display paths in "[page//]path1/.../pathN/name" format.
 *
 * @author onoprien
 */
public interface AgentChannelsFilter {
    
    /**
     * Trivial filter that accepts all channels and does not modify paths.
     * This filter uses default implementations for all methods.
     */
    static AgentChannelsFilter PASS = new AgentChannelsFilter() {
        @Override
        public String getName() {return "PASS";}
    };
    
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns the name of this filter.
     * 
     * @return Name of this filter.
     */
    default String getName() {
        return null;
    }
    
    /**
     * Returns a set of subsystem names accepted by this filter.
     * 
     * @return Subsystem names accepted by this filter, or {@code null} if any subsystem might be accepted.
     */
    default Set<String> getSubsystems() {
        return null;
    }
    
    /**
     * Returns a set of original channel paths accepted by this filter.
     * 
     * @return Original channel paths accepted by this filter, or {@code null} if any
     * original path for which {@link #getDisplayPath} returns a non-empty list is accepted.
     */
    default Set<String> getOriginChannels() {
        return null;
    }
    
    /**
     * Returns a set of display paths produced by this filter.
     * Typically, if this method returns a non-null set, components downstream from
     * this filter will display this fixed set of channels, whether or not the data
     * is available, regardless of what paths are returned by {@link #getDisplayPath}
     * applied to accepted original channels. Otherwise, any display channels for
     * which the data is available will be displayed.
     * 
     * @return Display paths produced by this filter, or {@code null} if the set is not fixed.
     */
    default Set<String> getDisplayChannels() {
        return null;
    }
    
    /**
     * Returns the original path corresponding to the specified display channel.
     * 
     * @param displayPath The display channel path.
     * @return The original path, or {@code null} if the specified path does not correspond to any original channel.
     */
    default String getOriginPath(String displayPath) {
        return displayPath;
    }
    
    /**
     * Returns a list of display channels for the specified original channel.
     * 
     * @param originPath The path of the original channel.
     * @return Display paths, or an empty list if the specified channel is not accepted by this filter.
     */
    default List<String> getDisplayPath(String originPath) {
        return Collections.singletonList(originPath);
    }
    
    /**
     * Returns attributes associated with the specified path.
     *
     * @param path Full or partial display path.
     * @return Map of attribute names to their values.
     */
    default Map<String,Object> getAttributes(String path) {
        return Collections.emptyMap();
    }
    
    
// -- Handling listeners : -----------------------------------------------------
    
    /**
     * Registers a listener that will be notified of any changes in this filter.
     * 
     * @param listener Listener to be notified.
     */
    default void addListener(Listener listener) {}
    
    /**
     * Removes a listener.
     * 
     * @param listener Listener to be removed.
     */
    default void removeListener(Listener listener) {}
    
    /**
     * Interface to be implemented by classes that should be notified of changes in the {@link AgentChannelsFilter}.
     */
    interface Listener {
        default void filterChanged(AgentChannelsFilter filter) {}
    }
    
}
