package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;

/**
 * A filter that defines content of one or more monitoring panels.
 *
 * @author onoprien
 */
public interface MonitorFilter {
    
    /**
     * Trivial filter that accepts all channels and does not modify paths.
     * This filter uses default implementations for all methods.
     */
    static MonitorFilter PASS = new MonitorFilter() {};
    
    /**
     * Returns a set of subsystem names accepted by this filter.
     * 
     * @return Subsystem names accepted by this filter, or {@code null} if any subsystem might be accepted.
     */
    default Set<String> getSubsystems() {
        return null;
    }
    
    /**
     * Returns a set of original channel paths accepted by this filter.
     * 
     * @return Original channel paths accepted by this filter, or {@code null} if any
     * original path for which {@link #getDisplayPath} returns a non-empty list is accepted.
     */
    default Set<String> getOriginChannels() {
        return null;
    }
    
    /**
     * Returns a set of display paths produced by this filter.
     * Typically, if this method returns a non-null set, components downstream from
     * this filter will display this fixed set of channels, whether or not the data
     * is available, regardless of what paths are returned by {@link #getDisplayPath}
     * applied to accepted original channels. Otherwise, any display channels for
     * which the data is available will be displayed.
     * 
     * @return Display paths produced by this filter, or {@code null} if the set is not fixed.
     */
    default Set<String> getDisplayChannels() {
        return null;
    }
    
    /**
     * Returns the original path corresponding to the specified display channel.
     * 
     * @param displayPath The display channel path.
     * @return The original path, or {@code null} if the specified path does not correspond to any original channel.
     */
    default String getOriginPath(String displayPath) {
        return displayPath;
    }
    
    /**
     * Returns a list of display channels for the specified original channel.
     * 
     * @param originPath The path of the original channel.
     * @return Display paths, or an empty list if the specified channel is not accepted by this filter.
     */
    default List<String> getDisplayPath(String originPath) {
        return Collections.singletonList(originPath);
    }
    
//    /**
//     * Returns the name of the page where the specified channel must be displayed.
//     * 
//     * @param displayPath The display channel path.
//     * @return The name of the page, or {@code null} if the page is not specified by this filter.
//     */
//    default String getPage(String displayPath) {
//        return null;
//    }
    
    /**
     * Returns the channel name for the specified display path.
     * 
     * @param displayPath The display channel path.
     * @return The name of the display channel, or {@code null} if the name is not specified by this filter.
     */
    default String getName(String displayPath) {
        return null;
    }
    
}
