package org.lsst.ccs.gconsole.plugins.alert;

import java.util.*;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.states.AlertState;

/**
 * An event fired by {@link AlertService} to notify listeners of raised or cleared alerts.
 *
 * @author onoprien
 */
public class AlertEvent extends EventObject {

// -- Fields : -----------------------------------------------------------------
    
    private final Alert alert;
    private final RaisedAlertSummary summary;
    private final List<String> clear;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructor for events triggered by a raised alert.
     * 
     * @param source Alert source (typically, subsystem name).
     * @param alert Alert instance.
     * @param summary Current alert summary for the specified source.
     */
    public AlertEvent(String source, Alert alert, RaisedAlertSummary summary) {
        super(source);
        this.alert = alert;
        this.summary = summary;
        this.clear = null;
    }
    
    /**
     * Constructor for events triggered by cleared alerts.
     * 
     * @param source Alert source (typically, subsystem name).
     * @param clearedIDs IDs of cleared alerts.
     * @param summary Current alert summary for the specified source.
     */
    public AlertEvent(String source, String[] clearedIDs, RaisedAlertSummary summary) {
        super(source);
        this.alert = null;
        this.summary = summary;
        this.clear = Collections.unmodifiableList(new ArrayList<>(Arrays.asList(clearedIDs)));
    }
    
    
// -- Getters : ----------------------------------------------------------------

    /**
     * Returns the source of the alert that triggered this event.
     * For alerts received over the buses, this is subsystem name.
     * 
     * @return the source of the alert.
     */
    @Override
    public String getSource() {
        return (String) super.getSource();
    }

    /**
     * Returns the {@code Alert} that triggered this event.
     * Returns {@code null} if this event was triggered by clearing alerts.
     * 
     * @return the {@code Alert} that triggered this event.
     */
    public Alert getAlert() {
        return alert;
    }

    /**
     * Returns the level ({@code WARNING} or {@code ALARM}) at which the alert that triggered this
     * event was raised. Returns {@code NOMINAL} if this event was triggered by clearing alerts.
     * 
     * @return the level at which the alert that triggered this event was raised.
     */
    public AlertState getLevel() {
        return alert == null ? AlertState.NOMINAL : summary.getRaisedAlert(alert.getAlertId()).getLatestAlertState();
    }

    /**
     * Returns the current summary of alerts for the source of the alert that triggered this event.
     * 
     * @return the current summary of alerts.
     */
    public RaisedAlertSummary getSummary() {
        return summary;
    }
    
    /**
     * Returns the list of cleared IDs.
     * Returns {@code null} if this event was not triggered by clearing alerts.
     * 
     * @return the list of cleared IDs.
     */
    public List<String> getClearedIds() {
        return clear;
    }
    
    
// -- Overriding Object : ------------------------------------------------------
    
    @Override
    public String toString() {
        return getSource() +": "+ getAlert();
    }
    
}
