package org.lsst.ccs.gconsole.base;

import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.swing.Action;

/**
 * Access to services provided by the framework to graphical console plugins.
 * Each plugin has an instance of {@code ConsolePluginServices} associated with it.
 *
 * @author onoprien
 */
abstract public class ConsolePluginServices implements Console.PropertyListener {

// -- Fields : -----------------------------------------------------------------
    
    private final ConsolePlugin plugin;
    private final ComponentDescriptor descriptor;
    private final Console console;
    
    protected final String prefix;

    
// -- Life cycle : -------------------------------------------------------------
    
    protected ConsolePluginServices(Console console, ConsolePlugin plugin, ComponentDescriptor descriptor) {
        this.console = console;
        this.plugin = plugin;
        this.descriptor = descriptor;
        prefix = "lsst.plugin."+ descriptor.getName() +".";
    }
    
    public void start() {
        console.addPropertyListener(this, prefix.replace(".", "\\.") + ".+");
    }
    
    public void stop() {
        console.removePropertyListener(this);
    }

    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns a reference to the graphical console subsystem.
     * @return Graphical console subsystem instance.
     */
    public Console getConsole() {
        return console;
    }
    
    /**
     * Returns a reference to the graphical console plugin associated with this {@code ConsolePluginServices} instance.
     * @return Graphical console plugin.
     */
    public ConsolePlugin getPlugin() {
        return plugin;
    }
    
    /**
     * Returns the descriptor for the graphical console plugin associated with this {@code ConsolePluginServices} instance.
     * @return Graphical console plugin descriptor.
     */
    public ComponentDescriptor getDescriptor() {
        return descriptor;
    }


// -- Adding menues : ----------------------------------------------------------
    
    /**
     * Adds an item to the graphical console menu bar.
     * <p>
     * The location strings specify a hierarchy of menus above the added item. Each
     * string should be in the {@code name[[:group]:position]} format, where
     * {@code name} is the text label of the menu the item should be added to,
     * {@code group} is the ordinal number of the item group inside the menu, and 
     * {@code position} is the position inside the group.
     * The first location string can also be prefixed with the desired location 
     * on the menu bar: {@code [barPosition:]name[[:group]:position]}.
     * 
     * The menu is divided into groups by separators.
     * If the group number is negative, items added with the same group number by 
     * different plugins are put into the same group. If the group is not specified
     * or is equal to 0, the item is added to the default group. The default group
     * is shared by all plugins.
     * 
     * @param action Action for the menu item to be added.
     * @param locations Strings that specifies locations of the added item at all levels in the menu hierarchy.
     */
    public void addMenu(Action action, String... locations) {
        console.addMenu(action, getDescriptor().getName(), locations);
    }
    
    
// -- Handling properties : ----------------------------------------------------
    
    /**
     * Defines a plugin property with the specified default value.
     * Console properties can be later modified by calls to the {@link #setProperty(String, Object)} method.
     * Properties that have been modified from their default values are saved between the graphical console sessions.
     * <p>
     * If a property with the specified key already exists:<ul>
     * <li>if the specified default value is of the same type as the existing one, the call has no effect;
     * <li>if the existing default value is of type {@code String} but can be converted to the type of the newly specified
     *     default value, the existing default value is retained but the type of object returned by subsequent calls to
     *     {@link #getProperty(String)} method is modified;
     * <li>if the specified default value is incompatible with the existing one, {@code IllegalArgumentException} is thrown;
     * <li>if the existing property does not have a default value, its current value is reset to the provided default.
     * </ul>
     * When a property is retrieved with a call to {@link #getProperty(String)} method, the type of the
     * returned object is determined by the type of the default value supplied when the property was defined.
     * 
     * @param key Property key. Can be used to retrieve the property value.
     * @param defaultValue Default value for the property. 
     *        Must be of type {@code Boolean}, {@code Enum}, {@code Integer}, {@code Long}, {@code Double}, {@code String},
     *        or an array containing objects of these types.
     * @return Current value of the property.
     * @throws IllegalArgumentException If the specified key already exists, and its default value is incompatible with {@code defaultValue};
     *                                  if either {@code key} or {@code defaultValue} is {@code null};
     *                                  if the type of {@code defaultValue} is not supported.
     */
    public Object addProperty(String key, Object defaultValue) {
        return getConsole().addProperty(translatePropertyKey(key), defaultValue);
    }
    
    /**
     * Removes a plugin property.
     * If the specified property has been previously defined through a call to {@link #addProperty(String, Object)}
     * method, the property is removed and its current value is removed from storage; calling this method for a 
     * property that has not been defined has no effect.
     * 
     * @param key Property key.
     * @return Current value of the property, or {@code null} if there was no such property.
     */
    public Object removeProperty(String key) {
        return getConsole().removeProperty(translatePropertyKey(key));
    }
    
    /**
     * Adds a setter for one or more previously defined properties to the preferences menu.
     * 
     * @param path Path in the preferences tree to the page where the property setter should be displayed.
     * @param group Group inside the preferences page. If {@code null}, the property setter is displayed in the default group.
     * @param format Specifies the format of the property setter.
     * @throws IllegalArgumentException If a property with the specified key already exists, and its value is incompatible with the provided default.
     */
    public void addPreference(String[] path, String group, String format) {
        StringBuffer sb = new StringBuffer();
        Matcher m = Pattern.compile("\\$\\{([^${}]+)\\}").matcher(format);
        while (m.find()) {
            m.appendReplacement(sb, "");
            sb.append("${").append(translatePropertyKey(m.group(1))).append("}");
        }
        m.appendTail(sb);
        format = sb.toString();
        getConsole().addPreference(path, group, format);
    }
    
    /**
     * Returns the value of the property with the specified key.
     * The type of the returned object is determined by the type of the default value supplied when the property was defined.
     * If the property with the specified key has never been defined through a call to {@link #addProperty(String, Object)}
     * but exists in the console storage, it will be returned as a {@code String}.
     * 
     * @param key Property key.
     * @return Value of the specified property, or {@code null} if the property does not exist.
     * @throws IllegalArgumentException If the stored value is of type incompatible with the property type.
     */
    public Object getProperty(String key) {
        return getConsole().getProperty(translatePropertyKey(key));
    }
   
    /**
     * Sets the value of the specified plugin property.
     * If the property with the specified key has never been defined through a call 
     * to {@link #addProperty(String, Object)} method, the value will be converted 
     * to a {@code String} and stored. Such keys cannot be used in setter format 
     * strings passed to {@link #addPreference(String[], String, String)} method.
     * <p>
     * If the property value is set to {@code null}, it is removed from storage, so
     * subsequent calls to {@link #getProperty(String)} will return the default
     * value, if it has been set.
     * 
     * @param key Property key.
     * @param value Property value, or {@code null} if the property is being reset to its default value.
     * @return The previous value of the specified property, or {@code null} if the property did not exist.
     */
    public Object setProperty(String key, Object value) {
        return getConsole().setProperty(translatePropertyKey(key), value);
    }
   
    /**
     * Sets values of a set of properties.
     * See {@link #setProperty(String, Object)} for details on setting properties.
     * 
     * @param properties
     */
    public void setProperties(Map<String,Object> properties) {
        properties = properties.entrySet().stream().collect(Collectors.toMap(e -> translatePropertyKey(e.getKey()), e -> e.getValue()));
        getConsole().setProperties(properties);
    }
    
    protected String translatePropertyKey(String key) {
        return prefix + key;
    }

    @Override
    public void propertiesChanged(Object source, Map<String, Object> changes) {
        HashMap<String, Object> out = new HashMap<>(changes.size()*2);
        changes.forEach((key,value) -> {
            if (key.startsWith(prefix)) {
                out.put(key.substring(prefix.length()), value);
            }
        });
        if (!out.isEmpty()) {
            plugin.propertiesChanged(source, out);
        }
    }
    
}
