package org.lsst.ccs.gconsole.plugins.trending;

import com.sun.jersey.api.client.WebResource;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import javax.ws.rs.core.MediaType;
import org.lsst.ccs.localdb.statusdb.server.ChannelMetaData;
import org.lsst.ccs.localdb.statusdb.server.Data;
import org.lsst.ccs.localdb.statusdb.server.DataChannel;
import org.lsst.ccs.localdb.statusdb.server.TrendingData;
import org.lsst.ccs.localdb.statusdb.server.TrendingResult;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Time history source that provides access to trending channels available through a REST server.
 *
 * @author onoprien
 */
public class RestServerSource extends TrendingSource {

// -- Fields : -----------------------------------------------------------------
    
    private final WebResource server;
    private final Logger logger;
    private final TrendingPreferences pref;
    private final Map<TrendingChannel,DataChannel> channels = new ConcurrentHashMap<>();


// -- Life cycle : -------------------------------------------------------------
    
    public RestServerSource(WebResource server, LsstTrendingPlugin plugin) {
        this.server = server;
        logger = plugin.getConsole().getLogger();
        pref = plugin.getPreferences();
    }
    
    
// -- Connecting : -------------------------------------------------------------
    
    public boolean connect() {
        channels.clear();
        try {
            DataChannel.DataChannelList channelList = server.path("listchannels").accept(MediaType.TEXT_XML).get(DataChannel.DataChannelList.class);
            logger.info("Connection successful, "+ channelList.list.size() +" channels read from "+ server.getURI());
            channels.clear();
            channelList.list.forEach(channel -> channels.put(new TrendingChannel(channel.getPathAsString()), channel));
            return true;
        } catch (RuntimeException e) {
            logger.warn("Unable to connect to the REST server at "+ server.getURI());
            return false;
        }
    }

// -- Implementing TrendingSource : -----------------------------------------

    @Override
    public List<TrendingChannel> getChannels() {
        return new ArrayList<>(channels.keySet());
    }

    @Override
    public TrendData get(TrendingChannel channel, long begin, long end, EnumSet<Trend.Meta> metadata, TrendData history) {
        
        DataChannel dc = channels.get(channel);
        if (dc == null) return null;
        WebResource resource = server.path("data").path(String.valueOf(dc.getId()));
        resource = resource.queryParam("t1", String.valueOf(begin)).queryParam("t2", String.valueOf(end));
        int nBins = pref.getNBins();
        if (nBins > 0) {
            if (!pref.isUseRawData()) {
                resource = resource.queryParam("flavor", "stat");
            }
            resource = resource.queryParam("n", Integer.toString(nBins));
        } else {
            resource = resource.queryParam("flavor", pref.isUseRawData() ? "raw" : "stat");
        }
        Data data = resource.accept(MediaType.TEXT_XML).get(Data.class);
        TrendingResult result = data.getTrendingResult();
            
        Set<String> onPoint, offPoint;
        if (metadata == null) {
            onPoint = null;
            offPoint = null;
        } else {
            onPoint = new HashSet<>();
            onPoint.add("value");
            offPoint = new HashSet<>();
            metadata.forEach(m -> {
                if (m.isOnPoint()) {
                    onPoint.addAll(m.getKeys());
                } else {
                    offPoint.addAll(m.getKeys());
                }
            });
        }
        
        Map<String, long[]> times = new HashMap<>();
        Map<String, double[]> values = new HashMap<>();
        
        long lowEdge = begin;
        long highEdge = end;
        TrendingData[] dataArray = result.getTrendingDataArray();
        if (dataArray != null) {
            int n = dataArray.length;
            long[] time = new long[n];
            times.put("value", time);
            for (int i=0; i<n; i++) {
               TrendingData td = dataArray[i];
               time[i] = td.getAxisvalue().getValue();
               for (TrendingData.DataValue dv : td.getDatavalue()) {
                   String key = dv.getName();
                   if (onPoint == null || onPoint.contains(key)) {
                       double[] valueArray = values.get(key);
                       if (valueArray == null) {
                           valueArray = new double[n];
                           values.put(key, valueArray);
                       }
                       valueArray[i] = dv.getValue();
                   }
               }
            }
//            if (n > 0) {
//                lowEdge = time[0];
//                highEdge = time[n-1];
//            }
        }
        
        List<ChannelMetaData> metaList = result.getChannelMetadata();
        HashMap<String,ArrayList<MetaPoint>> metaMap = new HashMap<>();
        for (ChannelMetaData meta : metaList) {
            String key = meta.getName();
            if (offPoint == null || offPoint.contains(key)) {
                try {
                    ArrayList<MetaPoint> ps = metaMap.get(key);
                    if (ps == null) {
                        ps = new ArrayList<>();
                        metaMap.put(key, ps);
                    }
                    double value = Double.parseDouble(meta.getValue());
                    long start = meta.getTstart();
                    long stop = meta.getTstop();
                    if ((start <= highEdge || start == -1L) && (stop >= lowEdge || stop == -1L)) {
                        if (start < lowEdge) {
                            start = lowEdge;
                        }
                        ps.add(new MetaPoint(start, value));
                        if (stop > highEdge || stop == -1) {
                            stop = highEdge;
                        }
                        ps.add(new MetaPoint(stop, value));
                    }
                } catch (NumberFormatException x) {
                }
            }
        }
        
        metaMap.forEach((key, points) -> {
            int n = points.size();
            long[] t = new long[n];
            double[] v = new double[n];
            for (int i=0; i<n; i++) {
                MetaPoint p = points.get(i);
                t[i] = p.time;
                v[i] = p.value;
            }
            times.put(key, t);
            values.put(key, v);
        });
        
//        System.out.println("Fetched data for "+ channel.getPath() +" from "+ new Date(begin) +" to "+ new Date(end));
//        System.out.println("Actual range: " +" from "+ new Date(lowEdge) +" to "+ new Date(highEdge) +", "+ times.get("value").length +" points.");
//        StringBuilder sb = new StringBuilder("Found: \n");
//        times.forEach((key,t) -> sb.append(key).append(":").append(t.length).append("  "));
//        System.out.println(sb);
        
        return new TrendData(times, values, new long[] {begin, end});
    }
    
    
// Local methods and classes : -------------------------------------------------
    
    static private class MetaPoint {
        MetaPoint(long t, double v) {
            time = t;
            value = v;
        }
        long time;
        double value;
    }
    
}
