package org.lsst.ccs.gconsole.jas3;

import java.lang.reflect.Type;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.utilities.conv.InputConversionEngine;
import org.lsst.ccs.utilities.conv.TypeConversionException;
import org.lsst.ccs.utilities.conv.TypeUtils;

/**
 * Handles storing and retrieving properties in the underlying {@code Properties} instance.
 * <p>
 * This class implements property handling methods specified by {@link Console}.
 * See that class javadoc for details.
 *
 * @author onoprien
 */
public class PropertyHandler {

// -- Fields : -----------------------------------------------------------------
    
    private final Properties prop;
    private final HashMap<String,Object> defaultProperties = new HashMap<>();
    private final CopyOnWriteArrayList<Listener> listeners = new CopyOnWriteArrayList<>();
    

// -- Life cycle : -------------------------------------------------------------
    
    PropertyHandler(Properties properties) {
        prop = properties;
    }
    
    
// -- Operations : -------------------------------------------------------------

    synchronized public void addProperty(String key, Object defaultValue) {
        if (key == null || defaultValue == null) throw new IllegalArgumentException("Arguments cannot be null.");
        Type type = getType(defaultValue);
        Object oldDefault = defaultProperties.get(key);
        if (oldDefault != null && !(getType(oldDefault).equals(type))) {
            throw new IllegalArgumentException("The key is already associated with a property of a different type.");
        }
        defaultProperties.put(key, defaultValue);
    }
    
    synchronized public Object removeProperty(String key) {
        Object value = null;
        Object defaultValue = defaultProperties.remove(key);
        if (defaultValue != null) {
            value = get(key, defaultValue);
            prop.remove(key);
        }
        return value;
    }

    synchronized public Object getProperty(String key) {
        Object defaultValue = defaultProperties.get(key);
        return get(key, defaultValue);
    }

    synchronized public Object setProperty(String key, Object value) {
        Object old = set(key, value);
        if (value == null && old != null) {
            value = defaultProperties.get(key);
            if (!old.equals(value)) {
                notifyListeners(null, key, value);
            }
        } else if (value != null && !value.equals(old)) {
            notifyListeners(null, key, value);
        }
        return old;
    }
    
    synchronized public void setProperties(Map<String,Object> keyValueMap) {
        HashMap<String,Object> changes = new HashMap<>(keyValueMap.size()*2);
        keyValueMap.forEach((key,value) -> {
            Object old = set(key, value);
            if (value == null && old != null) {
                value = defaultProperties.get(key);
                if (!old.equals(value)) {
                    changes.put(key, value);
                }
            } else if (value != null && !value.equals(old)) {
                changes.put(key, value);
            }
        });
        notifyListeners(null, changes);
    }
    
    public void addPropertyListener(Console.PropertyListener listener, String filter) {
        Pattern p;
        if (filter == null) {
            p = null;
        } else {
            try {
                p = Pattern.compile(filter);
            } catch (PatternSyntaxException x) {
                throw new IllegalArgumentException("Illegal regular expression in property listener filter.", x);
            }
        }
        listeners.add(new Listener(listener, p));
    }
    
    public boolean removePropertyListener(Console.PropertyListener listener) {
        return listeners.remove(new Listener(listener, null));
    }
    
    
// -- Local methods : ----------------------------------------------------------

    private Object set(String key, Object value) {
        Type type = getType(value);
        Object defaultValue = defaultProperties.get(key);
        Object previousValue = get(key, defaultValue);
        if (value == null || value.equals(defaultValue)) {
            prop.remove(key);
        } else {
            if (defaultValue != null) {
                Type defaultType = getType(defaultValue);
                if (defaultType != type) throw new IllegalArgumentException("The key is associated with a property of a different type.");
            }
            prop.setProperty(key, pack(value));
        }
        return previousValue;
    }
    
    private void notifyListeners(Object source, String key, Object value) {
        listeners.forEach(listener -> listener.notify(source, key, value));
    }
    
    private void notifyListeners(Object source, Map<String,Object> changes) {
        listeners.forEach(listener -> listener.notify(source, changes));
    }
    
    static Type getType(Object value) {
        if (value == null) {
            return null;
        } else {
            return value.getClass();
        }
    }
    
    /**
     * Returns stored value, or {@code defaultValue} if there is no stored value or the stored value is of incompatible type.
     * If the {@code defaultValue} is {@code null}, returns the stored {@code String}.
     */
    private Object get(String key, Object defaultValue) {
        String s = prop.getProperty(key);
        if (defaultValue == null) return s;
        if (s == null) return defaultValue;
        Type type = getType(defaultValue);
        try {
            return unpack(s, type);
        } catch (IllegalArgumentException x) {
            return defaultValue;
        }
    }
    
    private String pack(Object value) {
        return TypeUtils.stringify(value);
    }
    
    private Object unpack(String data, Type type) {
        try {
            return InputConversionEngine.convertArgToType(data, type);
        } catch (TypeConversionException x) {
            throw new IllegalArgumentException("String '" + data + "' cannot be converted to type "+ type.getTypeName(), x);
        }
    }
    
    
// -- Local classes : ----------------------------------------------------------
    
    private static class Listener {
        Console.PropertyListener listener;
        Pattern pattern;
        Listener(Console.PropertyListener listener, Pattern pattern) {
            this.listener = listener;
            this.pattern = pattern;
        }
        void notify(Object source, Map<String,Object> changes) {
            Map<String,Object> relevantChanges = new HashMap<>();
            changes.forEach((key,value) -> {
                if (pattern.matcher(key).matches()) {
                    relevantChanges.put(key, value);
                }
            });
            if (!relevantChanges.isEmpty()) {
                listener.propertiesChanged(source, Collections.unmodifiableMap(relevantChanges));
            }
        }
        void notify(Object source, String key, Object value) {
            if (pattern.matcher(key).matches()) {
                listener.propertiesChanged(source, Collections.singletonMap(key, value));
            }
        }
        @Override
        public boolean equals(Object obj) {
            if (obj instanceof Listener) {
                return ((Listener)obj).listener == this.listener;
            } else {
                return false;
            }
        }
    }
    
}
