package org.lsst.ccs.gconsole.jas3;

import java.awt.Window;
import java.awt.event.ActionEvent;
import java.beans.XMLDecoder;
import java.beans.XMLEncoder;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JFileChooser;
import javax.swing.SwingUtilities;
import javax.swing.filechooser.FileNameExtensionFilter;
import org.freehep.application.Application;
import org.freehep.application.studio.Studio;
import org.freehep.jas.services.WebBrowser;
import org.freehep.util.FreeHEPLookup;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.openide.util.Lookup;

/**
 * Jas3-based Graphical Console implementation.
 * 
 * @author onoprien
 */
public class Jas3Console extends Console {
    
// -- Fields : -----------------------------------------------------------------
    
    private final PropertyHandler prop;
    private final HashMap<String,PreferencePage> prefs = new HashMap<>();
    private final PanelManager panMan;
    
    
// -- Life cycle : -------------------------------------------------------------
    
    public Jas3Console() {
        prop = new PropertyHandler(getApplication().getUserProperties());
        panMan = new JasPanelManager(getApplication());
    }
    
    @Override
    public void start() {
        super.start();
        
        Action act = new AbstractAction("Save") {
            @Override
            public void actionPerformed(ActionEvent e) {
                save();
            }
        };
        addMenu(act, "", "File:-2:1", "Session:1");
        act = new AbstractAction("Save As...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                saveAs();
            }
        };
        addMenu(act, "", "File:-2:1", "Session:2");
        act = new AbstractAction("Restore") {
            @Override
            public void actionPerformed(ActionEvent e) {
                restore();
            }
        };
        addMenu(act, "", "File:-2:1", "Session:3");
        act = new AbstractAction("Restore From...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                restoreFrom();
            }
        };
        addMenu(act, "", "File:-2:1", "Session:4");
        
        act = new AbstractAction("CCS Console Developer's Guide") {
            @Override
            public void actionPerformed(ActionEvent e) {
                // FIXME: hardcoded url for now, at some point should convert all 
                // LSST links to GC infrastructure and move urls to properties file
                showInBrowser("https://confluence.slac.stanford.edu/display/LSSTCAM/Graphical+Console+Plugin+Developer%27s+Guide");
            }
        };
        addMenu(act, "", "Help", "Documentation:300");

    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    @Override
    public FreeHEPLookup getConsoleLookup() {
        return getApplication().getLookup();
    }

    @Override
    public Window getWindow() {
        return SwingUtilities.getWindowAncestor(Application.getApplication());
    }
    
    public final Studio getApplication() {
        return (Studio) Application.getApplication();
    }
    
    @Override
    public Path getHomeDirectory() {
        Path path = Paths.get(getApplication().getUserProperties().getProperty("lsst.console.home"));
        if (Files.notExists(path)) {
            try {
                Files.createDirectories(path);
            } catch (IOException x) {
            }
        }
        return path;
    }

    @Override
    public PanelManager getPanelManager() {
        return panMan;
    }


// -- Miscellaneous services : -------------------------------------------------
    
    @Override
    public void error(String message) {
        getApplication().error(message);
    }
    
    @Override
    public void error(String message, Exception x) {
        getApplication().error(message, x);
    }

    @Override
    public void showInBrowser(String url) {
        try {
            WebBrowser webBrowser = ((WebBrowser) getConsoleLookup().lookup(WebBrowser.class));
            webBrowser.showURL(new URL(url), true);
        } catch (MalformedURLException | RuntimeException x) {
            this.getLogger().warn("Unable to display "+ url +" in a browser.", x);
        }
    }
    
    @Override
    public void setStatusMessage(String message) {
        getApplication().setStatusMessage(message);
    }
    
    
// -- Handling properties : ----------------------------------------------------

    @Override
    public void addProperty(String key, Object defaultValue) {
        prop.addProperty(key, defaultValue);
    }
    
    @Override
    public Object removeProperty(String key) {
        return prop.removeProperty(key);
    }

    @Override
    public void addPreference(String[] path, String group, String format) {
        PreferencePage page;
        synchronized (prefs) {
            page = prefs.get(Arrays.toString(path));
            if (page == null) {
                page = new PreferencePage(this, path);
                prefs.put(Arrays.toString(path), page);
                getConsoleLookup().add(page);
            }
            page.add(group, format);
        }
    }

    @Override
    public Object getProperty(String key) {
        return prop.getProperty(key);
    }

    @Override
    public Object setProperty(String key, Object value) {
        return prop.setProperty(key, value);
    }
   
    @Override
    public void setProperties(Map<String,Object> properties) {
        prop.setProperties(properties);
    }
    
    @Override
    public void addPropertyListener(PropertyListener listener, String filter) {
        prop.addPropertyListener(listener, filter);
    }
    
    @Override
    public boolean removePropertyListener(PropertyListener listener) {
        return prop.removePropertyListener(listener);
    }


// -- Adding menus, etc. : -----------------------------------------------------
    
    @Override
    public void addMenu(Action action, String owner, String... locations) {
        getApplication().addMenu(action, owner, locations);
    }

    
// -- Restoring/Saving configuration: ------------------------------------------
    
    private Path getDefaultPath() {
        return getHomeDirectory().resolve("config").resolve("default.conf");
    }
    
    public void restore() {
        restore(getDefaultPath());
    }
    
    public void restoreFrom() {
        JFileChooser chooser = new JFileChooser(getHomeDirectory().resolve("config").toFile());
        FileNameExtensionFilter filter = new FileNameExtensionFilter("Graphical Console Configuration", "conf");
        chooser.setFileFilter(filter);
        int out = chooser.showOpenDialog(getWindow());
        if (out == JFileChooser.APPROVE_OPTION) {
            restore(chooser.getSelectedFile().toPath());
        }
    }
    
    public void save() {
        save(getDefaultPath());
    }
    
    public void saveAs() {
        JFileChooser chooser = new JFileChooser(getHomeDirectory().resolve("config").toFile());
        FileNameExtensionFilter filter = new FileNameExtensionFilter("Graphical Console Configuration", "conf");
        chooser.setFileFilter(filter);
        int out = chooser.showSaveDialog(getWindow());
        if (out == JFileChooser.APPROVE_OPTION) {
            Path path = chooser.getSelectedFile().toPath();
            String s = path.toString();
            if (!s.endsWith(".conf")) {
                s += ".conf";
                path = Paths.get(s);
            }
            save(path);
        }
    }
    
    private void restore(Path path) {
        
        // read descriptors from file
        
        Map<String,ComponentDescriptor> descriptors = readDescriptors(path);
        if (descriptors.isEmpty()) return;
            
        ComponentDescriptor consoleDescriptor = descriptors.get("");
        if (consoleDescriptor == null) {
            getLogger().info("Restoring the graphical console configuration.");
        } else {
            getLogger().info("Restoring configuration saved by the graphical console version " + consoleDescriptor.getVersion());
        }
        
        // compile a list of loaded plugins

        HashMap<String, ConsolePlugin> allPlugins = new HashMap<>();
        Lookup.Result result = getConsoleLookup().lookup(new Lookup.Template(ConsolePlugin.class));
        for (Object o : result.allInstances()) {
            ConsolePlugin plugin = (ConsolePlugin) o;
            allPlugins.put(plugin.getServices().getDescriptor().getName(), plugin);
        }
        
        // map descriptors to plugins
        
        LinkedHashMap<ComponentDescriptor,ConsolePlugin> plugins = new LinkedHashMap<>();
        descriptors.forEach((name, descriptor) -> {
            ConsolePlugin plugin = allPlugins.get(name);
            if (plugin != null) {
                plugins.put(descriptor, plugin);
            }
        });
        
        // call restore(...) methods on plugins in rounds, until all return true
        
        int maxRound = 100;
        boolean lastRound = plugins.isEmpty();
        for (int round=0; !lastRound; round++) {
            lastRound = (round == maxRound || plugins.size() == 1);
            Iterator<Map.Entry<ComponentDescriptor,ConsolePlugin>> it = plugins.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry<ComponentDescriptor,ConsolePlugin> e = it.next();
                try {
                    if (e.getValue().restore(e.getKey(), lastRound)) {
                        it.remove();
                    } else if (lastRound) {
                        getLogger().warn("Unable to finish restoring "+ e.getKey().getName() +" configuration.");
                    }
                } catch (RuntimeException x) {
                    getLogger().warn("Failed to restore configuration for "+ e.getKey().getName(), x);
                    it.remove();
                }
            }
        }
        
    }
    
    private void save(Path path) {
        XMLEncoder encoder = null;
        try {
            if (Files.notExists(path)) Files.createDirectories(path.getParent());
            BufferedOutputStream out = new BufferedOutputStream(Files.newOutputStream(path));
            encoder = new XMLEncoder(out);
//            encoder = new XMLEncoder(out, "UTF-8", false, 4);
            
            ComponentDescriptor cd = new ComponentDescriptor();
            cd.setName("");
            cd.setVersion(getApplication().getVersion());
            encoder.writeObject(cd);
            
            Lookup.Result result = getConsoleLookup().lookup(new Lookup.Template(ConsolePlugin.class));
            for (Object o : result.allInstances()) {
                ConsolePlugin plugin = (ConsolePlugin) o;
                ComponentDescriptor data = plugin.save();
                if (data != null) {
                    encoder.writeObject(data);
                }
            }
        } catch (IOException x) {
            getLogger().info("Unable to save configuration.", x);
        } finally {
            if (encoder != null) encoder.close();
        }
    }

    /**
     * Reads component descriptors from the specified XML file.
     * 
     * @param path Path to the file that contains descriptors; if {@code null}, {@link #getDefaultPath} is called to obtain the path.
     * @return Component descriptors mapped by their names.
     */
    public Map<String,ComponentDescriptor> readDescriptors(Path path) {
        if (path == null) path = getDefaultPath();
        Map<String,ComponentDescriptor> out = new HashMap<>();
        try (XMLDecoder decoder = new XMLDecoder( new BufferedInputStream(Files.newInputStream(path)), null, e->{} )) {
//            ComponentDescriptor consoleDescriptor = (ComponentDescriptor) decoder.readObject();
//            out.put("", consoleDescriptor);
            try {
                while (true) {
                    ComponentDescriptor cd = (ComponentDescriptor) decoder.readObject();
                    out.put(cd.getName(), cd);
                }
            } catch (ArrayIndexOutOfBoundsException x) {
            }
        } catch (IOException|ClassCastException|ArrayIndexOutOfBoundsException x) {
        }
        return out;
    }

}
