package org.lsst.ccs.gconsole.plugins.trending;

import java.awt.Color;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Function;
import javax.swing.SwingUtilities;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindow;

/**
 * Data object that can be plotted by {@link TrendPlotter}.
 *
 * @author onoprien
 */
public class Trend {
   
    /**
     * Enumeration of metadata types that might be plotted for a trending channel.
     * On-point metadata is plotted as vertical bars associated with every data point. 
     * Only one on-point metadata type can be displayed at a time.
     * Off-point metadata is plotted as lines. 
     */
    public enum Meta {
        /** Min/max range of data represented by a single point on the plot. On-point. */
        RANGE(null, 
              in -> {
                  int n = in[0].length;
                  double[] errMin = new double[n];
                  double[] errMax = new double[n];
                  for (int i = 0; i < n; i++) {
                      errMin[i] = Math.max(in[0][i] - in[1][i], 0.);
                      errMax[i] = Math.max(in[2][i] - in[0][i], 0.);
                  }
                  return new double[][]{errMin, errMax};
              },
              "min", "max"),
        /** RMS of data represented by a single point on the plot. On-point. */
        RMS(null, 
            in -> new double[][] {in[1], in[1]},
            "rms"), 
        /** Alarm triggering level. Off-point. */
        ALARM(Color.RED, null, "alarmLow", "alarmHigh"),
        /** Warning triggering level. Off-point. */
        WARNING(Color.GREEN, null, "warningLow", "warningHigh");
        
        static private final EnumSet<Meta> onPointSet = EnumSet.of(RANGE, RMS);
        static private final EnumSet<Meta> offPointSet = EnumSet.of(ALARM, WARNING);
        private final Color color;
        private final List<String> keys;
        private final Function<double[][],double[][]> computeBars;
        
        Meta(Color color, Function<double[][],double[][]> computeBars, String... keys) {
            this.color = color;
            this.computeBars = computeBars;
            this.keys = Collections.unmodifiableList(Arrays.asList(keys));
        }
        
        static public EnumSet<Meta> getOnPointSet() {
            return onPointSet;
        }
        static public EnumSet<Meta> getOffPointSet() {
            return offPointSet;
        }
            
        public boolean isOnPoint() {
            return onPointSet.contains(this);
        }
        public Color getColor() {
            return color;
        }
        public int size() {
            return keys.size();
        }
        public List<String> getKeys() {
            return keys;
        }
        /**
         * Computes data for displaying error bars.
         * @param in double[size()+1][n] array. The first index: 0 for value, then for all keys.
         * @return double[2][n] array. The first index is 0 for lower bar.
         */
        public double[][] getBars(double[][] in) {
            if (in == null || computeBars == null) return new double[2][];
            for (double[] data : in) {
                if (data == null) return new double[2][];
            }
            return computeBars.apply(in);
        }
    }

// -- Fields : -----------------------------------------------------------------
    
    private final TrendingChannel channel;
    private TrendPlotter plotter;
    private volatile TrendData data;
    private volatile TimeWindow timeWindow;
    private volatile long timestamp;
    
    private final CopyOnWriteArrayList<Listener> listeners = new CopyOnWriteArrayList<>();

// -- Life cycle : -------------------------------------------------------------
    
    public Trend(TrendingChannel channel) {
        this.channel = channel;
    }

// -- Getters and setters : ----------------------------------------------------

    public TrendingChannel getChannel() {
        return channel;
    }

    public TrendPlotter getPlotter() {
        return plotter;
    }

    void setPlotter(TrendPlotter plotter) {
        this.plotter = plotter;
    }

    public TrendData getData() {
        return data;
    }

    public void setData(TrendData data) {
        this.data = data;
        timestamp = System.currentTimeMillis();
    }

    public TimeWindow getTimeWindow() {
        return timeWindow;
    }

    public void setTimeWindow(TimeWindow timeWindow) {
        this.timeWindow = timeWindow;
    }

    public long getTimestamp() {
        return timestamp;
    }

    public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }
    
    
// -- Handling listeners : -----------------------------------------------------
    
    public void addListener(Listener listener) {
        listeners.add(listener);
    }
    
    public void removeListener(Listener listener) {
        listeners.remove(listener);
    }
    
    public void removeAllListeners() {
        listeners.clear();
    }
    
    public void fireEvent() {
        if (SwingUtilities.isEventDispatchThread()) {
            Event event = new Event(this);
            listeners.forEach(listener -> listener.processEvent(event));
        } else {
            SwingUtilities.invokeLater(this::fireEvent);
        }
    }
    
    public static interface Listener extends EventListener {
        public void processEvent(Event event);
    }
    
    public static class Event extends EventObject {
        Event(Trend source) {
            super(source);
        }
        @Override
        public Trend getSource() {
            return (Trend) source;
        }
    }

}
