package org.lsst.ccs.gconsole.plugins.trending;

import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindow;

/**
 * Source of time history data that can be handled by the trending plugin.
 * The source contains multiple channels identified by their paths.
 * <p>
 * This class accepts registration of listeners that are notified whenever the list
 * of trending channels available from this source changes.
 *
 * @author onoprien
 */
abstract public class TrendingSource {
    
// -- Fields : -----------------------------------------------------------------
    
    private final CopyOnWriteArrayList<Listener> listeners = new CopyOnWriteArrayList<>();
    
    
// -- Retrieving trending information : ----------------------------------------

    /**
     * Returns a list of data channels accessible through this source.
     * 
     * @return The list of channels paths.
     */
    abstract public List<TrendingChannel> getChannels();

    /**
     * Returns the time history data for the specified channel.
     * If the previously known time history is given, it may be updated and returned
     * by this method, or a new instance of {@code TrendData} may be created.
     * 
     * @param channel The channel.
     * @param begin Beginning of the time window.
     * @param end End of the time window.
     * @param metadata Metadata data to be included.
     * @param history Known history for this channel.
     * @return The time history, or {@code null} if this source does not provide data for the specified channel.
     */
    abstract public TrendData get(TrendingChannel channel, long begin, long end, EnumSet<Trend.Meta> metadata, TrendData history);
    
    
// -- Handling listeners : -----------------------------------------------------
    
    public void addListener(Listener listener) {
        listeners.add(listener);
    }
    
    public void removeListener(Listener listener) {
        listeners.remove(listener);
    }
    
    protected void fireEvent(List<TrendingChannel> removedChannels, List<TrendingChannel> addedChannels) {
        Event event = new Event(this, removedChannels, addedChannels);
        listeners.forEach(listener -> listener.processEvent(event));
    }
    
    static public interface Listener extends EventListener {
        void processEvent(Event event);
    }
    
    static public class Event extends EventObject {
        
        private final List<TrendingChannel> removed;
        private final List<TrendingChannel> added;
        
        public Event(TrendingSource source, List<TrendingChannel> removedChannels, List<TrendingChannel> addedChannels) {
            super(source);
            removed = removedChannels == null ? Collections.emptyList() : removedChannels;
            added = addedChannels == null ? Collections.emptyList() : addedChannels;
        }
        
        @Override
        public TrendingSource getSource() {
            return (TrendingSource) super.getSource();
        }
        
        public List<TrendingChannel> getRemovedChannels() {
            return removed == null ? Collections.emptyList() : removed;
        }
        
        public List<TrendingChannel> getAddedChannels() {
            return added == null ? Collections.emptyList() : added;
        }
        
    }
    
}
