package org.lsst.ccs.gconsole.plugins.trending;

import java.awt.Color;
import java.awt.Dimension;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.Properties;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.ButtonModel;
import javax.swing.InputVerifier;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.event.CaretListener;
import org.freehep.application.Application;
import org.freehep.application.PropertyUtilities;
import org.freehep.application.studio.Studio;
import org.freehep.jas.services.PreferencesTopic;

/**
 * Plugin manager preferences panel.
 *
 * @author onoprien
 * @version $Id: $
 */
public class TrendingPreferences implements PreferencesTopic {

// -- Private parts : ----------------------------------------------------------

    private final LsstTrendingPlugin plugin;
    private final Studio app;

    private final String keyPrefix = "org.lsst.trending.";
  
    private final String autoStartKey = "autoStart";
    private final String autoSaveKey = "autoSave";
    private final String useRawDataKey = "rawData";
    private final String nBinsKey = "bins";
    private final String refreshMinKey = "refreshMin";
    private final String refreshMaxKey = "refreshMax";
    private final String warnOnAlarmKey = "warnOnAlarm";
    private final String drawMetaKey = "drawMeta";
    private final String doubleClickKey = "doubleClick";
    private final String dndKey = "dnd";
  
    private final boolean autoStartDefault = false;
    private final boolean autoSaveDefault = false;
    private final boolean useRawDataDefault = false;
    private final int nBinsDefault = 50;
    private final int refreshMinDefault = 10;
    private final int refreshMaxDefault = 3600;
    private final boolean warnOnAlarmDefault = false;
    private final EnumSet<Trend.Meta> drawMetaDefault = EnumSet.noneOf(Trend.Meta.class);
    private final int doubleClickDefault = -1;
    private final int dndDefault = 0;
  
    /** property - start trending on application startup */
    private boolean autoStart;
    /** property - save/restore configuration on startup/shutdown */
    private boolean autoSave;
    /** property - use raw data if true, stat data if false */
    private boolean useRawData;
    /** property - number of bins to fetch from database and plot */
    private int nBins;
    /** property - minimum refresh interval */
    private int refreshMin;
    /** property - maximum refresh interval */
    private int refreshMax;
    /** property - warn if data outside alarm band */
    private boolean warnOnAlarm;
    /** property - metadata to draw */
    private EnumSet<Trend.Meta> drawMeta;
    /** property - double-click action */
    private int doubleClick;
    /** property - drag-and-drop action */
    private int dnd;
    

// -- Construction, reading and saving preferences : ---------------------------

    /** Constructs PluginPreferences object and optionally initializes property values from saved user preferences. */
    public TrendingPreferences(LsstTrendingPlugin plugin) {
        this.plugin = plugin;
        app = (Studio) Application.getApplication();
        restore();
    }

    /** Initializes current settings from saved user preferences. */
    protected final void restore() {
        Properties prop = app.getUserProperties();
        
        autoStart = PropertyUtilities.getBoolean(prop, keyPrefix + autoStartKey, autoStartDefault);
        autoSave = PropertyUtilities.getBoolean(prop, keyPrefix + autoSaveKey, autoSaveDefault);
        useRawData = PropertyUtilities.getBoolean(prop, keyPrefix + useRawDataKey, useRawDataDefault);
        nBins = PropertyUtilities.getInteger(prop, keyPrefix + nBinsKey, nBinsDefault);
        refreshMin = PropertyUtilities.getInteger(prop, keyPrefix + refreshMinKey, refreshMinDefault);
        refreshMax = PropertyUtilities.getInteger(prop, keyPrefix + refreshMaxKey, refreshMaxDefault);
        warnOnAlarm = PropertyUtilities.getBoolean(prop, keyPrefix + warnOnAlarmKey, warnOnAlarmDefault);
        
        String dm = PropertyUtilities.getString(prop, keyPrefix + drawMetaKey, "");
        String[] ss = dm.split("\\+");
        drawMeta = EnumSet.noneOf(Trend.Meta.class);
        for (String s : ss) {
            try {
                drawMeta.add(Trend.Meta.valueOf(s));
            } catch (IllegalArgumentException x) {
            }
        }
        
        doubleClick = PropertyUtilities.getInteger(prop, keyPrefix + doubleClickKey, doubleClickDefault);
        dnd = PropertyUtilities.getInteger(prop, keyPrefix + dndKey, dndDefault);
    }
    
    /** Saves user preferences. */
    protected void save() {
        Properties prop = app.getUserProperties();
        PropertyUtilities.setBoolean(prop, keyPrefix + autoStartKey, autoStart == autoStartDefault ? null : autoStart);
        PropertyUtilities.setBoolean(prop, keyPrefix + autoSaveKey, autoSave == autoSaveDefault ? null : autoSave);
        PropertyUtilities.setBoolean(prop, keyPrefix + useRawDataKey, useRawData == useRawDataDefault ? null : useRawData);
        PropertyUtilities.setInteger(prop, keyPrefix + nBinsKey, nBins == nBinsDefault ? null : nBins);
        PropertyUtilities.setInteger(prop, keyPrefix + refreshMinKey, refreshMin == refreshMinDefault ? null : refreshMin);
        PropertyUtilities.setInteger(prop, keyPrefix + refreshMaxKey, refreshMax == refreshMaxDefault ? null : refreshMax);
        PropertyUtilities.setBoolean(prop, keyPrefix + warnOnAlarmKey, warnOnAlarm == warnOnAlarmDefault ? null : warnOnAlarm);
        
        if (drawMeta.equals(drawMetaDefault)) {
            PropertyUtilities.setString(prop, keyPrefix + drawMeta, null);
        } else {
            StringBuilder sb = new StringBuilder();
            drawMeta.forEach(m -> sb.append(m).append("+"));
            PropertyUtilities.setString(prop, keyPrefix + drawMetaKey, sb.substring(0, sb.length() - 1));
        }
        
        PropertyUtilities.setInteger(prop, keyPrefix + doubleClickKey, doubleClick == doubleClickDefault ? null : doubleClick);
        PropertyUtilities.setInteger(prop, keyPrefix + dndKey, dnd == dndDefault ? null : dnd);
    }

// -- Getters : ----------------------------------------------------------------

    public boolean isAutoStart() {
        return autoStart;
    }

    public boolean isAutoSave() {
        return autoSave;
    }
    
    public boolean isUseRawData() {
        return useRawData;
    }

    public int getNBins() {
        return nBins;
    }
    
    public int getRefreshMin() {
        return refreshMin;
    }

   public int getRefreshMax() {
        return refreshMax;
    }

    public boolean isWarnOnAlarm() {
        return warnOnAlarm;
    }

    public EnumSet<Trend.Meta> getDrawMeta() {
        return drawMeta;
    }
    
    public int getDoubleClick() {
        return doubleClick;
    }
    
    public int getDnd() {
        return dnd;
    }


// -- Setters : ----------------------------------------------------------------

    public void setAutoStart(boolean autoStart, boolean save) {
        this.autoStart = autoStart;
        if (save) {
            PropertyUtilities.setBoolean(app.getUserProperties(), keyPrefix + autoStartKey, autoStart == autoStartDefault ? null : autoStart);
        }
    }

    public void setAutoSave(boolean autoSave, boolean save) {
        this.autoSave = autoSave;
        if (save) {
            PropertyUtilities.setBoolean(app.getUserProperties(), keyPrefix + autoSaveKey, autoSave == autoSaveDefault ? null : autoSave);
        }
    }

    public void setUseRawData(boolean useRawData, boolean save) {
        this.useRawData = useRawData;
        if (save) {
            PropertyUtilities.setBoolean(app.getUserProperties(), keyPrefix + useRawDataKey, useRawData == useRawDataDefault ? null : useRawData);
        }
    }
    
    public void setNBins(int nBins, boolean save) {
        this.nBins = nBins;
        if (save) {
            PropertyUtilities.setInteger(app.getUserProperties(), keyPrefix + nBinsKey, nBins == nBinsDefault ? null : nBins);
        }
    }
    
    public void setRefresh(int refreshMin, int refreshMax, boolean save) {
        this.refreshMin = refreshMin;
        this.refreshMax = refreshMax;
//        AutoRefresh refresher = plugin.getAutoRefresh();
//        if (refresher != null) {
//            refresher.updatePreferences();
//        }
        if (save) {
            PropertyUtilities.setInteger(app.getUserProperties(), keyPrefix + refreshMinKey, refreshMin == refreshMinDefault ? null : refreshMin);
            PropertyUtilities.setInteger(app.getUserProperties(), keyPrefix + refreshMaxKey, refreshMax == refreshMaxDefault ? null : refreshMax);
        }
    }

    public void setWarnOnAlarm(boolean warnOnAlarm, boolean save) {
        this.warnOnAlarm = warnOnAlarm;
        if (save) {
            PropertyUtilities.setBoolean(app.getUserProperties(), keyPrefix + warnOnAlarmKey, warnOnAlarm == warnOnAlarmDefault ? null : warnOnAlarm);
        }
    }

    public void setDrawMeta(EnumSet<Trend.Meta> drawMeta, boolean save) {
        this.drawMeta = drawMeta;
        if (save) {
            if (drawMeta.equals(drawMetaDefault)) {
                PropertyUtilities.setString(app.getUserProperties(), keyPrefix + drawMetaKey, null);
            } else {
                StringBuilder sb = new StringBuilder();
                drawMeta.forEach(m -> sb.append(m).append("+"));
                PropertyUtilities.setString(app.getUserProperties(), keyPrefix + drawMetaKey, sb.substring(0, sb.length() - 1));
            }
        }
    }
    
    public void setDoubleClick(int doubleClick, boolean save) {
        this.doubleClick = doubleClick;
        if (save) {
            PropertyUtilities.setInteger(app.getUserProperties(), keyPrefix + doubleClickKey, doubleClick == doubleClickDefault ? null : doubleClick);
        }
    }
    
    public void setDnd(int dnd, boolean save) {
        this.dnd = dnd;
        if (save) {
            PropertyUtilities.setInteger(app.getUserProperties(), keyPrefix + dndKey, dnd == dndDefault ? null : dnd);
        }
    }
    
    
// -- Implement PreferencesTopic : ---------------------------------------------

    @Override
    public String[] path() {
        return new String[] {"LSST","Trending"};
    }

    @Override
    public JComponent component() {
        return new GUI();
    }
    
    /** Reads GUI and updates current settings. */
    @Override
    public boolean apply(JComponent gui) {
        try {
            boolean out = ((GUI)gui).get();
            if (out) {
                save();
            }
            return out;
        } catch (ClassCastException x) {
            return false;
        }
    }
    

// -- GUI : --------------------------------------------------------------------
    
    
    private class GUI extends JPanel {
    
        private final int HSPACE = 10;
        private final int VSPACE = 5;
        
        JCheckBox _autoStartBox;
        JCheckBox _autoSaveBox;

        JRadioButton _useRawDataRadio;
        JRadioButton _useStatDataRadio;
        ButtonGroup _dataButtonGroup;
        JCheckBox _nBinsBox;
        JTextField _nBinsField;
        
        String[] _doubleClickOptions = new String[] {"Show", "Plot", "New Plot", "New Page"};
        JRadioButton[] _doubleClickRadio;
        ButtonGroup _doubleClickGroup;
        String[] _dndOptions = new String[] {"Replace", "Overlay"};
        JRadioButton[] _dndRadio;
        ButtonGroup _dndGroup;
        JCheckBox _warnOnAlarmBox;
        
        EnumMap<Trend.Meta,JCheckBox> _drawMetaBox = new EnumMap<>(Trend.Meta.class);
        JCheckBox selectedMetaBox;
        
        JRadioButton _refreshFixedRadio;
        JRadioButton _refreshAdaptiveRadio;
        ButtonGroup _refreshGroup;
        JTextField _refreshFixedField;
        JTextField _refreshMinField;
        JTextField _refreshMaxField;
        
        InputVerifier standardVerifier = new InputVerifier() {
            public boolean verify(JComponent input) {
                return input.getForeground().equals(Color.BLACK);
            }
        };

        GUI() {

            setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
            
            // Startup :

            Box panel = Box.createVerticalBox();
            panel.setBorder(BorderFactory.createTitledBorder("Startup"));
            panel.setAlignmentX(LEFT_ALIGNMENT);

            Box row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            row.add(_autoStartBox = new JCheckBox("Start trending on startup."));
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            row.add(_autoSaveBox = new JCheckBox("Restore configuration on startup."));
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            
            add(panel);
            add(Box.createRigidArea(new Dimension(0, VSPACE)));

            // Plotting:

            panel = Box.createVerticalBox();
            panel.setBorder(BorderFactory.createTitledBorder("Plotting"));
            panel.setAlignmentX(LEFT_ALIGNMENT);

            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            row.add(new JLabel("Double-click action: "));
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            
            _doubleClickRadio = new JRadioButton[_doubleClickOptions.length];
            _doubleClickGroup = new ButtonGroup();
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
           for (int i = 0; i<_doubleClickOptions.length; i++) {
                _doubleClickRadio[i] = new JRadioButton(_doubleClickOptions[i]);
                _doubleClickGroup.add(_doubleClickRadio[i]);
                row.add(_doubleClickRadio[i]);
            }
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            
            panel.add(Box.createRigidArea(new Dimension(0,VSPACE)));

            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            row.add(new JLabel("Drag-And-Drop action: "));
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            
            _dndRadio = new JRadioButton[_dndOptions.length];
            _dndGroup = new ButtonGroup();
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
           for (int i = 0; i<_dndOptions.length; i++) {
                _dndRadio[i] = new JRadioButton(_dndOptions[i]);
                _dndGroup.add(_dndRadio[i]);
                row.add(_dndRadio[i]);
            }
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            
            panel.add(Box.createRigidArea(new Dimension(0,VSPACE)));

            Box metaPanel = Box.createVerticalBox();
            metaPanel.setBorder(BorderFactory.createTitledBorder("Show metadata:"));
            metaPanel.setAlignmentX(LEFT_ALIGNMENT);
            ButtonGroup bg = new ButtonGroup();
            bg.add(new JCheckBox());
            for (Trend.Meta m : Trend.Meta.values()) {
                JCheckBox cb = new JCheckBox(m.toString());
                if (m.isOnPoint()) {
                    bg.add(cb);
                    cb.addActionListener(e -> {
                        if (cb == selectedMetaBox) {
                            selectedMetaBox = null;
                            bg.clearSelection();
                        } else {
                            selectedMetaBox = cb;
                        }
                    });
                }
                _drawMetaBox.put(m, cb);
                row = Box.createHorizontalBox();
                row.setAlignmentX(LEFT_ALIGNMENT);
                row.add(cb);
                row.add(Box.createHorizontalGlue());
                metaPanel.add(row);
            }
            panel.add(metaPanel);
            
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            _warnOnAlarmBox = new JCheckBox("Scan for plots with alarms");
            row.add(_warnOnAlarmBox);
            row.add(Box.createHorizontalGlue());
//            panel.add(row);
            
            add(panel);
            add(Box.createRigidArea(new Dimension(0, VSPACE)));

            // Data type and handling:

            panel = Box.createVerticalBox();
            panel.setBorder(BorderFactory.createTitledBorder("Data"));
            panel.setAlignmentX(LEFT_ALIGNMENT);

            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            row.add(new JLabel("Data type: "));
            _dataButtonGroup = new ButtonGroup();
            _useRawDataRadio = new JRadioButton("Raw");
            _dataButtonGroup.add(_useRawDataRadio);
            row.add(_useRawDataRadio);
            _useStatDataRadio = new JRadioButton("Statistical");
            _dataButtonGroup.add(_useStatDataRadio);
            row.add(_useStatDataRadio);
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            _nBinsBox = new JCheckBox("Reduce to");
            _nBinsBox.addActionListener(e -> {
                if (_nBinsBox.isSelected()) {
                    _nBinsField.setEnabled(true);
                    _nBinsField.setText(Integer.toString(nBinsDefault));
                } else {
                    _nBinsField.setEnabled(false);
                    _nBinsField.setText("");
                }
            });
            
            row.add(_nBinsBox);
            _nBinsField = new JTextField(3);
            _nBinsField.setMaximumSize(new Dimension(_nBinsField.getPreferredSize().width, _nBinsField.getPreferredSize().height));
            _nBinsField.addCaretListener(e -> {
                String text = _nBinsField.getText().trim();
                boolean valid = true;
                if (!text.isEmpty()) {
                    try {
                        int value = Integer.parseInt(text);
                        valid = value > 0;
                    } catch (NumberFormatException x) {
                        valid = false;
                    }
                }
                _nBinsField.setForeground(valid ? Color.BLACK : Color.RED);
            });
            _nBinsField.setInputVerifier(standardVerifier);
            row.add(_nBinsField);
            row.add(new JLabel(" bins if possible"));
            row.add(Box.createHorizontalGlue());
            panel.add(row);

            add(panel);
            add(Box.createRigidArea(new Dimension(0, VSPACE)));
            
            // Auto-refresh

            panel = Box.createVerticalBox();
            panel.setBorder(BorderFactory.createTitledBorder("Auto Refresh Interval"));
            panel.setAlignmentX(LEFT_ALIGNMENT);

            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            _refreshGroup = new ButtonGroup();
            _refreshFixedRadio = new JRadioButton("Fixed: ");
            _refreshGroup.add(_refreshFixedRadio);
            _refreshFixedRadio.addActionListener(e -> {
                if (_refreshFixedRadio.isSelected()) {
                    _refreshFixedField.setEnabled(true);
                    _refreshMinField.setEnabled(false);
                    _refreshMaxField.setEnabled(false);
                }
            });
            row.add(_refreshFixedRadio);
            _refreshFixedField = new JTextField(10);
            _refreshFixedField.setMaximumSize(new Dimension(_refreshFixedField.getPreferredSize().width, _refreshFixedField.getPreferredSize().height));
            _refreshFixedField.addCaretListener(e -> {
                String text = _refreshFixedField.getText().trim();
                boolean valid;
                try {
                    int value = Integer.parseInt(text);
                    valid = value > 0;
                } catch (NumberFormatException x) {
                    valid = false;
                }
                _refreshFixedField.setForeground(valid ? Color.BLACK : Color.RED);
            });
            _refreshFixedField.setInputVerifier(standardVerifier);
            row.add(_refreshFixedField);
            row.add(new JLabel(" seconds"));
            row.add(Box.createHorizontalGlue());
            panel.add(row);
 
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            _refreshAdaptiveRadio = new JRadioButton("Adaptive: ");
            _refreshGroup.add(_refreshAdaptiveRadio);
            _refreshAdaptiveRadio.addActionListener(e -> {
                if (_refreshAdaptiveRadio.isSelected()) {
                    _refreshFixedField.setEnabled(false);
                    _refreshMinField.setEnabled(true);
                    _refreshMaxField.setEnabled(true);
                }
            });
            row.add(_refreshAdaptiveRadio);
            row.add(Box.createHorizontalGlue());
            panel.add(row);
 
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            row.add(new JLabel("     Minimum "));
            _refreshMinField = new JTextField(6);
            _refreshMinField.setMaximumSize(new Dimension(_refreshMinField.getPreferredSize().width, _refreshMinField.getPreferredSize().height));
            row.add(_refreshMinField);
            row.add(new JLabel(" seconds"));
            row.add(Box.createHorizontalGlue());
            panel.add(row);
 
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            row.add(new JLabel("     Maximum "));
            _refreshMaxField = new JTextField(6);
            _refreshMaxField.setMaximumSize(new Dimension(_refreshMaxField.getPreferredSize().width, _refreshMaxField.getPreferredSize().height));
            row.add(_refreshMaxField);
            row.add(new JLabel(" seconds"));
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            
            CaretListener minMaxListener = e -> {
                int min = getInt(_refreshMinField);
                if (min < 0) {
                    _refreshMinField.setForeground(min == -2 ? Color.RED : Color.BLACK);
                    int max = getInt(_refreshMaxField);
                    _refreshMaxField.setForeground(max == -2 ? Color.RED : Color.BLACK);
                } else {
                    int max = getInt(_refreshMaxField);
                    if (max < 0) {
                        _refreshMinField.setForeground(Color.BLACK);
                        _refreshMaxField.setForeground(max == -2 ? Color.RED : Color.BLACK);
                    } else {
                        Color color = min < max ? Color.BLACK : Color.RED;
                        _refreshMinField.setForeground(color);
                        _refreshMaxField.setForeground(color);
                    }
                }
            };
            _refreshMinField.addCaretListener(minMaxListener);
            _refreshMaxField.addCaretListener(minMaxListener);
 
            add(panel);
            add(Box.createVerticalGlue());

            set();
        }
        
        final boolean get() {
            
            boolean autoStart = _autoStartBox.isSelected();
            boolean autoSave = _autoSaveBox.isSelected();
            boolean useRawData = _useRawDataRadio.isSelected();
            int nBins = 0;
            if (_nBinsBox.isSelected()) {
                try {
                    nBins = Integer.parseInt(_nBinsField.getText());
                    if (nBins < 1) {
                        return false;
                    }
                } catch (NumberFormatException x) {
                    return false;
                }
            }
            
            int doubleClick = -1;
            ButtonModel bm = _doubleClickGroup.getSelection();
            for (int i = 0; i<_doubleClickRadio.length; i++) {
                if (_doubleClickRadio[i].getModel().equals(bm)) {
                    doubleClick = -1 - i;
                }
            }
            int dnd = 0;
            bm = _dndGroup.getSelection();
            for (int i = 0; i<_dndRadio.length; i++) {
                if (_dndRadio[i].getModel().equals(bm)) {
                    dnd = i;
                }
            }
            
            boolean warnOnAlarm = _warnOnAlarmBox.isSelected();
            
            EnumSet<Trend.Meta> drawMeta = EnumSet.noneOf(Trend.Meta.class);
            for (Trend.Meta m : Trend.Meta.values()) {
                if (_drawMetaBox.get(m).isSelected()) {
                    drawMeta.add(m);
                }
            }
            
            int refreshMin;
            int refreshMax;
            if (_refreshFixedRadio.isSelected()) {
                try {
                    refreshMin = Integer.parseInt(_refreshFixedField.getText());
                    if (refreshMin < 1) {
                        return false;
                    }
                    refreshMax = refreshMin;
                } catch (NumberFormatException x) {
                    return false;
                }
            } else {
                refreshMin = getInt(_refreshMinField);
                if (refreshMin == -2) {
                    return false;
                } else if (refreshMin == -1) {
                    refreshMin = 0;
                }
                refreshMax = getInt(_refreshMaxField);
                if (refreshMax == -2) {
                    return false;
                } else if (refreshMax == -1) {
                    refreshMax = Integer.MAX_VALUE;
                }
                if (refreshMin >= refreshMax) {
                    return false;
                }
            }
            
            setAutoStart(autoStart, false);
            setAutoSave(autoSave, false);
            setDoubleClick(doubleClick, false);
            setDnd(dnd, false);
            setUseRawData(useRawData, false);
            setNBins(nBins, false);
            setRefresh(refreshMin, refreshMax, false);
            setWarnOnAlarm(warnOnAlarm, false);
            setDrawMeta(drawMeta, false);
            return true;
        }

        final void set() {
            
            _autoStartBox.setSelected(autoStart);
            _autoSaveBox.setSelected(autoSave);
            _useRawDataRadio.setSelected(useRawData);
            _useStatDataRadio.setSelected(!useRawData);
            _nBinsBox.setSelected(nBins > 0);
            if (nBins > 0) {
                _nBinsField.setText(Integer.toString(nBins));
            }
            
            try {
                _doubleClickRadio[-doubleClick-1].setSelected(true);
            } catch (IndexOutOfBoundsException x) {
                _doubleClickRadio[0].setSelected(true);
            }
            try {
                _dndRadio[dnd].setSelected(true);
            } catch (IndexOutOfBoundsException x) {
                _dndRadio[0].setSelected(true);
            }
            
            _warnOnAlarmBox.setSelected(warnOnAlarm);
            
            for (Trend.Meta m : Trend.Meta.values()) {
                if (drawMeta.contains(m)) {
                    JCheckBox cb = _drawMetaBox.get(m);
                    cb.setSelected(true);
                    selectedMetaBox = cb;
                }
            }
            
            boolean fixedRefreshRate = refreshMin == refreshMax;
            if (fixedRefreshRate) {
                _refreshFixedRadio.setSelected(true);
                _refreshFixedField.setText(Integer.toString(refreshMin));
                _refreshFixedField.setEnabled(true);
                _refreshMinField.setEnabled(false);
                _refreshMaxField.setEnabled(false);
            } else {
                _refreshAdaptiveRadio.setSelected(true);
                _refreshMinField.setText(refreshMin == 0 ? "" : Integer.toString(refreshMin));
                _refreshMaxField.setText(refreshMax == Integer.MAX_VALUE ? "" : Integer.toString(refreshMax));
                _refreshFixedField.setEnabled(false);
                _refreshMinField.setEnabled(true);
                _refreshMaxField.setEnabled(true);
            }
            
        }
        
        
        private int getInt(JTextField field) {
            String text = field.getText().trim();
            if (text.isEmpty()) {
                return -1;
            } else {
                try {
                    int out = Integer.parseInt(text);
                    return out < 0 ? -2 : out;
                } catch (NumberFormatException x) {
                    return -2;
                }
            }
        }
    }

}
