package org.lsst.ccs.gconsole.plugins.trending;

import jas.hist.DataSource;
import jas.hist.HasStyle;
import jas.hist.HistogramUpdate;
import jas.hist.JASHist1DHistogramStyle;
import jas.hist.JASHistStyle;
import jas.hist.XYDataSource;
import java.awt.Color;
import java.awt.Component;
import java.awt.Point;
import java.io.Serializable;
import java.util.*;
import javax.swing.ButtonGroup;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JSeparator;
import org.freehep.jas.plugin.plotter.DefaultPlotter;
import org.freehep.jas.plugin.plotter.DefaultRegion;
import org.freehep.jas.plugin.plotter.JAS3DataSource;
import org.freehep.jas.plugin.tree.FTreePath;
import org.freehep.jas.services.PlotPage;
import org.freehep.jas.services.PlotRegion;
import org.freehep.jas.services.Plotter;
import org.freehep.swing.popup.HasPopupItems;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindow;

/**
 * Plotter for {@link Trend} objects.
 *
 * @author onoprien
 */
public class TrendPlotter implements Plotter {
    
    /** Option that control plotting methods. */
    enum Option {
        /** Replace an existing plot for the same channel if possible. */ EXIST,
        /** Plot in a new page unless an existing plot is being replaced. */ NEWPAGE,
        /** Plot in a new region on the current page unless an existing plot is being replaced. */ NEWPLOT,
        /** Overlay on top of the current region */ OVERLAY,
        /** Fetch data at low priority. */ SLOW,
        /** Regions are redrawn whether or not any datasets they contain have been refreshed. */ REPLOT
    }

// -- Fields : -----------------------------------------------------------------
    
    private static final Color[] colors = {Color.BLUE, Color.RED, Color.GREEN, Color.MAGENTA, Color.ORANGE, Color.CYAN, Color.DARK_GRAY, Color.LIGHT_GRAY};

    private final LsstTrendingPlugin plugin;
    private final Plotter plotter;
    private final ArrayList<Trend> trends = new ArrayList<>(1);
    private final ArrayList<Jas3Source> data = new ArrayList<>(1);
    
    private PlotRegion region;
    
    private Trend.Meta metaOnPoint;
    private EnumSet<Trend.Meta> metaOffPoint;
    private long[] rangeT;
    private double[] rangeY;


// -- Life cycle : -------------------------------------------------------------
    
    public TrendPlotter(LsstTrendingPlugin plugin) {
        this.plugin = plugin;
        plotter = plugin.getPlotFactory().createPlotterFor(JAS3DataSource.class);
        try {
            DefaultPlotter dp = (DefaultPlotter) plotter;
            dp.getPlot().getXAxis().setRangeAutomatic(true);
            dp.getPlot().setShowStatistics(false);
        } catch (ClassCastException x) {
        }
        reset();
    }

    
// -- Getters and setters : ----------------------------------------------------

    public PlotRegion getRegion() {
        return region;
    }
    
    public void setRegion(PlotRegion region) {
        this.region = region;
        try {
            DefaultRegion dr = (DefaultRegion) region;
            dr.addPopupItems(new Popup());
        } catch (ClassCastException x) {
        }
    }
    
    public EnumSet<Trend.Meta> getMeta() {
        EnumSet<Trend.Meta> out = metaOnPoint == null ? EnumSet.noneOf(Trend.Meta.class) : EnumSet.of(metaOnPoint);
        out.addAll(metaOffPoint);
        return out;
    }
    
    public void setMeta(EnumSet<Trend.Meta> extras) {
        
    }
    
    /**
     * Returns {@code true} if this plotter is currently managed by the graphical console.
     * The plotter's viewable component does not have to be visible for the plotter to be active.
     * 
     * @return {@code True} if this plotter is active.
     */
    public boolean isActive() {
        if (region != null) {
            try {
                return ((JComponent)viewable()).getTopLevelAncestor() != null;
            } catch (ClassCastException|NullPointerException x) {
            }
        }
        return false;
    }
    
    public List<Trend> getTrends() {
        return Collections.unmodifiableList(trends);
    }
    
    public boolean isEmpty() {
        return trends.isEmpty();
    }


// -- Implementing Plotter : ---------------------------------------------------
    
    @Override
    public void plot(Object timeHistory, int mode) {
        plot(timeHistory, mode, null, "");
    }

    @Override
    public void plot(Object timeHistory, int mode, Object style, String options) {
        
        if (!(timeHistory instanceof Trend && (mode == Plotter.NORMAL || mode == Plotter.OVERLAY))) return;
        Trend trend = (Trend) timeHistory;
        trend.setPlotter(this);
        
        if (!trends.isEmpty()) {
            if (mode == Plotter.NORMAL) {
                clear();
            } else if (trends.size() < data.size()) {
                plotter.clear();
                Jas3Source other = data.get(0);
                data.clear();
                data.add(other);
                plotter.plot(other, Plotter.NORMAL);
            }
        }
        trends.add(trend);
        
        Jas3Source ds = new Jas3Source(trend, metaOnPoint, null);
        trend.addListener(ds);
        data.add(ds);
        plotter.plot(ds, mode);
        
        if (!metaOffPoint.isEmpty() && trends.size() == 1 && mode == Plotter.NORMAL) {
            for (Trend.Meta meta : metaOffPoint) {
                for (String key : meta.getKeys()) {
                    ds = new Jas3Source(trend, meta, key);
                    trend.addListener(ds);
                    data.add(ds);
                    plotter.plot(ds, Plotter.OVERLAY);
                }
            }
        }
        
        try {
            DefaultPlotter dp = (DefaultPlotter) plotter;
            dp.getPlot().getXAxis().setRangeAutomatic(true);
            if (trends.size() == 1) {
                dp.getPlot().setTitle(trends.get(0).getChannel().getTitle());
            }
        } catch (ClassCastException x) {
        }
        
    }

    @Override
    public void remove(Object data) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public void clear() {
        if (!trends.isEmpty()) {
            clean();
            reset();
        }
    }

    @Override
    public Component viewable() {
        return plotter.viewable();
    }

    @Override
    public List<Object> getData() {
        return Collections.unmodifiableList(trends);
    }
    
    
// -- Additional operations : --------------------------------------------------
    
    public void toFront() {
        if (region != null) {
            PlotPage page = region.getPage();
            if (page != null) {
                page.setCurrentRegion(region);
                page.showPage();
            }
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    /**
     * Plot the specified trends, without resetting the plotter.
     */
    private void plot(List<Trend> trendList) {
        if (trendList.size() == 1) {
            plot(trendList.get(0), Plotter.NORMAL);
        } else {
            trendList.forEach(trend -> plot(trend, Plotter.OVERLAY));
        }
    }
    
    /**
     * Re-plot trends currently displayed by this plotter.
     */
    private void plot() {
        if (trends.size() > 1) {
            trends.forEach(trend -> trend.fireEvent());
        } else if (trends.size() == 1) {
            Trend trend = trends.get(0);
            clean();
            plot(trend, Plotter.NORMAL);
        }
    }
    
    private void clean() {
        trends.forEach(trend -> trend.removeAllListeners());
        trends.clear();
        plotter.clear();
        data.clear();
    }
    
    private void reset() {
        metaOnPoint = null;
        metaOffPoint = EnumSet.noneOf(Trend.Meta.class);
        plugin.getPreferences().getDrawMeta().forEach(m -> {
            if (m.isOnPoint()) {
                metaOnPoint = m;
            } else {
                metaOffPoint.add(m);
            }
        });
    }
    
// -- Jas3 adapter class : -----------------------------------------------------
    
    private class Jas3Source extends Observable implements JAS3DataSource, XYDataSource, HasStyle, Trend.Listener {
        
        private final String title;
        private final String key;
        private long[] t;
        private double[] y, plusError, minusError;
        private final JASHist1DHistogramStyle style = new JASHist1DHistogramStyle();
        
        Jas3Source(Trend trend, Trend.Meta meta, String key) {
            this.key = key;
            Color color;
            if (key == null) {
                title = trend.getChannel().getTitle();
                color = colors[trends.indexOf(trend)%(colors.length)];
                style.setShowDataPoints(true);
                style.setShowErrorBars(true);
            } else {
                title = meta.toString();
                color = meta.getColor();
                style.setShowDataPoints(false);
                style.setShowErrorBars(false);
            }
            style.setDataPointColor(color);
            style.setErrorBarColor(color);
            style.setLineColor(color);
            style.setDataPointStyle(JASHist1DHistogramStyle.SYMBOL_DOT);
            style.setShowHistogramBars(false);
            style.setShowLinesBetweenPoints(true);
            populate(trend);
        }
        
        private void populate(Trend trend) {
            TrendData td = trend.getData();
            if (td == null) {
                t = null;
                y = plusError = minusError = null;
                return;
            }
            if (key == null) {
                t = td.getTime();
                y = td.getValue();
                if (metaOnPoint == null) {
                    plusError = null;
                    minusError = null;
                } else {
                    List<String> keys = metaOnPoint.getKeys();
                    int n = keys.size();
                    double[][] in = new double[n+1][];
                    in[0] = y;
                    for (int i=0; i<n; i++) {
                        in[i+1] = td.getValue(keys.get(i));
                    }
                    double[][] out = metaOnPoint.getBars(in);
                    minusError = out[0];
                    plusError = out[1];
                }
            } else {
                t = td.getTime(key);
                y = td.getValue(key);
            }
        }
        
        // Implement JAS3DataSource :

        @Override
        public void destroy() {
        }

        @Override
        public void modifyPopupMenu(JPopupMenu jPopupMenu, Component component) {
        }

        @Override
        public DataSource dataSource() {
            return this;
        }

        @Override
        public FTreePath path() {
            return new FTreePath(getTitle());
        }

        @Override
        public String[] axisLabels() {
            return null;
        }

        @Override
        public void setAxisType(int type) {
        }
        
        // Implement XYDataSource :

        @Override
        public int getNPoints() {
            return t == null ? 0 : t.length;
        }

        @Override
        public double getX(int i) {
            return t[i]/1000.;
        }

        @Override
        public double getY(int i) {
            return y[i];
        }

        @Override
        public double getPlusError(int i) {
            return plusError == null ? 0. : plusError[i];
        }

        @Override
        public double getMinusError(int i) {
            return minusError == null ? 0. : minusError[i];
        }

        @Override
        public int getAxisType() {
            return DataSource.DATE;
        }

        @Override
        public String getTitle() {
            return title;
        }
        
        // Implement HasStyle :

        @Override
        public JASHistStyle getStyle() {
            return style;
        }
        
        
        // Implement Trend.Listener :

        @Override
        public void processEvent(Trend.Event event) {
 
            if (rangeT == null) {
                try {
                    DefaultPlotter dp = (DefaultPlotter) plotter;
                    dp.getPlot().getXAxis().setRangeAutomatic(true);
//                    if (trends.size() == 1) {
//                        dp.getPlot().setTitle(trends.get(0).getChannel().getTitle());
//                    }
                } catch (ClassCastException x) {
                }
            }
         
           Trend trend = event.getSource();
            if (trend != null) {
                populate(trend);
                setChanged();
                notifyObservers(new HistogramUpdate(HistogramUpdate.DATA_UPDATE, false));
            }
        }
    }
    
    
// -- Popup menu handler class : -----------------------------------------------
    
    private class Popup implements HasPopupItems {

        @Override
        public JPopupMenu modifyPopupMenu(JPopupMenu menu, Component componentt, Point point) {
            menu.insert(new JSeparator(), 0);
            JMenu metaMenu = new JMenu("Show metadata");
            ButtonGroup bg = new ButtonGroup();
            Trend.Meta.getOnPointSet().forEach(meta -> {
                JRadioButtonMenuItem item = new JRadioButtonMenuItem(meta.toString(), metaOnPoint == meta);
                bg.add(item);
                item.addActionListener(e -> {
                    Trend.Meta selected = Trend.Meta.valueOf(e.getActionCommand());
                    if (selected != metaOnPoint) {
                        metaOnPoint = selected;
                    } else {
                        metaOnPoint = null;
                    }
                    trends.forEach(trend -> trend.fireEvent());
                });
                metaMenu.add(item);
            });
            metaMenu.addSeparator();
            Trend.Meta.getOffPointSet().forEach(meta -> {
                JCheckBoxMenuItem item = new JCheckBoxMenuItem(meta.toString(), metaOffPoint.contains(meta));
                item.setEnabled(trends.size() < 2);
                item.addActionListener(e -> {
                    Trend.Meta m = Trend.Meta.valueOf(e.getActionCommand());
                    boolean wasSelected = metaOffPoint.contains(m);
                    JCheckBoxMenuItem mi = (JCheckBoxMenuItem) e.getSource();
                    boolean isSelected = mi.isSelected();
                    if (wasSelected != isSelected) {
                        if (wasSelected) {
                            metaOffPoint.remove(m);
                        } else {
                            metaOffPoint.add(m);
                        }
                        plot();
                    }
                });
                metaMenu.add(item);
            });
            menu.insert(metaMenu, 0);
            return menu;
        }

    }
    
    
// -- Style class : ------------------------------------------------------------
    
    public static class Style {
        
        Trend.Meta onPoint;
        EnumSet<Trend.Meta> offPoint;
        
        public Style() {
        }
        
        public Style(Trend.Meta onPoint, EnumSet<Trend.Meta> offPoint) {
            this.onPoint = onPoint;
            this.offPoint = offPoint;
        }

        public Trend.Meta getOnPoint() {
            return onPoint;
        }

        public void setOnPoint(Trend.Meta onnPoint) {
            this.onPoint = onnPoint;
        }

        public EnumSet<Trend.Meta> getOffPoint() {
            return offPoint;
        }

        public void setOffPoint(EnumSet<Trend.Meta> offPoint) {
            this.offPoint = offPoint;
        }

    }
    
    
// -- Saving/Restoring : -------------------------------------------------------
    
    Descriptor save() {
        Descriptor desc = new Descriptor();
        
        if (!trends.isEmpty()) {
            int n = trends.size();
            TrendDescriptor[] tds = new TrendDescriptor[n];
            for (int i = 0; i < n; i++) {
                Trend trend = trends.get(i);
                TrendDescriptor td = new TrendDescriptor();
                String path = trend.getChannel().getPath();
                td.setPath(path);
                String title = trend.getChannel().getTitle();
                if (!path.equals(title)) {
                    td.setTitle(title);
                }
                td.setTimeWindow(trend.getTimeWindow().toCompressedString());
                tds[i] = td;
            }
            desc.setTrends(tds);
        }
        
        if (metaOnPoint != null) {
            desc.setOnPointMeta(metaOnPoint.name());
        }
        if (!metaOffPoint.isEmpty()) {
            StringBuilder sb = new StringBuilder();
            metaOffPoint.forEach(m -> sb.append(m.name()).append("+"));
            desc.setOffPointMeta(sb.substring(0, sb.length()-1));
        }
        
        if (rangeT != null) {
            desc.setRangeT(Arrays.copyOf(rangeT, rangeT.length));
        }
        if (rangeY != null) {
            desc.setRangeY(Arrays.copyOf(rangeY, rangeY.length));
        }
                
        return desc;
    }
    
    void restore(Descriptor desc) {
        
        rangeT = desc.getRangeT();
        rangeY = desc.getRangeY();
        
        String ms = desc.getOnPointMeta();
        if (ms != null) metaOnPoint = Trend.Meta.valueOf(ms);
        ms = desc.getOffPointMeta();
        metaOffPoint = EnumSet.noneOf(Trend.Meta.class);
        if (ms != null) {
            String[] ss = ms.split("\\+");
            for (String s : ss) {
                metaOffPoint.add(Trend.Meta.valueOf(s));
            }
        }
        
        TrendDescriptor[] tds = desc.getTrends();
        if (tds != null) {
            ArrayList<Trend> trendList = new ArrayList<>(tds.length);
            ArrayList<TrendingChannel> knownChannels = plugin.getSourcesManager().getChannels();
            for (TrendDescriptor td : tds) {
                String path = td.getPath();
                String title = td.getTitle();
                TrendingChannel channel = null;
                for (TrendingChannel ch : knownChannels) {
                    if (title.equals(ch.getTitle())) {
                        if (path.equals(ch.getPath())) {
                            channel = ch;
                            break;
                        } else if (channel == null) {
                            channel = ch;
                        }
                    }
                }
                if (channel != null) {
                    Trend trend = new Trend(channel);
                    TimeWindow timeWindow = plugin.getTimeWindowSelector().getTimeWindow(td.getTimeWindow());
                    trend.setTimeWindow(timeWindow);
                    trendList.add(trend);
                }
            }
            if (!trendList.isEmpty()) {
                plot(trendList);
                plugin.refresh(trends);
                toFront();
            }
        }
    }
    
    public static class Descriptor implements Serializable {

        private TrendDescriptor[] trends;
        private String onPointMeta;
        private String offPointMeta;
        private long[] rangeT;
        private double[] rangeY;
        
        public double[] getRangeY() {
            return rangeY;
        }

        public void setRangeY(double[] rangeY) {
            this.rangeY = rangeY;
        }

        public long[] getRangeT() {
            return rangeT;
        }

        public void setRangeT(long[] rangeT) {
            this.rangeT = rangeT;
        }

        public String getOffPointMeta() {
            return offPointMeta;
        }

        public void setOffPointMeta(String offPointMeta) {
            this.offPointMeta = offPointMeta;
        }

        public String getOnPointMeta() {
            return onPointMeta;
        }

        public void setOnPointMeta(String onPointMeta) {
            this.onPointMeta = onPointMeta;
        }

        public TrendDescriptor[] getTrends() {
            return trends;
        }

        public void setTrends(TrendDescriptor[] trends) {
            this.trends = trends;
        }
        
    }
    
    public static class TrendDescriptor implements Serializable {
        
        private String path;
        private String title;
        private String timeWindow;

        public String getTimeWindow() {
            return timeWindow;
        }

        public void setTimeWindow(String timeWindow) {
            this.timeWindow = timeWindow;
        }

        public String getTitle() {
            return title == null ? path : title;
        }

        public void setTitle(String title) {
            this.title = title;
        }

        public String getPath() {
            return path;
        }

        public void setPath(String path) {
            this.path = path;
        }
        
    }
    
}
