package org.lsst.ccs.gconsole.plugins.alert;

import java.awt.event.ActionEvent;
import java.util.Map;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.StatusClearedAlert;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.gconsole.ConsolePlugin;
import org.lsst.ccs.gconsole.Plugin;
import org.lsst.ccs.gconsole.plugins.tracer.LsstTracerPlugin;
import org.lsst.ccs.gconsole.plugins.tracer.Tracer;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.CommandOriginator;

/**
 * Graphical console plugin that handles alerts received from CCS subsystems.
 *
 * @author onoprien
 */
@Plugin(name="LSST Alert Notification Plugin",
        description="LSST CCS alert notification service.")
public class LsstAlertPlugin extends ConsolePlugin {
    
// -- Fields : -----------------------------------------------------------------
    
    static private final String OPT_AUTO_START_VIEWER = "autoStartViewer";
    static private final String OPT_AUTO_START_TRACER = "autoStartTracer";
    static private final String OPT_SHOW_TESTER = "showTester";
    static final String OPT_HISTORY = "history";
    static final String OPT_MUTE = "mute";
    static final String OPT_TOFRONT = "toFront";
    
    private AlertHandler alertHandler;
    
    private AlertViewer viewer;
    private Action viewerAction;
    
    private JFrame emitter;

    
// -- Lifecycle : --------------------------------------------------------------

    @Override
    synchronized public void initialize() {
        
        getServices().addProperty(OPT_AUTO_START_VIEWER, false);
        getServices().addPreference(new String[] {"LSST","Alerts"}, "Startup", "${"+ OPT_AUTO_START_VIEWER +"} Start viewer on console sturtup.");
        getServices().addProperty(OPT_AUTO_START_TRACER, false);
        getServices().addPreference(new String[] {"LSST","Alerts"}, "Startup", "${"+ OPT_AUTO_START_TRACER +"} Start tracer on console sturtup.");
        
        getServices().addProperty(OPT_SHOW_TESTER, false);
        getServices().addPreference(new String[] {"LSST","Alerts"}, null, "${"+ OPT_SHOW_TESTER +"} Show tester controls.");
        
        getServices().addProperty(OPT_HISTORY, false);
        getServices().addProperty(OPT_MUTE, false);
        getServices().addProperty(OPT_TOFRONT, true);
        
        viewerAction = new AbstractAction("Viewer") {
            @Override
            public void actionPerformed(ActionEvent e) {
                onNewAlertViewer();
            }
        };
        getServices().addMenu(viewerAction, "400: CCS Tools :-1:4", "Alerts:1");
        
        Action act  = new AbstractAction("Tracer") {
            @Override
            public void actionPerformed(ActionEvent e) {
                onNewAlertTracer();
            }
        };
        getServices().addMenu(act, "400: CCS Tools :-1:4", "Alerts:2");
        
        if ((boolean) getServices().getProperty(OPT_SHOW_TESTER)) {
            showTester();
        }
        
        alertHandler = new AlertHandler(this);
        getConsole().getLookup().add(alertHandler);
    }

    @Override
    synchronized public void start() {
        if ((boolean) getServices().getProperty(OPT_AUTO_START_VIEWER)) {
            onNewAlertViewer();
        }
        if ((boolean) getServices().getProperty(OPT_AUTO_START_TRACER)) {
            onNewAlertTracer();
        }
    }
  
    @Override
    synchronized public void shutdown() {
        if (alertHandler != null) {
            getConsole().getLookup().remove(alertHandler);
            alertHandler = null;
        }
    }
    
    
// -- Operations : -------------------------------------------------------------
    
    public void onNewAlertViewer() {
        if (viewer == null) {
            viewer = new AlertViewer(this);
            alertHandler.addListener(viewer);
            viewerAction.setEnabled(false);
        }
    }
    
    public void stopAlertViewer() {
        alertHandler.removeListener(viewer);
        viewer = null;
        viewerAction.setEnabled(true);
    }
    
    public void onNewAlertTracer() {
        LsstTracerPlugin tracerPlugin = (LsstTracerPlugin) getConsole().getLookup().lookup(LsstTracerPlugin.class);
        Tracer tracer = tracerPlugin.createTracer("Alerts", "BuiltIn/Alerts");
        tracer.setFilter(filteredMessage -> {
            BusMessage bm = filteredMessage.getBusMessage();
            if (bm instanceof StatusRaisedAlert) {
                StatusRaisedAlert am = (StatusRaisedAlert) bm;
                Alert alert = am.getRaisedAlert();
                String id = alert.getAlertId();
                StringBuilder sb = new StringBuilder();
                sb.append(AlertViewer.formatTimeStamp(am.getTimeStamp())).append(" : ");
                sb.append(am.getRaisedAlertSummary().getRaisedAlert(id).getLatestAlertState());
                sb.append(" from ").append(am.getOriginAgentInfo().getName()).append(".\n");
                sb.append("ID: ").append(id).append(".  Description:\n");
                sb.append(alert.getDescription()).append("\n");
                filteredMessage.setMessage(sb.toString());
                return filteredMessage;
            } else if (bm instanceof StatusClearedAlert) {
                StatusClearedAlert am = (StatusClearedAlert) bm;
                StringBuilder sb = new StringBuilder();
                sb.append(AlertViewer.formatTimeStamp(am.getTimeStamp())).append(" : ");
                sb.append(am.getOriginAgentInfo().getName()).append(" cleared alerts ");
                for (String id : am.getClearAlertIds()) {
                    sb.append(id).append(", ");
                }
                sb.delete(sb.length()-2, sb.length()).append(".\n");
                filteredMessage.setMessage(sb.toString());
                return filteredMessage;
            } else {
                return null;
            }
        });
    }
    
    public void clearAlerts(Map<String,String[]> alarms) {
        AgentMessagingLayer messenger = getConsole().getMessagingAccess();
        CommandOriginator originator = new CommandOriginator() {
            public void processAck(CommandAck ack) {}
            public void processResult(CommandResult result) {}
            public void processNack(CommandNack nack) {}
        };
        alarms.forEach((source, ids) -> {
            if (source.startsWith("Local")) {
                for (String id : ids) {
                    alertHandler.submitAlert(source, new Alert(id, ""), AlertState.NOMINAL);
                }
            } else {
                CommandRequest request;
                if (ids == null) {
                    request = new CommandRequest(source, "clearAllAlerts");
                } else {
                    request = new CommandRequest(source, "clearAlerts", (Object) ids);
                }
                messenger.sendCommandRequest(request, originator);
            }
        });
    }
    
    
// -- Testing : ----------------------------------------------------------------

    @Override
    public void propertiesChanged(Object source, Map<String, Object> changes) {
        Boolean tester = (Boolean) changes.get(OPT_SHOW_TESTER);
        if (tester != null && tester) {
            showTester();
        }
    }
    
    private void showTester() {
        Action act = new AbstractAction("Tester") {
            @Override
            public void actionPerformed(ActionEvent ee) {
                setEnabled(false);
                SwingUtilities.invokeLater(() -> {
                    Box treeButtonsPanel = Box.createHorizontalBox();

                    JButton emitAlert = new JButton("Bus Alarm");
                    emitAlert.addActionListener(e -> {
                        String command = JOptionPane.showInputDialog(getConsole().getWindow(), "Enter ID:A|W|N:description", "ID:A:description");
                        String[] tokens = command.split(":");
                        AlertState level = tokens[1].equalsIgnoreCase("A") ? AlertState.ALARM : (tokens[1].equalsIgnoreCase("W") ? AlertState.WARNING : AlertState.NOMINAL);
                        if (level == AlertState.NOMINAL) {
                            if (tokens[0].isEmpty()) {
                                getConsole().clearAllAlerts();
                            } else {
                                getConsole().clearAlerts(tokens[0]);
                            }
                        } else {
                            getConsole().raiseAlert(new Alert(tokens[0], tokens[2]), level);
                        }
                    });
                    treeButtonsPanel.add(emitAlert);

                    emitAlert = new JButton("Local Alarm");
                    emitAlert.addActionListener(e -> {
                        String command = JOptionPane.showInputDialog(emitter, "Enter Source:ID:A|W|N:description", "Source:ID:W:description");
                        String[] tokens = command.split(":");
                        AlertState level = tokens[2].equalsIgnoreCase("A") ? AlertState.ALARM : (tokens[2].equalsIgnoreCase("W") ? AlertState.WARNING : AlertState.NOMINAL);
                        alertHandler.submitAlert(tokens[0], new Alert(tokens[1], tokens[3]), level);
                    });
                    treeButtonsPanel.add(emitAlert);

                    emitter = new JFrame("Alert emitter");
                    emitter.add(treeButtonsPanel);
                    emitter.pack();
                    emitter.setVisible(true);
                });
            }
        };
        getServices().addMenu(act, "400: CCS Tools :-1:4", "Alerts:1:1");
    }
    
}
