package org.lsst.ccs.gconsole.plugins.alert;

import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import org.lsst.ccs.RaisedAlertBookkeeper;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.StatusAlert;
import org.lsst.ccs.bus.messages.StatusClearedAlert;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.CommandOriginator;
import org.lsst.ccs.messaging.StatusMessageListener;

/**
 * Alert handling service implementation.
 *
 * @author onoprien
 */
public class AlertHandler implements AlertService, StatusMessageListener, CommandOriginator {
    
// -- Fields : -----------------------------------------------------------------
    
    private final LsstAlertPlugin plugin;
    private final CopyOnWriteArrayList<AlertListener> listeners = new CopyOnWriteArrayList<>();
    
    private final HashMap<String,Instant> source2time = new HashMap<>();
    private final ConcurrentHashMap<String,RaisedAlertSummary> source2summary = new ConcurrentHashMap<>();
    
// -- Lifecycle : --------------------------------------------------------------
    
    public AlertHandler(LsstAlertPlugin plugin) {
        this.plugin = plugin;
    }


// -- Processing alerts : ------------------------------------------------------

    @Override
    synchronized public void submitAlert(String source, Alert alert, AlertState level) {
        if (source == null || source.isEmpty()) {
            source = "Local";
        } else if (!source.startsWith("Local")) {
            source = "Local : "+ source;
        }
        RaisedAlertSummary oldSummary = source2summary.get(source);
        if (oldSummary == null) {
            if (level == AlertState.NOMINAL) return;
            oldSummary = new RaisedAlertBookkeeper();
            source2summary.put(source, oldSummary);
        }
        String id = alert.getAlertId();
        try {
            RaisedAlertBookkeeper keeper = (RaisedAlertBookkeeper) oldSummary;
            if (level == AlertState.NOMINAL) {
                keeper.clearAlert(id);
                String[] ids = new String[] {id};
                fireAlertEvent(new AlertEvent(source, ids, keeper));
            } else {
                keeper.raiseAlert(alert, level, System.currentTimeMillis());
                fireAlertEvent(new AlertEvent(source, alert, keeper));
            }
        } catch (ClassCastException x) {
        }
    }

    @Override
    public void onStatusMessage(StatusMessage msg) {
        if (msg instanceof StatusAlert) {
            String source = msg.getOriginAgentInfo().getName();
            StatusAlert message = (StatusAlert) msg;
            Instant timeStamp = getTimeStamp(message);
            synchronized (this) {
                Instant latest = source2time.merge(source, timeStamp, (s, t) -> timeStamp.isAfter(t) ? timeStamp : t);
                if (latest.equals(timeStamp)) {
                    RaisedAlertSummary summary = message.getRaisedAlertSummary();
                    source2summary.put(source, summary);
                    if (message instanceof StatusRaisedAlert) {
                        Alert alert = ((StatusRaisedAlert)message).getRaisedAlert();
                        fireAlertEvent(new AlertEvent(source, alert, summary));
                    } else if (message instanceof StatusClearedAlert) {
                        String[] ids = ((StatusClearedAlert)message).getClearAlertIds();
                        fireAlertEvent(new AlertEvent(source, ids, summary));
                    }
                }
            }
        }
    }

    @Override
    public void processAck(CommandAck ack) {
    }

    @Override
    public void processNack(CommandNack nack) {
    }

    @Override
    public void processResult(CommandResult result) {
        Object o = result.getResult();
        if (o instanceof RaisedAlertSummary) {
            String source = result.getOriginAgentInfo().getName();
            Instant timeStamp = getTimeStamp(result);
            synchronized (this) {
                Instant latest = source2time.merge(source, timeStamp, (s, t) -> timeStamp.isAfter(t) ? timeStamp : t);
                if (latest.equals(timeStamp)) {
                    RaisedAlertSummary summary = (RaisedAlertSummary) o;
                    source2summary.put(source, summary);
                    fireAlertEvent(new AlertEvent(source, new String[0], summary));
                }
            }
        }
    }


// -- Handling listeners and initialization : ----------------------------------
    
    @Override
    public void addListener(AlertListener listener) {
        synchronized (listeners) {
            boolean ok = listeners.addIfAbsent(listener);
            if (!ok) throw new IllegalArgumentException("The listener is already registered");
            if (listeners.size() == 1) {
                AgentMessagingLayer messenger = plugin.getConsole().getMessagingAccess();
                messenger.addStatusMessageListener(this);
                List<AgentInfo> agents = messenger.getAgentPresenceManager().listConnectedAgents();
                for (AgentInfo agent : agents) {
                    try {
                        CommandRequest req = new CommandRequest(agent.getName(), "getRaisedAlertSummary");
                        messenger.sendCommandRequest(req, this);
                    } catch (Exception x) {
                    }
                }
            }
        }
    }

    @Override
    public void removeListener(AlertListener listener) {
        synchronized (listeners) {
            listeners.remove(listener);
            if (listeners.isEmpty()) {
                plugin.getConsole().getMessagingAccess().removeStatusMessageListener(this);
            }
        }
    }
    
    private void fireAlertEvent(AlertEvent event) {
        listeners.forEach(listener -> listener.onAlert(event));
    }
    
    
// -- Getters : ----------------------------------------------------------------

    @Override
    public RaisedAlertSummary getSummary(String source) {
        return source2summary.get(source);
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    /** FIXME */
    private Instant getTimeStamp(StatusAlert message) {
        return Instant.ofEpochMilli(message.getTimeStamp());
    }
    
    /** FIXME */
    private Instant getTimeStamp(CommandResult result) {
        return Instant.ofEpochMilli(result.getTimeStamp());
    }
    
}
