package org.lsst.ccs.gconsole.plugins.trending.timeselection;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.ListCellRenderer;
import org.freehep.application.studio.Studio;

/**
 * GUI for selecting time window for trending plots.
 * Provides a way to select time window from a list of presets and to open custom time window
 * creation and preset management dialogs, as well as access the currently selected time window.
 *
 * @author onoprien
 */
final public class TimeWindowComboBox extends JComboBox {
    
// -- Private parts : ----------------------------------------------------------
       
    static private final String CUSTOM_BUTTON = "Choose...";
    static private final String PRESETS_BUTTON = "Presets...";
    static private final String DEFAULT_NAME = "custom";
    static private final int MAX_PRESETS = 10; // Maximum number of presets to include in drop-down list
    
    private final PresetList timeWindowList;
    private volatile TimeWindow selection;
    

// -- Construction and initialization : ----------------------------------------
    
    public TimeWindowComboBox(Studio application) {
        
        timeWindowList = new PresetList(application);
        
        setRenderer(new Renderer(getRenderer()));
        setMaximumRowCount(MAX_PRESETS + 5);
        
        Model model = new Model(timeWindowList);
        setModel(model);
        setSelectedIndex(1);
        selection = model.getSelectedItem();
        
        addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                TimeWindow ts = myModel().getSelectedItem();
                
                switch (ts.getName()) {
                    
                    case CUSTOM_BUTTON: // custom time window has been entered
                        
                        ts = TimeWindowPanel.editTimeWindow(TimeWindowComboBox.this, null);
                        if (ts == null) {
                            setSelectedItem(selection);
                        } else {
                            if (ts.getName().isEmpty()) {
                                ts.setName(DEFAULT_NAME);
                                if (DEFAULT_NAME.equals(selection.getName())) {
                                    myModel().removeElementAt(4);
                                }
                                myModel().insertElementAt(ts, 4);
                                setSelectedIndex(4);
                            } else {
                                timeWindowList.insert(ts);
                                Model model = new Model(timeWindowList);
                                model.setSelectedItem(ts);
                                setModel(model);
                            }
                            selection = ts;
                        }
                        break;
                        
                    case PRESETS_BUTTON: // one of the presets has been selected through presets dialog
                        
                        ts = PresetsDialog.managePresets(TimeWindowComboBox.this, timeWindowList);
                        if (ts == null) ts = selection;
                        ts.touch();
                        Model model = new Model(timeWindowList);
                        int index = model.getIndexOf(ts);
                        if (index == -1) index = 1;
                        selection = model.getElementAt(index);
                        model.setSelectedItem(selection);
                        setModel(model);
                        break;
                        
                    default: // one of the presets has been selected from drop-down list
                        
                        if (!DEFAULT_NAME.equals(ts.getName())) {
                            ts.touch();
                            if (DEFAULT_NAME.equals(selection.getName())) {
                                myModel().removeElementAt(4);
                            }
                        }
                        selection = ts;
                        
                }
            }
        });
        
    }
    
// -- Getters : ----------------------------------------------------------------
    
    private Model myModel() {
        return (Model) super.getModel();
    }
    
    /** Returns currently selected time window. */
    public TimeWindow getSelectedTimeWindow() {
        return selection;
    }
    
    /** 
     * Returns <tt>TimeSelectionList</tt> instance that maintains a list of time 
     * window presets currently known to the application.
     */
    public PresetList getPresetList() {
        return timeWindowList;
    }
    
// -- Custom renderer : --------------------------------------------------------
    
    private class Renderer implements ListCellRenderer {

        private final ListCellRenderer horse;
        private final JPanel separatorPanel = new JPanel(new BorderLayout());
        private final JSeparator separator = new JSeparator();

        Renderer(ListCellRenderer renderer) {
            horse = renderer;
        }

        @Override
        public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
            Component comp = horse.getListCellRendererComponent(list, ((TimeWindow)value).getName(), index, isSelected, cellHasFocus);
            if (index == 0 || index == 3) {
                separatorPanel.removeAll();
                separatorPanel.add(comp, BorderLayout.CENTER);
                separatorPanel.add(separator, BorderLayout.SOUTH);
                comp = separatorPanel;
            } else if (index > 4 && index == TimeWindowComboBox.this.getItemCount()-1) {
                separatorPanel.removeAll();
                separatorPanel.add(comp, BorderLayout.CENTER);
                separatorPanel.add(separator, BorderLayout.NORTH);
                comp = separatorPanel;
            }
            return comp;
        }

    }
    
// -- Custom model : -----------------------------------------------------------
    
    private class Model extends DefaultComboBoxModel<TimeWindow> {
        
        Model(PresetList tsList) {

        // Add element for launching time range construction dialog :
            
            addElement(new TimeWindow(CUSTOM_BUTTON));

        // Add standard ranges :
            
            TimeWindow selectedItem = new TimeWindow("Last Hour", "now-3600", "now", false);
            addElement(selectedItem);
            addElement(new TimeWindow("Last 6 Hours", "now-21600", "now", false));
            addElement(new TimeWindow("Last 24 Hours", "now-86400", "now", false));

        // Add custom ranges :
            
            List<TimeWindow> customRanges = tsList.getRecent(MAX_PRESETS);

            for (TimeWindow ts : customRanges) {
                addElement(ts);
            }

        // Add element for launching time range presets dialog :
            
            addElement(new TimeWindow(PRESETS_BUTTON));

        // Set default selection :
            
            setSelectedItem(selectedItem);
        }
    
        @Override
        public TimeWindow getSelectedItem() {
            return (TimeWindow) super.getSelectedItem();
        }
        
    }
    
}
