package org.lsst.ccs.gconsole.jas3;

import java.awt.Dimension;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JTextField;
import org.freehep.jas.services.PreferencesTopic;
import org.lsst.ccs.gconsole.Console;

/**
 *
 * @author onoprien
 */
public class PreferencePage  implements PreferencesTopic {

// -- Fields : -----------------------------------------------------------------
    
    private final Console console;
    private final String[] path;
    private final ArrayList<Setter> setters = new ArrayList<>(1);

// -- Life cycle : -------------------------------------------------------------
    
    public PreferencePage(Console console, String[] path) {
        this.console = console;
        this.path = path;
    }
    
    synchronized public void add(String group, String format) {
        setters.add(new Setter(group,format));
        setters.trimToSize();
    }

// -- Implementing PreferencesTopic --------------------------------------------

    @Override
    public String[] path() {
        return path;
    }

    @Override
    synchronized public JComponent component() {
        try {
            return new GUI();
        } catch (IllegalArgumentException x) {
            return new JLabel("  "+ x.getMessage() +"  ");
        }
    }

    @Override
    public boolean apply(JComponent panel) {
        try {
            return ((GUI) panel).apply();
        } catch (ClassCastException x) {
            return true;
        }
    }


// -- Local classes : ----------------------------------------------------------
    
    private class GUI extends Box {
    
        private final int HSPACE = 10;
        private final int VSPACE = 5;
        private final Pattern ptrn = Pattern.compile("\\$\\{[^}]+\\}");
        
        private final HashMap<String,PropertyData> data = new HashMap<>();
        
        GUI() {
            super(BoxLayout.Y_AXIS);
            setBorder(BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE));
            
            HashMap<String,Box> groups = new HashMap<>();
            
            for (Setter setter : setters) {
                
                Box line = Box.createHorizontalBox();
                if (setter.group == null) {
                    add(line);
                    add(Box.createRigidArea(new Dimension(0,VSPACE)));
                } else {
                    Box groupBox = groups.get(setter.group);
                    if (groupBox == null) {
                        groupBox = Box.createVerticalBox();
                        groupBox.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createTitledBorder(setter.group), BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));
                        groups.put(setter.group, groupBox);
                        add(Box.createRigidArea(new Dimension(0,VSPACE)));
                        add(groupBox);
                        add(Box.createRigidArea(new Dimension(0,VSPACE)));
                    }
                    groupBox.add(line);
                    groupBox.add(Box.createRigidArea(new Dimension(0,VSPACE)));
                }
                
                Matcher mat = ptrn.matcher(setter.format);
                int i=0;
                while (mat.find()) {
                    String s = setter.format.substring(i, mat.start());
                    if (!s.isEmpty()) {
                        line.add(new JLabel(s));
                        line.add(Box.createRigidArea(new Dimension(HSPACE,0)));
                    }
                    i = mat.end();
                    String key = setter.format.substring(mat.start()+2, mat.end()-1);
                    Object currentValue = console.getProperty(key);
                    if (currentValue == null) throw new IllegalArgumentException("No such property: "+ key);
                    PropertyData pd = PropertyData.create(currentValue);
                    line.add(pd.getComponent());
                    line.add(Box.createRigidArea(new Dimension(HSPACE,0)));
                    PropertyData old = data.put(key, pd);
                    if (old != null) throw new IllegalArgumentException("Multiple setters for property: "+ key);
                }
                String s = setter.format.substring(i, setter.format.length());
                if (!s.isEmpty()) {
                    line.add(new JLabel(s));
                    line.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
                }

                line.add(Box.createHorizontalGlue());
            }
            
            add(Box.createVerticalGlue());
        }
        
        boolean apply() {
            try {
                Map<String,Object> newValues = data.entrySet().stream().collect(Collectors.toMap(e->e.getKey(), e->e.getValue().getValue()));
                console.setProperties(newValues);
                return true;
            } catch (IllegalArgumentException x) {
                return false;
            }
        }
        
    }
    
    private class Setter {
        Setter(String group, String format) {
            this.group = group;
            this.format = format;
        }
        String group;
        String format;
    }
    
    static abstract private class PropertyData {
        static PropertyData create(Object value) {
            switch (PropertyHandler.getType(value)) {
                case BOOLEAN:
                    return new BooleanData(value);
                case INTEGER:
                    return new IntegerData(value);
                case LONG:
                    return new LongData(value);
                case DOUBLE:
                    return new DoubleData(value);
                case STRING:
                    return new StringData(value);
                case ENUM:
                    return new EnumData(value);
                default:
                    throw new AssertionError("Unknown property type: " + PropertyHandler.getType(value));

            }
        }
        abstract Object getValue();
        abstract JComponent getComponent();
    }
    
    static private class BooleanData extends PropertyData {
        
        final JCheckBox checkBox;
        
        BooleanData(Object value) {
            checkBox = new JCheckBox();
            checkBox.setSelected((Boolean)value);
        }

        @Override
        Object getValue() {
            return checkBox.isSelected();
        }

        @Override
        JComponent getComponent() {
            return checkBox;
        }
    }
    
    static private class StringData extends PropertyData {
        
        final JTextField textField;
        
        StringData(Object value) {
            textField = new JTextField(value.toString());
            textField.setMaximumSize(new Dimension(Integer.MAX_VALUE, textField.getPreferredSize().height));
        }

        @Override
        String getValue() {
            return textField.getText();
        }

        @Override
        JComponent getComponent() {
            return textField;
        }
    }
    
    static private class IntegerData extends PropertyData {
        
        final JTextField textField;
        
        IntegerData(Object value) {
            textField = new JTextField(value.toString(), 11);
            textField.setMaximumSize(textField.getPreferredSize());
        }

        @Override
        Object getValue() {
            return Integer.valueOf(textField.getText());
        }

        @Override
        JComponent getComponent() {
            return textField;
        }
    }
    
    static private class LongData extends PropertyData {
        
        final JTextField textField;
        
        LongData(Object value) {
            textField = new JTextField(value.toString(), 20);
            textField.setMaximumSize(textField.getPreferredSize());
        }

        @Override
        Object getValue() {
            return Long.valueOf(textField.getText());
        }

        @Override
        JComponent getComponent() {
            return textField;
        }
    }
    
    static private class DoubleData extends PropertyData {
        
        final JTextField textField;
        
        DoubleData(Object value) {
            textField = new JTextField(value.toString(), 12);
            textField.setMaximumSize(textField.getPreferredSize());
        }

        @Override
        Object getValue() {
            return Double.valueOf(textField.getText());
        }

        @Override
        JComponent getComponent() {
            return textField;
        }
    }
    
    static private class EnumData extends PropertyData {
        
        final JComboBox combo;
        
        EnumData(Object value) {
            Enum enumValue = (Enum)value;
            enumValue.getDeclaringClass().getEnumConstants();
            
            combo = new JComboBox(enumValue.getDeclaringClass().getEnumConstants());
            combo.setSelectedItem(enumValue);
            combo.setMaximumSize(combo.getPreferredSize());
        }

        @Override
        Object getValue() {
            return combo.getSelectedItem();
        }

        @Override
        JComponent getComponent() {
            return combo;
        }
    }


    public static void main(String... args) {
        Pattern p = Pattern.compile("\\$\\{[^{}]+\\}");
        Matcher m = p.matcher("pre1${par1}pre2${par2}post");
        while (m.find()) {
            System.out.println("start "+ m.start() +" end "+ m.end() +" group "+ m.group() +" count "+ m.groupCount());
        }
        System.out.println(m.appendTail(new StringBuffer()));
    }

}
