package org.lsst.ccs.gconsole;

import java.awt.Window;
import java.util.Map;
import org.freehep.util.FreeHEPLookup;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Base class for implementing Graphical Console subsystem.
 * 
 * @author onoprien
 */
abstract public class Console extends Subsystem {
    
// -- Construction : -----------------------------------------------------------
    
    protected Console() {
        super("CCS-Graphical-Console", AgentInfo.AgentType.CONSOLE);
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns a lookup instance that can be used objects registered by various 
     * components of the graphical console.
     * @return Lookup instance.
     */
    abstract public FreeHEPLookup getLookup();

    /**
     * Returns the root component for the main Graphical Console window.
     * @return root component for the main Graphical Console window, or {@code null}
     *         if the root component is not a {@link Window}.
     */
    abstract public Window getWindow();
    
    
    /**
     * Returns the default logger associated with the graphical console.
     * @return The logger associated with the graphical console.
     */
    @Override
    public Logger getLogger() {
        return Logger.getLogger("org.lsst.ccs.ui");
    }
    
// -- Services : ---------------------------------------------------------------
    
    /**
     * Notifies the user of an application error.
     * @param message Message to be displayed.
     */
    abstract public void error(String message);
    
    /**
     * Notifies the user of an application error.
     * @param message Message to be displayed.
     * @param x Exception to be reported.
     */
    abstract public void error(String message, Exception x);
    
    /**
     * Defines a console property with the specified default value.
     * Console properties can be later modified by calls to the {@link #setProperty(String, Object)} 
     * method or through the preference dialog if their keys are used in setter format strings passed to
     * the {@link #addPreference(String[], String, String) method. Properties that have been modified 
     * from their default values are saved between the graphical console sessions.
     * <p>
     * When a property is retrieved with a call to {@link #getProperty(String)} method, the type of the
     * returned object is determined by the type of the default value supplied when the property was defined.
     * 
     * @param key Property key. Can be used to retrieve the property value.
     * @param defaultValue Default value for the property. 
     *        Must be of type {@code Boolean}, {@code Enum}, {@code Integer}, {@code Long}, {@code Double}, or {@code String}.
     * @throws IllegalArgumentException If the specified key already exists, and the corresponding value is incompatible with
     *                                  the provided default; if either key or default value is {@code null}. 
     */
    abstract public void addProperty(String key, Object defaultValue);
    
    /**
     * Removes a console property.
     * If the specified property has been previously defined through a call to {@link #addProperty(String, Object)}
     * method, the property is removed and its current value is removed from storage; calling this method for a 
     * property that has not been defined has no effect.
     * 
     * @param key Property key.
     * @return Current value of the property, or {@code null} if there was no such property.
     */
    abstract public Object removeProperty(String key);
    
    /**
     * Adds a setter for one or more previously defined properties to the preferences menu.
     * 
     * @param path Path in the preferences tree to the page where the property setter should be displayed.
     * @param group Group inside the preferences page. If {@code null}, the property setter is displayed in the default group.
     * @param format Specifies the format of the property setter.
     * @throws IllegalArgumentException If a property with the specified key already exists, and its value is incompatible with the provided default.
     */
    abstract public void addPreference(String[] path, String group, String format);
    
    /**
     * Returns the value of the property with the specified key.
     * The type of the returned object is determined by the type of the default value supplied when the property was defined.
     * If the property with the specified key has never been defined through a call to {@link #addProperty(String, Object)}
     * but exists in the console storage, it will be returned as a {@code String}.
     * 
     * @param key Property key.
     * @return Value of the specified property, or {@code null} if the property does not exist.
     * @throws IllegalArgumentException If the stored value is of type incompatible with the property type.
     */
    abstract public Object getProperty(String key);
   
    /**
     * Sets the value of the specified property.
     * If the property with the specified key has never been defined through a call 
     * to {@link #addProperty(String, Object)} method, the value will be converted 
     * to a {@code String} and stored. Such keys cannot be used in setter format 
     * strings passed to {@link #addPreference(String[], String, String) method.
     * <p>
     * If the property value is set to {@code null}, it is removed from storage, so
     * subsequent calls to {@link #getProperty(String)} will return the default
     * value, if it has been set.
     * 
     * @param key Property key.
     * @param value Property value, or {@code null} if the property is being reset to its default value.
     * @return The previous value of the specified property, or {@code null} if the property did not exist.
     */
    abstract public Object setProperty(String key, Object value);
   
    /**
     * Sets values of a set of properties.
     * See {@link #setProperty(String, Object) for details on setting properties.
     * 
     * @param properties
     */
    public void setProperties(Map<String,Object> properties) {
        properties.forEach((key,value) -> setProperty(key,value));
    }
    
    /**
     * Registers a listener that will be notified of property changes.
     * 
     * @param listener The listener to be notified.
     * @param filter Regular expression the property key has to match to trigger notification.
     */
    abstract public void addPropertyListener(PropertyListener listener, String filter);
    
    /**
     * Removes the specified property listener.
     * 
     * @param listener Listener to be removed.
     * @return {@code True} if the specified listener was registered with this console.
     */
    abstract public boolean removePropertyListener(PropertyListener listener);
    
    public interface PropertyListener {
        
        /**
         * Called by the framework when plugin properties are modified.
         *
         * @param source Source of notification.
         * @param changes For the changed properties, map of keys to their new values.
         */
        void propertiesChanged(Object source, Map<String,Object> changes);
        
    }
    
}
