package org.lsst.ccs.gconsole.plugins.tracer;

import java.awt.Color;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.SAXReader;
import org.dom4j.io.XMLWriter;
import org.freehep.util.FreeHEPLookup;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusClearedAlert;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.messaging.BusMessageFilter;
import org.openide.util.Lookup;
import org.openide.util.LookupListener;

/**
 * Maintains a collection of named message filters.
 * <p>
 * Filters come from four possible sources. Built-In filters are created by the graphical
 * console itself. Local filters are created and registered with the application lookup 
 * by the console plugins. User filters are defined by the user and stored in the 
 * "tracer/filters.xml" file in the graphical console home directory. Finally, additional
 * filters might be provided by other subsystem connected to the buses <i>(The mechanism
 * for doing this has yet to be finalized)</i>.
 * <p>
 * Each filter has a short name and a (possibly the same) full name. The full name is the
 * short name prefixed by {@link #BUILTIN}, {@link #LOCAL}, {@link #USER}, or nothing,
 * depending on the filter origin.
 * The <tt>getName()</tt> method of any filter object returns its full name. All public 
 * methods of this class that take filter names as parameters also require full names.
 * Filters and filter factories registered with the lookup have their names modified to
 * add the {@link #LOCAL} prefix.
 *
 * @author onoprien
 */
public class FilterRegistry {

// -- Fields : -----------------------------------------------------------------
    
    static final String BUILTIN = "BuiltIn/";
    static final String LOCAL = "Local/";
    static final String USER = "User/";
    
    private final LsstTracerPlugin plugin;
    
    private final Map<String,MessageFilterFactory> filters = new ConcurrentHashMap<>(4, .75f, 1);
    
    private Lookup.Result result1;
    private Lookup.Result result2;


// -- Construction and initialization : ----------------------------------------
    
    FilterRegistry(LsstTracerPlugin plugin) {
        this.plugin = plugin;
    }
    
    void init() {
        
        addFactory(new StandardFormatter(), BUILTIN);
//        addFactory(new HeartBeatSelector(true), BUILTIN);
        addFactory(new HeartBeatSelector(false), BUILTIN);
        addFactory(new CommandBusSelector(), BUILTIN);
        addFactory(new LogBusSelector(), BUILTIN);
        addFactory(new StatusBusSelector(), BUILTIN);
        addFactory(new AlertSelector(), BUILTIN);
        
        FreeHEPLookup lookup = plugin.getConsole().getLookup();
        LookupListener listener = e -> updateLocalFilters();
        Lookup.Template template = new Lookup.Template(MessageFilterFactory.class);
        result1 = lookup.lookup(template);
        result1.addLookupListener(listener);
        template = new Lookup.Template(BusMessageFilter.class);
        result2 = lookup.lookup(template);
        result2.addLookupListener(listener);
        updateLocalFilters();
        
        addUserFilters(getUserFiltersFile());
    }
    
    
// -- Operations : -------------------------------------------------------------
    
    /**
     * Returns the filter specified by the name.
     * 
     * @param name Full name of the required filter.
     * @return The filter specified by the name, or {@code null} if there is no such filter.
     * @throws IllegalArgumentException if the filter cannot be found or created.
     */
    public MessageFilter getFilter(String name) {
        MessageFilterFactory factory = filters.get(name);
        if (factory == null) throw new IllegalArgumentException("Unrecognized filter: "+ name);
        try {
            MessageFilter filter = factory.get();
            if (filter == null) throw new IllegalArgumentException("Unable to retrieve filter: "+ name);
            return filter;
        } catch (Exception x) {
            throw new IllegalArgumentException("Unable to retrieve filter: "+ name, x);
        }
    }
    
    /**
     * Saves user-defined filter to local storage.
     *
     * @param filter Filter to be saved.
     * @throws IOException if the filter cannot be saved.
     */
    public void saveFilter(UserFilter filter) throws IOException {
        try {
            
            File inputFile = getUserFiltersFile();            
            SAXReader reader = new SAXReader();
            Document document;
            try {
                document = reader.read(inputFile);
            } catch (DocumentException x) {
                document = DocumentHelper.createDocument();
                document.addElement( "filters" );
            }
            
            String name = filter.getName();
            if (name.startsWith(USER)) {
                name = name.substring(USER.length(), name.length());
            }
            Element filtersElement = (Element) document.selectSingleNode("/filters");
            Element filterElement = (Element) filtersElement.selectSingleNode("filter[@name='"+ name +"']");
            if (filterElement != null) {
                filtersElement.remove(filterElement);
            }
            filterElement = filtersElement.addElement("filter");
            filterElement.addAttribute("name", name);
            String description = filter.getDescription();
            if (description != null) {
                filterElement.addElement("description").addText(description);
            }
            Element stepsElement = filterElement.addElement("steps");
            int nSteps = filter.filters.length;
            for (int step=0; step<nSteps; step++) {
                TracerFilter tf = filter.filters[step];
                Element stepElement = stepsElement.addElement("step");
                if (filter.or[step]) stepElement.addElement("or");
                stepElement.addElement("mode").addText(tf.getMode().name());
                if (tf.isInverted()) stepElement.addElement("invert");
                if (tf.isFormatting()) stepElement.addElement("format");
                stepElement.addElement("target").addText(tf.getTarget().name());
                stepElement.addElement("method").addText(tf.getMethod().name());
                stepElement.addElement("code").addText(tf.getCode());
                Color color = tf.getColor();
                if (color != null) {
                  stepElement.addElement("color").addText(encode(color));
                }
                FilteredMessage.Flag action = tf.getAction();
                if (action != null) {
                    stepElement.addElement("action").addText(action.name());
                }
            }
            
            OutputFormat format = OutputFormat.createPrettyPrint();
            if (!inputFile.exists()) {
                inputFile.getParentFile().mkdirs();
            }
            XMLWriter writer = new XMLWriter(new FileOutputStream(inputFile), format);
            writer.write(document);
            
            addFactory(new UserFilterFactory(name, description), null);
            
        } catch (NullPointerException|ClassCastException|IndexOutOfBoundsException|
                 IllegalArgumentException|IOException x) {
            throw new IOException("Unable to save filter "+ filter.getName(), x);
        }
    }
    
    /**
     * Deletes user-defined filter from local storage.
     * Calling this method with a name of a non-existing filter has no effect.
     * 
     * @param name Full name of the filter to be deleted.
     * @throws IOException if the filter cannot be deleted.
     */
    public void deleteFilter(String name) throws IOException {
        
        if (!name.startsWith(USER)) return;
            
        filters.remove(name);
        name = name.substring(USER.length(), name.length());
        
        try {
            File inputFile = getUserFiltersFile();            
            SAXReader reader = new SAXReader();
            Document document = reader.read(inputFile);
            Element filtersElement = (Element) document.selectSingleNode("/filters");
            Element filterElement = (Element) filtersElement.selectSingleNode("filter[@name='"+ name +"']");
            if (filterElement != null) {
                filtersElement.remove(filterElement);
                OutputFormat format = OutputFormat.createPrettyPrint();
                XMLWriter writer = new XMLWriter(new FileOutputStream(inputFile), format);
                writer.write(document);
            }
        } catch (DocumentException|NullPointerException|ClassCastException|IllegalArgumentException|IOException x) {
            throw new IOException("Unable to delete filter "+ USER + name, x);
        }
    }
    
    /**
     * Returns a map of available filters full names to their descriptions.
     * @return Map of available filters full names to their descriptions.
     */
    public Map<String,String> availableFilters() {
        Map<String,String> out = new TreeMap<>();
        filters.forEach((name, factory) -> {
            out.put(name, factory.getDescription());
        });
        return out;
    }


// -- Local methods : ----------------------------------------------------------
    
    void addFactory(MessageFilterFactory factory, String category) {
        if (category != null) {
            factory = new CategorizedFilter(factory, category);
        }
        filters.put(factory.getName(), factory);
    }
    
    private UserFilter openFilter(String shortName) {
        try {
            File inputFile = getUserFiltersFile();            
            SAXReader reader = new SAXReader();
            Document document = reader.read(inputFile);
            Node filterNode = document.selectSingleNode("/filters/filter[@name='"+ shortName +"']");
            List<Node> stepNodes = filterNode.selectNodes("steps/step");
            int nSteps = stepNodes.size();
            TracerFilter[] steps = new TracerFilter[nSteps];
            boolean[] ors = new boolean[nSteps];
            for (int step=0; step<nSteps; step++) {
                Node stepNode = stepNodes.get(step);
                ors[step] = stepNode.selectSingleNode("or") != null;
                String s = stepNode.selectSingleNode("mode").getText().trim();
                TracerFilter.Mode mode = TracerFilter.Mode.valueOf(s);
                boolean invert = stepNode.selectSingleNode("invert") != null;
                boolean format = stepNode.selectSingleNode("format") != null;
                s = stepNode.selectSingleNode("target").getText().trim();
                TracerFilter.Target target = TracerFilter.Target.valueOf(s);
                s = stepNode.selectSingleNode("method").getText().trim();
                TracerFilter.Method method = TracerFilter.Method.valueOf(s);
                String code = stepNode.selectSingleNode("code").getText();
                Node node = stepNode.selectSingleNode("color");
                Color color = null;
                if (node != null) {
                    s = node.getText().trim();
                    if (!s.isEmpty()) color = Color.decode("#"+ s);
                }
                node = stepNode.selectSingleNode("action");
                FilteredMessage.Flag action = null;
                if (node != null) {
                    s = node.getText().trim();
                    if (!s.isEmpty()) action = FilteredMessage.Flag.valueOf(s);
                }
                steps[step] = new TracerFilter(mode, invert, format, target, method, code, color, action, this);
            }
            return new UserFilter(USER + shortName, steps, ors);
        } catch (DocumentException|NullPointerException|ClassCastException|IndexOutOfBoundsException|IllegalArgumentException x) {
            throw new IllegalArgumentException("Unable to read filter "+ shortName +" from storage", x);
        }
    }
    
    private void updateLocalFilters() {
        
        filters.entrySet().removeIf(e -> e.getKey().startsWith(LOCAL));

        Collection c = result1.allInstances();
        c.forEach(o -> {
            addFactory((MessageFilterFactory)o, LOCAL);
        });

        c = result2.allItems();
        c.forEach(o -> {
            Lookup.Item item = (Lookup.Item) o;
            BusMessageFilter bmf = (BusMessageFilter) item.getInstance();
            addFactory(new MessageFilterAdapter(LOCAL+item.getId(), item.getDisplayName(), bmf), null);
        });
    }
    
    private File getUserFiltersFile() {
        String consoleHome = plugin.getConsole().getProperty("lsst.console.home").toString();
        if (consoleHome == null) return null;
        return new File(consoleHome +"/tracer/filters.xml");
    }
    
    private void addUserFilters(File file) {
        if (file == null) return;
        try {
            SAXReader reader = new SAXReader();
            Document document = reader.read(file);
            List<Node> nodes = document.selectNodes("/filters/filter");
            nodes.forEach(node -> {
                String name = node.valueOf("@name");
                Node descNode = node.selectSingleNode("description");
                String description = descNode == null ? null : descNode.getText().trim();
                addFactory(new UserFilterFactory(name, description), null);
            });
        } catch (DocumentException|NullPointerException x) {
//            plugin.getApplication().error("Error reading filters", x);
        }
    }
    
    static final String encode(Color color) {
        return String.format("%06x", color.getRGB() & 0x00FFFFFF);
    }


// -- Local classes : ----------------------------------------------------------

    private class UserFilterFactory implements MessageFilterFactory {
        
        private final String name;
        private final String desc;
        
        private UserFilterFactory(String shortName, String description) {
            name = shortName;
            desc = description;
        }

        @Override
        public String getName() {
            return USER + name;
        }

        @Override
        public String getDescription() {
            return desc;
        }

        @Override
        public MessageFilter get() {
            return openFilter(name);
        }

    }
    
    private class CategorizedFilter implements MessageFilter {
        
        private final MessageFilterFactory delegate;
        private final String prefix;
        
        private CategorizedFilter(MessageFilterFactory factory, String category) {
            delegate = factory;
            prefix = category;
        }

        @Override
        public String getName() {
            return prefix + delegate.getName();
        }

        @Override
        public String getDescription() {
            return delegate.getDescription();
        }

        @Override
        public MessageFilter get() {
            return new CategorizedFilter(delegate.get(), prefix);
        }

        @Override
        public FilteredMessage test(FilteredMessage filteredMessage) {
            return ((MessageFilter)delegate).test(filteredMessage);
        }
        
    }

}


// -- Builtin filters : --------------------------------------------------------

class StandardFormatter implements MessageFilter {
    
    final private SimpleDateFormat dtFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

    @Override
    public String getDescription() {
        return "<html>Accepts all messages.<p>Formats output as<br>[Date Time] [Origin]: [Message]<br>The message depends on the type of the bus message.";
    }

    @Override
    public String getName() {
        return "Format/Standard";
    }

    @Override
    public FilteredMessage test(FilteredMessage filteredMessage) {
        
        BusMessage bm = filteredMessage.getBusMessage();
        
        if (filteredMessage.getMessage() != null) {
            return filteredMessage;
        } else if (bm instanceof LogMessage) {
            LogMessage lm = (LogMessage)bm;
            filteredMessage.setMessage(lm.getFormattedDetails());
            return filteredMessage;
        }
        
        Date date = new Date(bm.getTimeStamp());
        StringBuilder sb = new StringBuilder(dtFormat.format(date));
        sb.append(" ").append(bm.getOriginAgentInfo().getName()).append(": ");
        
        if (bm instanceof StatusHeartBeat) {
            StatusHeartBeat sm = (StatusHeartBeat)bm;
            sb.append(" Heart Beat.").append(" State: ").append(sm.getState());
        } else if (bm instanceof StatusMessage) {
            StatusMessage sm = (StatusMessage)bm;
            sb.append(sm.getClassName()).append(" State: ").append(sm.getState());
        } else if (bm instanceof CommandRequest) {
            CommandRequest cr = (CommandRequest)bm;
            sb.append("Command ").append(cr.getBasicCommand().getCommand()).append(" to ").append(cr.getDestination()).append(".");
        } else if (bm instanceof CommandNack) {
            CommandNack cn = (CommandNack)bm;
            sb.append("Command not accepted. Reason: ").append(cn.getReason());
        } else if (bm instanceof CommandAck) {
            sb.append("Command not accepted.");
        } else if (bm instanceof CommandResult) {
            CommandResult cr = (CommandResult)bm;
            if (cr.wasSuccessful()) {
                sb.append("Command executed.");
            } else {
                sb.append("Command failed.");
            }
        } else {
            sb.append(bm);
        }
        
        filteredMessage.setMessage(sb.toString());
        return filteredMessage;
    }

}

class HeartBeatSelector implements MessageFilter {
    
    private final String name;
    private final String description;
    
    private final boolean invert;
    
    HeartBeatSelector(boolean invert) {
        this.invert = invert;
        if (invert) {
            name = "Reject Heart Beat";
            description ="Rejects heart beat status messages";
        } else {
            name = "Heart Beat Selector";
            description ="Accepts only heart beat status messages";
        }
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public FilteredMessage test(FilteredMessage filteredMessage) {
        boolean accept = filteredMessage.getBusMessage() instanceof StatusHeartBeat;
        return invert^accept ? filteredMessage : null;
    }
    
}

class CommandBusSelector implements MessageFilter {
    @Override
    public String getName() {
        return "Bus/Command";
    }
    @Override
    public String getDescription() {
        return "Selects command bus messages";
    }
    @Override
    public FilteredMessage test(FilteredMessage filteredMessage) {
        boolean accept = filteredMessage.getBusMessage() instanceof org.lsst.ccs.bus.messages.CommandMessage;
        return accept ? filteredMessage : null;
    }
}

class LogBusSelector implements MessageFilter {
    @Override
    public String getName() {
        return "Bus/Log";
    }
    @Override
    public String getDescription() {
        return "Selects log bus messages";
    }
    @Override
    public FilteredMessage test(FilteredMessage filteredMessage) {
        boolean accept = filteredMessage.getBusMessage() instanceof org.lsst.ccs.bus.messages.LogMessage;
        return accept ? filteredMessage : null;
    }
}

class StatusBusSelector implements MessageFilter {
    @Override
    public String getName() {
        return "Bus/Status";
    }
    @Override
    public String getDescription() {
        return "Selects status bus messages";
    }
    @Override
    public FilteredMessage test(FilteredMessage filteredMessage) {
        boolean accept = filteredMessage.getBusMessage() instanceof org.lsst.ccs.bus.messages.StatusMessage;
        return accept ? filteredMessage : null;
    }
}
    

class AlertSelector implements MessageFilter {
    @Override
    public String getName() {
        return "Alerts";
    }
    @Override
    public String getDescription() {
        return "Selects alert status messages";
    }
    @Override
    public FilteredMessage test(FilteredMessage filteredMessage) {
        BusMessage bm = filteredMessage.getBusMessage();
        boolean accept = bm instanceof StatusRaisedAlert || bm instanceof StatusClearedAlert;
        return accept ? filteredMessage : null;
    }
}
