package org.lsst.ccs.gconsole.plugins.trending;

import java.awt.Component;
import java.util.ArrayList;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import javax.swing.SwingUtilities;
import org.freehep.application.mdi.PageContext;
import org.freehep.application.mdi.PageEvent;
import org.freehep.application.mdi.PageListener;
import org.freehep.application.mdi.PageManager;
import org.freehep.jas.services.PlotPage;
import org.freehep.jas.services.PlotRegion;
import org.freehep.jas.services.Plotter;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Handles automatic refreshing of trending plots.
 * The class is thread-safe, all public methods can be called from any thread.
 *
 * @author onoprien
 */
final public class AutoRefresh {

// -- Private parts : ----------------------------------------------------------
    
    private static final Logger logger = Logger.getLogger(AutoRefresh.class.getName());
    private final LsstTrendingPlugin plugin;
    
    private ScheduledFuture periodicRefresh;
    private ScheduledThreadPoolExecutor executor;
    private final int DATA_FETCHER_MAX_THREADS = 2;
    
    private PageListener pageListener;
    
    private volatile long[] delayBounds;
    private volatile boolean cancel;
    

// -- Construction and initialization : ----------------------------------------
    
    public AutoRefresh(LsstTrendingPlugin plugin) {
        this.plugin = plugin;
    }
    
    /**
     * Sets the allowed range of refresh frequency.
     * 
     * @param min Minimum refresh interval.
     * @param max Maximum refresh interval.
     * @param unit Time unit used to specify minimum and maximum intervals.
     */
    public void setDelay(long min, long max, TimeUnit unit) {
        if (min > max) {
            throw new IllegalArgumentException();
        } else if (min == max) {
            delayBounds = new long[] {TimeUnit.MILLISECONDS.convert(min, unit)};
        } else {
            delayBounds = new long[] {TimeUnit.MILLISECONDS.convert(min, unit), TimeUnit.MILLISECONDS.convert(max, unit)};
        }
    }


// -- Operation : --------------------------------------------------------------
    
    /**
     * Starts auto-refreshing.
     * Compiles a list of all regions that need refreshing, puts them in the queue,
     * and starts data fetcher thread.
     */
    public synchronized void start() {
        
        TrendingPreferences pref = plugin.getPreferences();
        setDelay(pref.refreshMin(), pref.refreshMax(), TimeUnit.SECONDS);
        
        // if executor exists, cancel tasks; if not, start executor
        
        if (executor == null) {
            
            ThreadFactory tFactory = new ThreadFactory() {
                ThreadFactory horse = Executors.defaultThreadFactory();
                int id = 0;
                public Thread newThread(Runnable r) {
                    Thread thread = horse.newThread(r);
                    thread.setPriority(Math.min(Thread.NORM_PRIORITY, Thread.currentThread().getPriority()-1));
                    thread.setName("Autorefresh executor "+ id++);
                    return thread;
                }
            };
            executor = new ScheduledThreadPoolExecutor(DATA_FETCHER_MAX_THREADS, tFactory);
            
        } else {
            
            if (periodicRefresh != null) {
                periodicRefresh.cancel(true);
            }
            
        }
        
        // submit periodic task
        
        schedulePeriodicTask();
        
        // listen to page selections

        pageListener = pe -> {
            if (pe.getID() == PageEvent.PAGESELECTED) {
                try {
                    PlotPage page = (PlotPage) pe.getPageContext().getPage();
                    executor.execute(() -> {
                        try {
                            refreshPage(page);
                        } catch (Throwable t) {
                        }
                    });
                } catch (ClassCastException|NullPointerException x) {
                }
            }
        };
        plugin.getApplication().getPageManager().addPageListener(pageListener);
        
    }
    
    /**
     * Stops auto-refreshing.
     * Cancels auto-refreshing tasks and stops data fetcher executor.
     */
    public synchronized void stop() {
//        System.out.println("stop autorefresh");
        cancel = true;
        plugin.getApplication().getPageManager().removePageListener(pageListener);
        pageListener = null;
        executor.shutdown();
        executor = null;
    }
    
    public synchronized void updatePreferences() {
//        System.out.println("Autorefresh updating preferences");
        TrendingPreferences pref = plugin.getPreferences();
        long minDelay = delayBounds[0];
        setDelay(pref.refreshMin(), pref.refreshMax(), TimeUnit.SECONDS);
        if (minDelay != delayBounds[0] && periodicRefresh != null) {
            periodicRefresh.cancel(true);
            schedulePeriodicTask();
        }
    }


// -- Code to run on data fetcher thread : -------------------------------------
    
    private void refreshAll() {
        PageManager pm = plugin.getApplication().getPageManager();
        for (Object o : new ArrayList<>(pm.pages())) {
            try {
                PlotPage page = (PlotPage) ((PageContext) o).getPage();
                refreshPage(page);
            } catch (NullPointerException | ClassCastException x) {
            }
        }
    }
    
    private void refreshVisible() {
        PageManager pm = plugin.getApplication().getPageManager();
        for (Object o : new ArrayList<>(pm.pages())) {
            try {
                PlotPage page = (PlotPage) ((PageContext) o).getPage();
                if (((Component)page).isShowing()) {
                    refreshPage(page);
                }
            } catch (NullPointerException | ClassCastException x) {
            }
        }
    }
    
    private void refreshPage(PlotPage page) {
        int n = page.numberOfRegions();
        for (int i=0; i<n; i++) {
            if (cancel || Thread.currentThread().isInterrupted()) return;
            PlotRegion region = page.region(i);
            if (isRefreshRequired(region)) {
                Plotter plotter = region.currentPlot();
                if (plotter != null) {
                    boolean needUpdate = false;
                    long time = System.currentTimeMillis();
                    for (Object plottedObject : PlotMaker.getPlotterData(plotter)) {
                        try {
                            Plot plot = (Plot) plottedObject;
                            needUpdate = plot.getChannel().refresh(plot, time) || needUpdate;
                        } catch (ClassCastException x) {
                        }
                    }
                    if (needUpdate) {
                        SwingUtilities.invokeLater(() -> {
                            plugin.getPlotMaker().refresh(region, null);
                        });
                    }
                }
            }
        }
    }


// -- Local methods : ----------------------------------------------------------
    
    private long computeDelay(Plot pd) {
        if (delayBounds.length == 1) {
            return delayBounds[0];
        } else {
            long delay = Math.round(pd.getGranularity()*1.1);
            return Math.min(delayBounds[1], Math.max(delay, delayBounds[0]));
        }
    }
    
    private boolean isRefreshRequired(PlotRegion region) {
        Plotter plotter = region.currentPlot();
        if (plotter == null) return false;
        return PlotMaker.getPlotterData(plotter).stream().anyMatch(plottedObject -> {
            try {
                Plot pd = (Plot) plottedObject;
                if (!pd.getTimeWindow().isFixed()) {
                    return pd.getTimeStamp() + computeDelay(pd) < System.currentTimeMillis();
                }
            } catch (ClassCastException x) {
            }
            return false;
        });
    }
    
    private void schedulePeriodicTask() {
        Runnable periodicTask = () -> {
            try {
                refreshVisible();
            } catch (Throwable t) {
                // anything happens - simply abort this refresh task
            }
        };
        cancel = false;
        periodicRefresh = executor.scheduleWithFixedDelay(periodicTask, delayBounds[0], delayBounds[0], TimeUnit.MILLISECONDS);
    }
    
}
