package org.lsst.ccs.gconsole.plugins.tracer;

import java.awt.Component;
import java.awt.Point;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import org.freehep.application.mdi.PageEvent;
import org.freehep.application.mdi.PageListener;
import org.freehep.swing.popup.HasPopupItems;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandMessage;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.CommandMessageListener;
import org.lsst.ccs.messaging.LogMessageListener;
import org.lsst.ccs.messaging.StatusMessageListener;

/**
 * Tracer that receives, filters, and prints bus messages to a console panel.
 * An instance of this class is associated with each tracer output console.
 * Each Tracer can be independently configured.
 *
 * @author tonyj
 */
public class Tracer implements PageListener, StatusMessageListener, LogMessageListener, CommandMessageListener, HasPopupItems {
    
// -- Private parts : ----------------------------------------------------------
    
    private final LsstTracerPlugin plugin;
    private final AgentMessagingLayer messagingAccess;
    private TracerWriter out;
    private volatile MessageFilter filter;
    
// -- Construction and initialization : ----------------------------------------

    Tracer(LsstTracerPlugin plugin) {
        this(plugin, null);
    }

    Tracer(LsstTracerPlugin plugin, MessageFilter filter) {
        this.plugin = plugin;
        this.messagingAccess = plugin.getMessagingAccess();
        this.filter = filter;
    }
    
// -- Operations : -------------------------------------------------------------
    
    public void setFilter(MessageFilter filter) {
        this.filter = filter;
        if (out != null) out.reset();
    }
    
    @Override
    public void onStatusMessage(StatusMessage msg) {
        onBusMessage(msg);
    }

    @Override
    public void onLogMessage(LogMessage msg) {
        onBusMessage(msg);
    }

    @Override
    public void onCommandMessage(CommandMessage msg) {
        onBusMessage(msg);
    }
    
    private void onBusMessage(BusMessage message) {
        if (filter == null) {
            out.println(message);
        } else {
            FilteredMessage filteredMessage = new TracerMessage(message);
            filteredMessage = filter.test(filteredMessage);
            if (filteredMessage != null) {
                out.println(filteredMessage);
            }
        }
    }

    @Override
    public void pageChanged(PageEvent pe) {
        if (pe.getID() == PageEvent.PAGECLOSED) {
            messagingAccess.removeStatusMessageListener(this);
        }
    }

    void start(TracerWriter out) {
        this.out = out;
        messagingAccess.addStatusMessageListener(this);
    }

    @Override
    public JPopupMenu modifyPopupMenu(JPopupMenu menu, final Component source, Point p) {

        JMenu filterMenu = new JMenu("Filter...");

        JMenuItem item = new JMenuItem("New...");
        item.setToolTipText("Define new filter");
        item.addActionListener(e -> {
            UserFilter f = FilterEditor.showDialog(source, null, plugin);
            if (f != null) setFilter(f);
        });
        filterMenu.add(item);

        item = new JMenuItem("Load...");
        item.setToolTipText("Load named filter.");
        item.addActionListener(e -> {
            FilterRegistry registry = plugin.getFilterRegistry();
            String name = FilterChooser.showDialog(source, registry);
            if (name != null) {
                try {
                    MessageFilter f = registry.getFilter(name);
                    if (f == null) {
                        plugin.getApplication().error("Unknown message filter: "+ name);
                    } else {
                        setFilter(f);
                    }
                } catch (IllegalArgumentException x) {
                    plugin.getApplication().error("Error initializing message filter "+ name, x);
                }
            }
        });
        filterMenu.add(item);

        item = new JMenuItem("Edit...");
        item.setToolTipText("Edit filter");
        item.addActionListener(e -> {
            UserFilter f = FilterEditor.showDialog(source, filter, plugin);
            if (f != null) setFilter(f);
        });
        filterMenu.add(item);

        item = new JMenuItem("Remove");
        item.setToolTipText("Remove filter");
        item.setEnabled(filter != null);
        item.addActionListener(e -> setFilter(null));
        filterMenu.add(item);
        
        menu.insert(filterMenu,0);
        return menu;
    }
        
}
