package org.lsst.ccs.gconsole.plugins.tracer;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Window;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.IOException;
import java.util.*;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import javax.swing.AbstractCellEditor;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.CellEditor;
import javax.swing.DefaultCellEditor;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.WindowConstants;
import javax.swing.border.Border;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableModel;
import org.freehep.swing.JComboBoxWithDisabledItems;
import org.lsst.ccs.gconsole.plugins.tracer.TracerFilter.Method;
import org.lsst.ccs.gconsole.plugins.tracer.TracerFilter.Mode;
import org.lsst.ccs.gconsole.plugins.tracer.TracerFilter.Target;
import org.openide.awt.ColorComboBox;

/**
 * GUI editor for {@link UserFilter}.
 *
 * @author onoprien
 */
public class FilterEditor extends JDialog {

// -- Fields : -----------------------------------------------------------------
    
    private final int VSPACE = 5;
    private final int HSPACE = 5;
    
    private final int C_OR = 0;
    private final int C_MODE = 1;
    private final int C_INVERT = 2;
    private final int C_FORMAT = 3;
    private final int C_TARGET = 4;
    private final int C_METHOD = 5;
    private final int C_CODE = 6;
    private final int C_COLOR = 7;
    private final int C_FLAG = 8;
    private final int C_VALID = 9;
    private final String[] columnNames = {"OR", "Mode", "Invert", "Format", "Target", "Method", "Definition", "Color", "Flag"};
    private final Class[] columnClasses = {Boolean.class, TracerFilter.Mode.class, Boolean.class, Boolean.class, TracerFilter.Target.class, TracerFilter.Method.class, String.class, Color.class, FilteredMessage.Flag.class};
    
    private final Predicate<String> isValidName = Pattern.compile("[\\w-\\. /]+").asPredicate();
    
    private JTable table;
    private Model model;
    private JButton insertButton;
    private JButton removeButton;
    private JButton upButton;
    private JButton downButton;
    private JButton saveButton;
    private JButton saveAsButton;
    private JButton okButton;
    
    private UserFilter out;
    private String name;
    private final LsstTracerPlugin plugin;

// -- Construction and initialization : ----------------------------------------
    
    private FilterEditor(Component parent, MessageFilter filter, LsstTracerPlugin plugin) {
        
        super(getWindow(parent), "Unnamed Filter", DEFAULT_MODALITY_TYPE.APPLICATION_MODAL);
        
        this.plugin = plugin;
        setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
        if (filter != null && filter instanceof UserFilter) {
            name = filter.getName();
            if (name != null) setTitle(name);
        }
        setPreferredSize(new Dimension(800,600));
        
        model = new Model(filter);
        table = new Table(model);
        table.setFillsViewportHeight(true);
        table.setSelectionMode(ListSelectionModel.SINGLE_INTERVAL_SELECTION);
        table.getSelectionModel().addListSelectionListener(e -> {
            int first = table.getSelectedRow();
            int count = table.getSelectedRowCount();
            insertButton.setEnabled(count > 0);
            removeButton.setEnabled(count > 0);
            upButton.setEnabled(first > 0);
            downButton.setEnabled(count > 0 && first+count < table.getRowCount());
        });
        add(new JScrollPane(table), BorderLayout.CENTER);
        
        // mode
        
        JComboBox combo = new JComboBox();
        for (Mode state : Mode.values()) {
            combo.addItem(state);
        }
        DefaultCellEditor editor = new DefaultCellEditor(combo);
        editor.setClickCountToStart(2);
        table.getColumnModel().getColumn(C_MODE).setCellEditor(editor);
        
        // invert
        
        JCheckBox cb = new JCheckBox();
        cb.setMaximumSize(cb.getMinimumSize());
        cb.setPreferredSize(cb.getMinimumSize());
        editor = new DefaultCellEditor(cb);
        table.getColumnModel().getColumn(C_INVERT).setCellEditor(editor);

        // target
        
        combo = new JComboBox();
        for (Target target : Target.values()) {
            combo.addItem(target);
        }
        editor = new DefaultCellEditor(combo);
        editor.setClickCountToStart(2);
        table.getColumnModel().getColumn(C_TARGET).setCellEditor(editor);
        
        // method
        
        table.getColumnModel().getColumn(C_METHOD).setCellEditor(new MethodEditor());
        
        // code
        
        table.getColumnModel().getColumn(C_CODE).setCellRenderer(new CodeRenderer());
        table.getColumnModel().getColumn(C_CODE).setCellEditor(new CodeEditor());
        
        // color
        
        table.getColumnModel().getColumn(C_COLOR).setCellRenderer(new ColorRenderer());
        table.getColumnModel().getColumn(C_COLOR).setCellEditor(new ColorEditor());
        
        // action
        
        combo = new JComboBox();
        for (FilteredMessage.Flag act : FilteredMessage.Flag.values()) {
            combo.addItem(act);
        }
        editor = new DefaultCellEditor(combo);
        editor.setClickCountToStart(2);
        table.getColumnModel().getColumn(C_FLAG).setCellEditor(editor);
        
        // top controls
        
        Box controls = Box.createHorizontalBox();
        controls.setBorder(BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE));
        add(controls, BorderLayout.NORTH);
        JButton button = new JButton("Add");
        button.setToolTipText("Add filter");
        button.addActionListener(e -> {
            CellEditor ce = table.getCellEditor();
            if (ce == null || ce.stopCellEditing()) {
                model.insertRow(model.getRowCount());
            }
        });
        controls.add(button);
        controls.add(Box.createHorizontalStrut(HSPACE));
        insertButton = new JButton("Insert");
        insertButton.setToolTipText("Insert filter");
        insertButton.setEnabled(false);
        insertButton.addActionListener(e -> {
            CellEditor ce = table.getCellEditor();
            if (ce == null || ce.stopCellEditing()) {
                int i = table.getSelectedRow();
                model.insertRow(i < 0 ? 0 : i);
            }
        });
        controls.add(insertButton);
        controls.add(Box.createHorizontalStrut(HSPACE));
        removeButton = new JButton("Remove");
        removeButton.setToolTipText("Remove selected filter");
        removeButton.setEnabled(false);
        removeButton.addActionListener(e -> {
            CellEditor ce = table.getCellEditor();
            if (ce == null || ce.stopCellEditing()) {
                model.removeRows(table.getSelectedRows());
            }
        });
        controls.add(removeButton);
        controls.add(Box.createHorizontalGlue());
        controls.add(Box.createHorizontalStrut(2*HSPACE));
        upButton = new JButton(" Up ");
        upButton.setToolTipText("Move selected filter up in sequence");
        upButton.setEnabled(false);
        upButton.addActionListener(e -> move(-1));
        controls.add(upButton);
        controls.add(Box.createHorizontalStrut(HSPACE));
        downButton = new JButton("Down");
        downButton.setToolTipText("Move selected filter down in sequence");
        downButton.setEnabled(false);
        downButton.addActionListener(e -> move(1));
        controls.add(downButton);
        
        // bottom controls
        
        controls = Box.createHorizontalBox();
        controls.setBorder(BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE));
        add(controls, BorderLayout.SOUTH);
        saveButton = new JButton("Save");
        saveButton.setEnabled(name != null);
        saveButton.addActionListener(e -> {
            try {
                UserFilter f = model.makeFilter();
                if (f == null) plugin.getApplication().error("Unable to save filter "+ name);
                plugin.getFilterRegistry().saveFilter(f);
            } catch (IllegalStateException|IOException x) {
                plugin.getApplication().error("Unable to save filter "+ name, x);
            }
        });
        controls.add(saveButton);
        controls.add(Box.createHorizontalStrut(HSPACE));
        saveAsButton = new JButton("Save As...");
        saveAsButton.addActionListener(e -> {
            String oldName = name;
            try {
                String newName = JOptionPane.showInputDialog(FilterEditor.this, 
                        "Enter filter name, using\n(\"/\", \"-\", \"_\", \".\", \" \", and alphanumeric symbols.",
                        "Filter name:", JOptionPane.QUESTION_MESSAGE);
                if (newName == null) return;
                newName = newName.trim();
                if (isValidName.test(newName)) {
                    name = newName;
                } else {
                    plugin.getApplication().error("Illegal filter name: "+ newName);
                    return;
                }
                UserFilter f = model.makeFilter();
                if (f == null) plugin.getApplication().error("Unable to save filter "+ name);
                plugin.getFilterRegistry().saveFilter(f);
                setTitle(name);
                saveButton.setEnabled(true);
            } catch (IllegalStateException|IOException x) {
                plugin.getApplication().error("Unable to save filter "+ name, x);
                name = oldName;
            }
        });
        controls.add(saveAsButton);
        controls.add(Box.createHorizontalStrut(2*HSPACE));
        controls.add(Box.createHorizontalGlue());
        okButton = new JButton("OK");
        okButton.addActionListener(e -> {
            out = model.makeFilter();
            if (out != null) {
                FilterEditor.this.setVisible(false);
            }
        });
        controls.add(okButton);
        controls.add(Box.createHorizontalStrut(HSPACE));
        button = new JButton("Cancel");
        button.addActionListener(e -> FilterEditor.this.setVisible(false));
        controls.add(button);

        table.getColumnModel().getColumn(C_OR).setPreferredWidth(100);
        table.getColumnModel().getColumn(C_MODE).setPreferredWidth(150);
        table.getColumnModel().getColumn(C_INVERT).setPreferredWidth(150);
        table.getColumnModel().getColumn(C_FORMAT).setPreferredWidth(150);
        table.getColumnModel().getColumn(C_TARGET).setPreferredWidth(170);
        table.getColumnModel().getColumn(C_METHOD).setPreferredWidth(170);
        table.getColumnModel().getColumn(C_CODE).setPreferredWidth(800);
        table.getColumnModel().getColumn(C_COLOR).setPreferredWidth(170);
        table.getColumnModel().getColumn(C_FLAG).setPreferredWidth(170);
        table.clearSelection();
        pack();
        setLocationRelativeTo(getOwner());
    }
    
// -- Showing chooser dialog : -------------------------------------------------
    
    static public UserFilter showDialog(Component parent, MessageFilter filter, LsstTracerPlugin plugin) {
        FilterEditor editor = new FilterEditor(parent, filter, plugin);
        editor.setVisible(true);
        UserFilter out = editor.getFilter();
        editor.dispose();
        return out;
    }
    
    static public Window getWindow(Component parent) {
        try {
            Object out;
            if (parent == null) {
                out = jas.util.Application.getApplication().getTopLevelAncestor();
            } else {
                out = ((JComponent) parent).getTopLevelAncestor();
            }
            return (Window) out;
        } catch (ClassCastException x) {
            return null;
        }
    }
    
    public UserFilter getFilter() {
        return out;
    }
    
// -- Local methods : ----------------------------------------------------------
    
    private void move(int dir) {
        
        CellEditor ce = table.getCellEditor();
        if (ce != null && !ce.stopCellEditing()) return;
       
        int begSel = table.getSelectedRow();
        int k = table.getSelectedRowCount();
        if (k == 0) return;
        int endSel = begSel+k-1;
        int n = table.getRowCount();
        
        boolean isOrFragment = true;
        for (int i=0; i<k; i++) {
            isOrFragment = isOrFragment && isOR(begSel+i);
        }
        int begNext = dir > 0 ? endSel+1 : begSel-1;
        if (begNext < 0 || begNext >= n) return;
        isOrFragment = isOrFragment && isOR(begNext);
        
        if (isOrFragment) {
            if (dir > 0) {
                model.move(begSel, endSel, begNext, begNext);
                table.getSelectionModel().setSelectionInterval(begSel+1, endSel+1);
            } else {
                model.move(begNext, begNext, begSel, endSel);
                table.getSelectionModel().setSelectionInterval(begSel-1, endSel-1);
            }
        } else {
            if (isOR(begSel)) {
                while (begSel > 0 && isOR(begSel - 1)) {
                    begSel--;
                }
            }
            if (isOR(endSel)) {
                while ((endSel+1) < n && isOR(endSel+1)) {
                    endSel++;
                }
            }
            begNext = dir > 0 ? endSel + 1 : begSel - 1;
            if (begNext < 0 || begNext >= n) return;
            int endNext = begNext;
            if (isOR(endNext)) {
                int bound = dir > 0 ? n : -1;
                while ((endNext+dir != bound) && isOR(endNext+dir)) {
                    endNext += dir;
                }
            }
            k = endNext-begNext+1;
            if (dir > 0) {
                model.move(begSel, endSel, begNext, endNext);
                table.getSelectionModel().setSelectionInterval(begSel+k, endSel+k);
            } else {
                model.move(endNext, begNext, begSel, endSel);
                table.getSelectionModel().setSelectionInterval(begSel-k, endSel-k);
            }
        }

    }
    
    private boolean isOR(int rowIndex) {
        return (Boolean) model.getValueAt(rowIndex, 0);
    }
    
    private void validateTable() {
        boolean allValid = true;
        for (int i = 0; i < model.data.length; i++) {
            allValid = (Boolean) model.data[i][C_VALID];
            if (!allValid) break;
        }
        okButton.setEnabled(allValid);
        saveButton.setEnabled(allValid && name != null);
        saveAsButton.setEnabled(allValid);
    }
    
    
// -- FILTER table model : -----------------------------------------------------
    
    private class Model extends AbstractTableModel {

        Object[][] data;
        
        Model(MessageFilter filter) {
            if (filter == null) {
                data = new Object[1][];
                data[0] = makeRow(null);
            } else if (filter instanceof UserFilter) {
                UserFilter filterChain = (UserFilter) filter;
                data = new Object[filterChain.filters.length][];
                for (int i = 0; i < filterChain.filters.length; i++) {
                    data[i] = makeRow(filterChain.filters[i]);
                    data[i][0] = filterChain.or[i];
                }
            } else {
                data = new Object[1][];
                data[0] = makeRow(filter);
            }
        }
        
        @Override
        public Class<?> getColumnClass(int columnIndex) {
            return columnClasses[columnIndex];
        }

        @Override
        public String getColumnName(int column) {
            return columnNames[column];
        }

        @Override
        public int getRowCount() {
            return data.length;
        }

        @Override
        public int getColumnCount() {
            return columnNames.length;
        }

        @Override
        public Object getValueAt(int rowIndex, int columnIndex) {
            return data[rowIndex][columnIndex];
        }

        @Override
        public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
            data[rowIndex][columnIndex] = aValue;
            if (columnIndex == C_TARGET || columnIndex == C_METHOD || columnIndex == C_CODE) {
                boolean wasValid = (Boolean) data[rowIndex][C_VALID];
                try {
                    makeFilter(rowIndex);
                    data[rowIndex][C_VALID] = true;
                } catch (IllegalArgumentException x) {
                    data[rowIndex][C_VALID] = false;
                }
                if (wasValid != (Boolean) data[rowIndex][C_VALID]) {
                    fireTableRowsUpdated(rowIndex, rowIndex);
                    validateTable();
                }
            }
        }

        @Override
        public boolean isCellEditable(int rowIndex, int columnIndex) {
            return true;
        }
        
        void insertRow(int i) {
            Object[] row = makeRow(null);
            int n = data.length;
            Object[][] newData = new Object[n+1][];
            System.arraycopy(data, 0, newData, 0, i);
            newData[i] = row;
            if (i != n) System.arraycopy(data, i, newData, i+1, n-i);   
            data = newData;
            fireTableRowsInserted(i, i);
        }
        
        void removeRows(int[] rows) {
            if (rows.length == 0) return;
            int n = data.length;
            Object[][] newData = new Object[n-rows.length][];
            Arrays.sort(rows);
            int s = 0;
            int t = 0;
            for (int r : rows) {
                int k = r-s;
                System.arraycopy(data, s, newData, t, k);
                s = r+1;
                t += k;
            }
            if (s<n) System.arraycopy(data, s, newData, t, n-s);
            data = newData;
            fireTableRowsDeleted(rows[0], rows[rows.length-1]);
            validateTable();
        }
        
        private Object[] makeRow(MessageFilter filter) {
            if (filter == null) {
                return new Object[] {false, Mode.ON, false, false, Target.OBJECT, Method.CONTAINS, "", null, null, true};
            } else if (filter instanceof TracerFilter) {
                TracerFilter f = (TracerFilter) filter;
                Object[] row = new Object[columnNames.length+1];
                row[C_OR] = false;
                row[C_MODE] = f.getMode();
                row[C_INVERT] = f.isInverted();
                row[C_FORMAT] = f.isFormatting();
                row[C_TARGET] = f.getTarget();
                row[C_METHOD] = f.getMethod();
                row[C_CODE] = f.getCode();
                row[C_COLOR] = f.getColor();
                row[C_FLAG] = f.getAction();
                row[C_VALID] = true;
                return row;
            } else {
                String name = filter.getName();
                if (name == null) name = "";
                return new Object[] {false, Mode.ON, false, false, Target.OBJECT, Method.FILTER, name, null, null, true};
            }
        }
    
        private void move(int b1, int e1, int b2, int e2) {
            Object[][] newData = new Object[data.length][];
            System.arraycopy(data, 0, newData, 0, b1);
            int i=b1;
            int k = e2-b2+1;
            System.arraycopy(data, b2, newData, i, k);
            i += k;
            k = e1-b1+1;
            System.arraycopy(data, b1, newData, i, k);
            System.arraycopy(data, e2+1, newData, e2+1, data.length-1-e2);
            data = newData;
            fireTableRowsUpdated(b1, e2);
        }
    
        private UserFilter makeFilter() {
            int nSteps = data.length;
            if (nSteps == 0) return null;
            TracerFilter[] steps = new TracerFilter[nSteps];
            boolean[] ors = new boolean[nSteps];
            FilterRegistry registry = plugin.getFilterRegistry();
            for (int i = 0; i < nSteps; i++) {
                Object[] step = data[i];
                ors[i] = (Boolean) step[C_OR];
                steps[i] = new TracerFilter((Mode) step[C_MODE], (Boolean) step[C_INVERT], (Boolean) step[C_FORMAT],
                        (Target) step[C_TARGET], (Method) step[C_METHOD], (String) step[C_CODE],
                        (Color) step[C_COLOR], (FilteredMessage.Flag) step[C_FLAG], registry);
            }
            return new UserFilter(name, steps, ors);
        }
        
        private TracerFilter makeFilter(int row) {
            Object[] step = data[row];
            return new TracerFilter((Mode) step[C_MODE], (Boolean) step[C_INVERT], (Boolean) step[C_FORMAT],
                    (Target) step[C_TARGET], (Method) step[C_METHOD], (String) step[C_CODE],
                    (Color) step[C_COLOR], (FilteredMessage.Flag) step[C_FLAG], plugin.getFilterRegistry());
        }

    }
    
    
// -- FILTER table class : -----------------------------------------------------
    
    private class Table extends JTable {
        
        Table(TableModel model) {
            super(model);
        }

        @Override
        protected JTableHeader createDefaultTableHeader() {
            return new JTableHeader(columnModel) {
                @Override
                public String getToolTipText(MouseEvent e) {
                    int index = columnModel.getColumnIndexAtX(e.getPoint().x);
                    int realIndex = columnModel.getColumn(index).getModelIndex();
                    return getToolTip(realIndex);
                }
            };
        }
        
        String getToolTip(int column) {
            StringBuilder sb;
            switch (column) {
                case C_OR:
                    return "Check if this filter is a part of an OR group";
                case C_MODE:
                    sb = new StringBuilder("<html><h3>Mode:</h3><dl>");
                    for (Mode mode : Mode.values()) {
                        sb.append("<dt>").append(mode).append("</dt><dd>").append(mode.getToolTip()).append("</dd>");
                    }
                    return sb.append("</dl></html>").toString();
                case C_INVERT:
                    return "Check if this filter acceptance criteria should be inverted";
                case C_FORMAT:
                    return "<html>Check if the target string produced by this filter should be saved and passed to subsequent filters</html>";
                case C_TARGET:
                    sb = new StringBuilder("<html><b>Target to which this filter should be applied:</b><dl>");
                    for (Target target : Target.values()) {
                        sb.append("<dt>").append(target).append("</dt><dd>").append(target.getToolTip()).append("</dd>");
                    }
                    return sb.append("</dl></html>").toString();
                case C_METHOD:
                    sb = new StringBuilder("<html>Method that determines how the filter definition string is<br> interpreted to decide whether the target satisfies the filter.<dl>");
                    for (Method method : Method.values()) {
                        sb.append("<dt>").append(method).append("</dt><dd>").append(method.getToolTip()).append("</dd>");
                    }
                    return sb.append("</dl></html>").toString();                    
                case C_CODE:
                    return "Filter definition string";
                case C_COLOR:
                    return "Color to be assigned to messages that satisfy this filter.";
                case C_FLAG:
                    return "Flag to be added to messages that satisfy this filter.";
                default:
                    return null;
            }

        }
        
    }
    
    
// -- FILTER table cell editors : ----------------------------------------------
    
    private class MethodEditor extends AbstractCellEditor implements TableCellEditor {
        
        private final JComboBoxWithDisabledItems<Method> combo;
        
        MethodEditor() {
            combo = new JComboBoxWithDisabledItems<>();
            for (Method m : Method.values()) {
                combo.addItem(m);
            }
        }

        @Override
        public Object getCellEditorValue() {
            return combo.getSelectedItem();
        }

        @Override
        public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
            Target target = (Target) table.getValueAt(row, C_TARGET);
            EnumSet<Method> tabu;
            switch (target) {
                case OBJECT:
                    tabu = EnumSet.noneOf(Method.class);
                    break;
                default:
                    tabu = EnumSet.of(Method.CLASS);
            }
            combo.setDisabled(tabu);
            combo.setSelectedItem((Method)value);
            return combo;
        }
        
    }
    
    private class ColorRenderer extends JLabel implements TableCellRenderer {
        Border unselectedBorder = null;
        Border selectedBorder = null;
        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            if (isSelected) {
                if (selectedBorder == null) {
                    selectedBorder = BorderFactory.createMatteBorder(2,3,2,3, table.getSelectionBackground());
                }
                setBorder(selectedBorder);
            } else {
                if (unselectedBorder == null) {
                    unselectedBorder = BorderFactory.createMatteBorder(2,3,2,3, table.getBackground());
                }
                setBorder(unselectedBorder);
            }
            if (value != null) {
                setOpaque(true);
                setBackground((Color) value);
                setText("    ");
             } else {
                setOpaque(false);
                setText("None");
            }
            return this;
        }
    }
    
    private class ColorEditor extends AbstractCellEditor implements TableCellEditor {
        
        private final ColorComboBox combo;
        
        ColorEditor() {
            combo = new ColorComboBox(
                    new Color[]{null, Color.RED, Color.GREEN, Color.BLUE, Color.BLACK},
                    new String[]{"None", "Red", "Green", "Blue", "Black"},
                    true);
        }

        @Override
        public Color getCellEditorValue() {
            return combo.getSelectedColor();
        }

        @Override
        public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
            combo.setSelectedColor((Color)value);
            return combo;
        }
        
    }
    
    private class CodeRenderer extends JLabel implements TableCellRenderer {
        Border unselectedBorder;
        Border selectedBorder;
        CodeRenderer() {
            setOpaque(false);
        }
        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            if (isSelected) {
                if (selectedBorder == null) {
                    selectedBorder = BorderFactory.createMatteBorder(2,3,2,3, table.getSelectionBackground());
                }
                setBorder(selectedBorder);
            } else {
                if (unselectedBorder == null) {
                    unselectedBorder = BorderFactory.createMatteBorder(2,3,2,3, table.getBackground());
                }
                setBorder(unselectedBorder);
            }
            if (value == null) {
                setText("");
             } else {
                String code = (String)value;
                setForeground((Boolean)(FilterEditor.this.model.data[row][C_VALID]) ? Color.BLACK : Color.RED);
                setText(code);
            }
            return this;
        }
    }

    
    private class CodeEditor extends AbstractCellEditor implements TableCellEditor {
        
        private String code = "";
        private Target target;
        private Method method;
        private final CodeRenderer button = new CodeRenderer();
        
        CodeEditor() {
            button.addMouseListener(new MouseAdapter() {
                public void mouseClicked(MouseEvent e) {
                    if (e.getClickCount()>1) {
                        Object s = null;
                        StringPanel targetPanel = getTargetPanel();
                        StringPanel patternPanel = getPatternPanel();
                        if (targetPanel == null) {
                            if (method == Method.FILTER) {
                                s = FilterChooser.showDialog(button, plugin.getFilterRegistry());
                            } else {
                                int ok = JOptionPane.showConfirmDialog(button, patternPanel, "Filter definition", JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE);
                                if (ok == JOptionPane.OK_OPTION) {
                                    s = patternPanel.get();
                                }
                            }
                        } else {
                            Box box = Box.createVerticalBox();
                            box.add(targetPanel);
                            box.add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));
                            box.add(patternPanel);
                            box.add(Box.createRigidArea(new Dimension(HSPACE, VSPACE)));
                            int ok = JOptionPane.showConfirmDialog(button, box, "Filter definition", JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE);
                            if (ok == JOptionPane.OK_OPTION) {
                                s = targetPanel.get();
                                if (s != null) {
                                    String p = patternPanel.get();
                                    if (p != null) s = s + TracerFilter.T_P_DELIMETER + p;
                                }
                            }
                        }
                        if (s != null) code = s.toString();
                        fireEditingStopped();
                    }
                }
            });
        }

        @Override
        public String getCellEditorValue() {
            return code;
        }

        @Override
        public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
            code = (String) value;
            target = (Target) FilterEditor.this.model.data[row][C_TARGET];
            method = (Method) FilterEditor.this.model.data[row][C_METHOD];
            return button.getTableCellRendererComponent(table, value, true, true, row, column);
        }
        
        private StringPanel getTargetPanel() {
            switch (target) {
                case TEMPLATE:
                    return new TextFieldPanel("Enter template:", code.split(TracerFilter.T_P_DELIMETER)[0]);
                default:
                    return null;
            }
        }
        
        private StringPanel getPatternPanel() {
            int i = code.indexOf(TracerFilter.T_P_DELIMETER);
            String pattern = i==-1 ? code : code.substring(i+TracerFilter.T_P_DELIMETER.length());
            switch (method) {
                case REGEX:
                    return new TextFieldPanel("Enter regular expression:", pattern);
                case WILDCARD:
                    return new TextFieldPanel("Enter wildcard:", pattern);
                case CONTAINS:
                case EQUALS:
                    return new TextFieldPanel("Enter pattern string:", pattern);
                case CLASS:
                    return new TextFieldPanel("Enter full class name:", pattern);
                case FILTER:
                    return new FilterNamePanel();
                default:
                    return null;
            }
        }
        
        private abstract class StringPanel extends JPanel implements Supplier<String> {
            StringPanel() {
                setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
                setAlignmentX(LEFT_ALIGNMENT);
            }
        }
        
        private class TextFieldPanel extends StringPanel {
            JTextField field;
            TextFieldPanel(String message, String value) {
                setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
                JLabel label = new JLabel(message);
                label.setAlignmentX(LEFT_ALIGNMENT);
                add(label);
                add(Box.createRigidArea(new Dimension(0, VSPACE)));
                field = new JTextField(40);
                field.setAlignmentX(LEFT_ALIGNMENT);
                if (value != null) field.setText(value);
                add(field);
            }
            @Override
            public String get() {
                return field.getText();
            }
        }
        
        private class FilterNamePanel extends StringPanel {
            JButton button;
            String result;
            FilterNamePanel() {
                setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
                button = new JButton("Choose filter");
                button.setMaximumSize(new Dimension(Integer.MAX_VALUE, button.getMaximumSize().height));
                button.setAlignmentX(LEFT_ALIGNMENT);
                button.addActionListener(e -> {
                    String s = FilterChooser.showDialog(button, plugin.getFilterRegistry());
                    if (s != null) result = s;
                });
                add(button);
            }
            @Override
            public String get() {
                return result;
            }
        }
        
    }
    
}
