package org.lsst.ccs.gconsole.plugins.alert;

import java.net.URL;
import java.util.Map;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import org.freehep.application.studio.Studio;
import org.freehep.util.commanddispatcher.CommandProcessor;
import org.freehep.xml.menus.XMLMenuBuilder;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.StatusClearedAlert;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.gconsole.ConsolePlugin;
import org.lsst.ccs.gconsole.plugins.tracer.FilteredMessage;
import org.lsst.ccs.gconsole.plugins.tracer.LsstTracerPlugin;
import org.lsst.ccs.gconsole.plugins.tracer.MessageFilter;
import org.lsst.ccs.gconsole.plugins.tracer.Tracer;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.CommandOriginator;

/**
 * Alert notification plugin.
 *
 * @author onoprien
 */
public class LsstAlertPlugin extends ConsolePlugin {
    
// -- Fields : -----------------------------------------------------------------
    
    private AlertHandler alertHandler;
    private AlertViewer viewer;
    
    private final JFrame emitter = new JFrame("Alert emitter");
    
// -- Lifecycle : --------------------------------------------------------------

    @Override
    synchronized public void init() throws Throwable {
        Studio studio = getConsole().getApplication();
        studio.getCommandTargetManager().add(new CommandProcessor() {
            public void onLsstAlertViewer() {
                onNewAlertViewer();
            }
            public void onLsstAlertTracer() {
                onNewAlertTracer();
            }
        });
        XMLMenuBuilder builder = studio.getXMLMenuBuilder();
        URL xml = getClass().getResource("LsstAlertPlugin.menus");
        builder.build(xml);
    }

    @Override
    synchronized public void postInit() {
        
        alertHandler = new AlertHandler(this);
        getConsole().getLookup().add(alertHandler);
        
        // --- Test: emitting alarms :
                
//        SwingUtilities.invokeLater(() -> {
//            Box treeButtonsPanel = Box.createHorizontalBox();
//            
//            JButton emitAlert = new JButton("Bus Alarm");
//            emitAlert.addActionListener(e -> {
//                String command = JOptionPane.showInputDialog(getConsole().getWindow(), "Enter ID:A|W|N:description", "ID:A:description");
//                String[] tokens = command.split(":");
//                AlertState level = tokens[1].equalsIgnoreCase("A") ? AlertState.ALARM : (tokens[1].equalsIgnoreCase("W") ? AlertState.WARNING : AlertState.NOMINAL);
//                if (level == AlertState.NOMINAL) {
//                    if (tokens[0].isEmpty()) {
//                        getConsole().clearAllAlerts();
//                    } else {
//                        getConsole().clearAlerts(tokens[0]);
//                    }
//                } else {
//                    getConsole().raiseAlert(new Alert(tokens[0], tokens[2]), level);
//                }
//            });
//            treeButtonsPanel.add(emitAlert);
//
//            emitAlert = new JButton("Local Alarm");
//            emitAlert.addActionListener(e -> {
//                String command = JOptionPane.showInputDialog(emitter, "Enter Source:ID:A|W|N:description", "Source:ID:W:description");
//                String[] tokens = command.split(":");
//                AlertState level = tokens[2].equalsIgnoreCase("A") ? AlertState.ALARM : (tokens[2].equalsIgnoreCase("W") ? AlertState.WARNING : AlertState.NOMINAL);
//                alertHandler.submitAlert(tokens[0], new Alert(tokens[1], tokens[3]), level);
//            });
//            treeButtonsPanel.add(emitAlert);
//            
//            emitter.add(treeButtonsPanel);
//            emitter.pack();
//            emitter.setVisible(true);
//        });
        
        // --- end Test: emitting alarms
    }

    @Override
    synchronized public void applicationVisible() {
//        onNewAlertViewer();
    }

    @Override
    synchronized public void shutdown() {
        if (alertHandler != null) {
            getConsole().getLookup().remove(alertHandler);
            alertHandler = null;
        }
    }
    
    
// -- Operations : -------------------------------------------------------------
    
    public void onNewAlertViewer() {
        if (viewer == null) {
            viewer = new AlertViewer(this);
            alertHandler.addListener(viewer);
        } else {
            viewer.ensureVisible();
        }
    }
    
    public void stopAlertViewer() {
        alertHandler.removeListener(viewer);
        viewer = null;
    }
    
    public void onNewAlertTracer() {
        LsstTracerPlugin tracerPlugin = (LsstTracerPlugin) getConsole().getLookup().lookup(LsstTracerPlugin.class);
        Tracer tracer = tracerPlugin.createTracer("Alerts", "BuiltIn/Alerts");
        tracer.setFilter(filteredMessage -> {
            BusMessage bm = filteredMessage.getBusMessage();
            if (bm instanceof StatusRaisedAlert) {
                StatusRaisedAlert am = (StatusRaisedAlert) bm;
                Alert alert = am.getRaisedAlert();
                String id = alert.getAlertId();
                StringBuilder sb = new StringBuilder();
                sb.append(AlertViewer.formatTimeStamp(am.getTimeStamp())).append(" : ");
                sb.append(am.getRaisedAlertSummary().getRaisedAlert(id).getLatestAlertState());
                sb.append(" from ").append(am.getOriginAgentInfo().getName()).append(".\n");
                sb.append("ID: ").append(id).append(".  Description:\n");
                sb.append(alert.getDescription()).append("\n");
                filteredMessage.setMessage(sb.toString());
                return filteredMessage;
            } else if (bm instanceof StatusClearedAlert) {
                StatusClearedAlert am = (StatusClearedAlert) bm;
                StringBuilder sb = new StringBuilder();
                sb.append(AlertViewer.formatTimeStamp(am.getTimeStamp())).append(" : ");
                sb.append(am.getOriginAgentInfo().getName()).append(" cleared alerts ");
                for (String id : am.getClearAlertIds()) {
                    sb.append(id).append(", ");
                }
                sb.delete(sb.length()-2, sb.length()).append(".\n");
                filteredMessage.setMessage(sb.toString());
                return filteredMessage;
            } else {
                return null;
            }
        });
    }
    
    public void clearAlarms(Map<String,String[]> alarms) {
        AgentMessagingLayer messenger = getConsole().getMessagingAccess();
        CommandOriginator originator = new CommandOriginator() {
            public void processAck(CommandAck ack) {}
            public void processResult(CommandResult result) {}
            public void processNack(CommandNack nack) {}
        };
        alarms.forEach((source, ids) -> {
            CommandRequest request;
            if (ids == null) {
                request = new CommandRequest(source, "clearAllAlerts");
            } else {
                request = new CommandRequest(source, "clearAlerts", (Object)ids);
            }
            messenger.sendCommandRequest(request, originator);
        });
    }
    
}
