package org.lsst.ccs.gconsole.plugins.alert;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.lang.reflect.InvocationTargetException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Set;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.border.Border;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.states.AlertState;

/**
 * Display of CCS alert summary.
 *
 * @author onoprien
 */
public class AlertViewer implements AlertListener {

// -- Fields : -----------------------------------------------------------------
    
    private final LsstAlertPlugin plugin;
    
    private Box view;
    private JFrame frame;
    
    private Border viewBorder;
    private AlertTree tree;
    private JEditorPane infoPanel;
    
    private Action clearAction;
    private Action muteAction;
    private Action ignoreAction;
    private Action historyAction;
    
    static private final int HSPACE = 10;
    static private final int VSPACE = 5;
    static private final EnumMap<AlertState,String> ALCOLOR = new EnumMap<>(AlertState.class);
    {
        ALCOLOR.put(AlertState.NOMINAL, "#00AA00");
        ALCOLOR.put(AlertState.WARNING, "#0000AA");
        ALCOLOR.put(AlertState.ALARM, "#AA0000");
    }

// -- Life cycle : -------------------------------------------------------------
    
    public AlertViewer(LsstAlertPlugin plugin) {
        this.plugin = plugin;
    }
    
    /** Wake up and get ready to display alerts. */
    private void start() {
                
        clearAction = new AbstractAction("Clear") {
            @Override
            public void actionPerformed(ActionEvent e) {
                AlertNode node = tree.getSelectedNode();
                if (node == null) return;
                HashMap<String, String[]> alerts = getAlertMap(node);
                new SwingWorker<Void, Void>() {
                    @Override
                    protected Void doInBackground() throws Exception {
                        plugin.clearAlarms(alerts);
                        return null;
                    }
                }.execute();
            }
        };
        muteAction = new AbstractAction("Mute") {
            @Override
            public void actionPerformed(ActionEvent e) {
                AlertNode node = tree.getSelectedNode();
                if (node != null) {
                    node.setMute(!node.isMute());
                }
                
            }
        };
        ignoreAction = new AbstractAction("Ignore") {
            @Override
            public void actionPerformed(ActionEvent e) {
                AlertNode node = tree.getSelectedNode();
                if (node != null) {
                    node.setIgnore(!node.isIgnore());
                }
            }
        };
        historyAction = new AbstractAction("History") {
            @Override
            public void actionPerformed(ActionEvent e) {
                System.out.println("History action");
            }
        };
        
        Box buttonPanel = Box.createHorizontalBox();
        buttonPanel.add(Box.createRigidArea(new Dimension(HSPACE, VSPACE)));
//        buttonPanel.add(Box.createHorizontalGlue());
        buttonPanel.add(new JButton(clearAction));
        buttonPanel.add(Box.createRigidArea(new Dimension(3*HSPACE, VSPACE)));
        buttonPanel.add(new JButton(muteAction));
        buttonPanel.add(Box.createRigidArea(new Dimension(HSPACE, VSPACE)));
        buttonPanel.add(new JButton(ignoreAction));
        buttonPanel.add(Box.createRigidArea(new Dimension(3*HSPACE, VSPACE)));
        buttonPanel.add(Box.createHorizontalGlue());
        buttonPanel.add(new JButton(historyAction));
        buttonPanel.add(Box.createRigidArea(new Dimension(HSPACE, VSPACE)));
        buttonPanel.setBorder(BorderFactory.createEmptyBorder(VSPACE, VSPACE, VSPACE, VSPACE));
    
        tree = new AlertTree();
        tree.addTreeSelectionListener(e -> {
            AlertNode node = tree.getSelectedNode();
            if (node == null) { // nothing selected
                clearAction.setEnabled(false);
                muteAction.setEnabled(false);
                ignoreAction.setEnabled(false);
                historyAction.setEnabled(false);
                infoPanel.setText("");
            } else {
                clearAction.setEnabled(true);
                muteAction.setEnabled(node.getWatchState() != WatchState.IGNORE);
                infoPanel.setText(node.getInfo());
                if (node.isRoot()) { // root
                    ignoreAction.setEnabled(true);
                    historyAction.setEnabled(false);
                } else {
                    ignoreAction.setEnabled(node.getParent().getWatchState() != WatchState.IGNORE);
                    historyAction.setEnabled(node.getLeafCount() == 1);
                }
            }
        });
        tree.setBorder(BorderFactory.createEmptyBorder(VSPACE, VSPACE, VSPACE, VSPACE));
    
        infoPanel = new JEditorPane();
        infoPanel.setEditable(false);
        infoPanel.setContentType("text/html");
        infoPanel.setBorder(BorderFactory.createEmptyBorder(VSPACE, VSPACE, VSPACE, VSPACE));
        infoPanel.setText("");
        
        JPanel rightPanel = new JPanel(new BorderLayout());
        rightPanel.add(new JScrollPane(infoPanel), BorderLayout.CENTER);
        rightPanel.add(buttonPanel, BorderLayout.SOUTH);
        
        view = Box.createVerticalBox();
        viewBorder = BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE);
  //      view.setBorder(viewBorder);
        JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, new JScrollPane(tree), rightPanel);
        view.add(splitPane);
        view.setPreferredSize(new Dimension(700,700));
    }

    /** Release resources and go to sleep. */
    private void stop() {
        
        if (frame != null) {
            frame.dispose();
            frame = null;
        }
        
        clearAction = null;
        muteAction = null;
        ignoreAction = null;
        historyAction = null;
        tree = null;
        infoPanel = null;
        view = null;
    }


// -- Processing alert events --------------------------------------------------
    
    @Override
    public void onAlert(AlertEvent event) {
        if (SwingUtilities.isEventDispatchThread()) {
            update(event);
        } else {
            try {
                SwingUtilities.invokeAndWait(() -> update(event));
            } catch (InterruptedException|InvocationTargetException x) {
            }
        }
    }

    public void update(AlertEvent event) {
        if (view == null) start();
        
        String source = event.getSource();
        RaisedAlertSummary summary = event.getSummary();
        
        Set<RaisedAlertHistory> currentAlerts = summary.getAllRaisedAlertHistories();
        AlertNode sourceNode = getSourceNode(source);
        if (sourceNode == null) {
            sourceNode = new AlertNode(source);
            for (RaisedAlertHistory h : currentAlerts) {
                updateAlert(sourceNode, h);
            }
            tree.getRoot().add(sourceNode);
        } else {
            HashMap<String,RaisedAlertHistory> id2history = new HashMap<>(currentAlerts.size()*2);
            currentAlerts.forEach(h -> id2history.put(h.getLatestAlert().getAlertId(), h));
            for (IDNode idNode : sourceNode.getLeaves()) {
                    String id = idNode.getID();
                    RaisedAlertHistory history = id2history.remove(id);
                    if (history == null) {
                        removeAlert(idNode);
                    } else {
                        if (getLatestAlertTimestamp(history).compareTo(idNode.getTimeStamp()) > 0) {
                            updateAlert(sourceNode, history);
                        }
                    }
            }
            for (RaisedAlertHistory h : id2history.values()) {
                updateAlert(sourceNode, h);
            }
        }
        
        tree.getModel().reload();
        TreePath path;
        Alert alert = event.getAlert();
        if (alert == null) {
            path = new TreePath(new Object[] {tree.getRoot(), sourceNode});
        } else {
            String id = alert.getAlertId();
            IDNode idNode = tree.findID(id);
            if (idNode == null) {
                path = new TreePath(new Object[] {tree.getRoot(), sourceNode});
            } else {
                path = new TreePath(idNode.getPath());
            }
        }
        tree.setSelectionPath(path);
        tree.expandPath(path);
        
        ensureVisible();
    }
    
    public void ensureVisible() {
        if (frame == null) {
            frame = new JFrame("Alert Viewer");
            frame.addWindowListener(new WindowAdapter() {
                @Override
                public void windowClosed(WindowEvent e) {
                    stop();
                }
            });
            frame.add(view);
            frame.pack();
        }
//        if (!view.isShowing()) {
//            view.setVisible(true);
            frame.setVisible(true);
//        }
        
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private AlertNode getSourceNode(String source) {
        AlertNode root = tree.getRoot();
        for (int i=0; i<root.getChildCount(); i++) {
            AlertNode node = root.getChildAt(i);
            if (node.getUserObject().equals(source)) return node;
        }
        return null;
    }
    
    private HashMap<String,String[]> getAlertMap(AlertNode node) {
        HashMap<String, String[]> alerts = new HashMap<>();
        if (node instanceof IDNode) {
            IDNode idNode = (IDNode) node;
            alerts.put(node.getSource(), new String[]{idNode.getID()});
        } else if (node.isRoot()) {
            Enumeration e = node.children();
            while (e.hasMoreElements()) {
                AlertNode sourceNode = (AlertNode) e.nextElement();
                alerts.put(sourceNode.getSource(), null);
            }
        } else if (node.getLevel() == 1) {
            alerts.put(node.getSource(), null);
        } else {
            String source = node.getSource();
            ArrayList<String> ids = new ArrayList<>();
            Enumeration e = node.depthFirstEnumeration();
            while (e.hasMoreElements()) {
                Object o = e.nextElement();
                if (o instanceof IDNode) {
                    IDNode idNode = (IDNode) o;
                    ids.add(idNode.getID());
                }
            }
            alerts.put(source, ids.toArray(new String[0]));
        }
        return alerts;
    }
    
    static public String formatTimeStamp(long millis) {
        Instant instant = Instant.ofEpochMilli(millis);
        return formatTimeStamp(instant);
    }
    
    static public String formatTimeStamp(Instant instant) {
        DateTimeFormatter formatter = DateTimeFormatter.ofLocalizedDateTime(FormatStyle.SHORT).withZone(ZoneId.systemDefault());
        return formatter.format(instant);
    }
        
    static private String toColoredString(String text, AlertState level) {
        if (text == null) text = level.toString();
        String color = ALCOLOR.get(level);
        return "<font color="+ color +">"+ text +"</font>";
    }
    
    private void updateAlert(AlertNode sourceNode, RaisedAlertHistory history) {
        
        String id = history.getLatestAlert().getAlertId();
        String[] ss = id.split("/");
        ArrayList<String> tokens = new ArrayList<>(ss.length);
        for (String s : ss) {
            s = s.trim();
            if (!s.isEmpty()) tokens.add(s);
        }
        if (tokens.isEmpty()) return;
        
        for (int i=0; i<tokens.size()-1; i++) {
            String s = tokens.get(i);
            AlertNode tokenNode = null;
            Enumeration e = sourceNode.children();
            while (e.hasMoreElements()) {
                AlertNode node = (AlertNode) e.nextElement();
                if (!node.isLeaf()) {
                    if (node.getUserObject().equals(s)) {
                        tokenNode = node;
                        break;
                    }
                }
            }
            if (tokenNode == null) {
                tokenNode = new AlertNode(s);
                sourceNode.add(tokenNode);
            }
            sourceNode = tokenNode;
        }
        
        IDNode idNode = null;
        Enumeration e = sourceNode.children();
        while (e.hasMoreElements()) {
            Object o = e.nextElement();
            if (o instanceof IDNode) {
                IDNode node = (IDNode) o;
                if (node.getID().equals(id)) {
                    idNode = node;
                    break;
                }
            }
        }
        
        if (idNode == null) {
            idNode = new IDNode(history);
            sourceNode.add(idNode);
        } else {
            idNode.setHistory(history);
        }
        
    }
    
    private void removeAlert(IDNode idNode) {
        AlertNode node = idNode;
        while (true) {
            AlertNode parent = node.getParent();
            if (parent == null) return;
            if (parent.isRoot() || parent.getChildCount() > 1) {
                node.removeFromParent();
                return;
            } else {
                node = parent;
            }
        }
    }
    
    // FIXME: remove once history uses Instant time stamps.
    static private Instant getLatestAlertTimestamp(RaisedAlertHistory history) {
        return Instant.ofEpochMilli(history.getLatestAlertTimestamp());
    }


// -- Alert tree utilities : ---------------------------------------------------
    
    static private enum WatchState {ACTIVE, MUTE, IGNORE}
    
    private class AlertTree extends JTree {
        
        AlertTree() {
            super(new AlertTreeModel());
            getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
            setCellRenderer(new AlertTreeRenderer());
        }
        
        public AlertNode getSelectedNode() {
            return (AlertNode) getLastSelectedPathComponent();
        }
        
        @Override
        public AlertTreeModel getModel() {
            return (AlertTreeModel) super.getModel();
        }
        
        public AlertNode getRoot() {
            return getModel().getRoot();
        }
        
        public IDNode findID(String id) {
            Enumeration e = getRoot().depthFirstEnumeration();
            while (e.hasMoreElements()) {
                Object node = e.nextElement();
                if (node instanceof IDNode && ((IDNode)node).getID().equals(id))  {
                    return (IDNode)node;
                }
            }
            return null;
        }

//        @Override
//        public String convertValueToText(Object value, boolean selected, boolean expanded, boolean leaf, int row, boolean hasFocus) {
//            return super.convertValueToText(value, selected, expanded, leaf, row, hasFocus);
//        }
        
    }
    
    private class AlertTreeModel extends DefaultTreeModel {
        
        AlertTreeModel() {
            super(new AlertNode("CCS Alerts") {
                
                @Override
                WatchState getWatchState() {
                    return isIgnore() ? WatchState.IGNORE : (isMute() ? WatchState.MUTE : WatchState.ACTIVE);
                }
                
                @Override
                String getInfo() {
                    StringBuilder sb = new StringBuilder();
                    sb.append("<html>CCS Control System</b><p>");
                    sb.append("Subsystems with active alerts: <b>").append(getChildCount()).append("</b><br>");
                    sb.append("Overall alert level: <b>").append(getAlertLevel()).append("</b><br>");
                    sb.append("Last changed at <b>").append(formatTimeStamp(getTimeStamp())).append("</b></html>");
                    return sb.toString();
                }

                @Override
                String getSource() {
                    throw new UnsupportedOperationException("Trying to look up source subsystem for the root node.");
                }
                
            });
        }
        
        @Override
        public AlertNode getRoot() {
            return (AlertNode) super.getRoot();
        }
        
    }
    
    /** Mutable tree node that contains an instance of AlertNodeContent as its user object. */
    static class AlertNode extends DefaultMutableTreeNode implements Comparable<AlertNode> {
        
        private boolean mute;
        private boolean ignore;
        
        AlertNode(String userObject) {
            super(userObject);
        }

        @Override
        public String getUserObject() {
            return (String) super.getUserObject();
        }

        @Override
        public int compareTo(AlertNode other) {
            return getUserObject().compareTo(other.getUserObject());
        }
        
        @Override
        public void add(MutableTreeNode newChildNode) {
            AlertNode newChild = (AlertNode) newChildNode;
            Enumeration<TreeNode> e = children();
            int i = 0;
            boolean replace = false;
            while (e.hasMoreElements()) {
                AlertNode child = (AlertNode) e.nextElement();
                int c = newChild.compareTo(child);
                if (c > 0) {
                    i++;
                } else {
                    if (c == 0) replace = true;
                    break;
                }
            }
            if (replace) remove(i);
            insert(newChild, i);
        }

        @Override
        public AlertNode getParent() {
            return (AlertNode) super.getParent();
        }

        @Override
        public AlertNode getChildAt(int index) {
            return (AlertNode) super.getChildAt(index);
        }
        
        public void forEachChild(Consumer<? super AlertNode> action) {
            if (children == null) return;
            children.forEach(action);
        }
        
        public ArrayList<IDNode> getLeaves() {
            ArrayList<IDNode> out = new ArrayList<>(this.getLeafCount());
            Enumeration e = this.depthFirstEnumeration();
            while (e.hasMoreElements()) {
                Object node = e.nextElement();
                if (node instanceof IDNode) {
                    out.add((IDNode)node);
                }
            }
            return out;
        }
        
        String getInfo() {
            StringBuilder sb = new StringBuilder();
            sb.append("<html>Source: <b>").append(getSource()).append("</b><br>");
            sb.append("Alerts selected: <b>").append(getLeafCount()).append("</b><br>");
            sb.append("Highest alert level: <b>").append(toColoredString(null, getAlertLevel())).append("</b><br>");
            sb.append("Last changed at <b>").append(formatTimeStamp(getTimeStamp())).append("</b></html>");
            return sb.toString();
        }

        String getSource() {
            int level = getLevel();
            if (level > 1) {
                return getParent().getSource();
            } else {
                return getUserObject();
            }
        }
        
        Instant getTimeStamp() {
            Instant out = Instant.EPOCH;
            for (Object child : children) {
                Instant time = ((AlertNode)child).getTimeStamp();
                if (time.compareTo(out) > 0) out = time;
            }
            return out;
        }
        
        AlertState getAlertLevel() {
            AlertState alertState = AlertState.NOMINAL;
            Enumeration e = this.children();
            while (e.hasMoreElements()) {
                AlertState childState = ((AlertNode) e.nextElement()).getAlertLevel();
                if (childState.compareTo(alertState) > 0) {
                    alertState = childState;
                    if (alertState.equals(AlertState.ALARM)) break;
                }
            }
            return alertState;
        }
        
        WatchState getWatchState() {
            WatchState thisState = isIgnore() ? WatchState.IGNORE : (isMute() ? WatchState.MUTE : WatchState.ACTIVE);
            if (thisState == WatchState.IGNORE) return WatchState.IGNORE;
            WatchState parentState = this.getParent().getWatchState();
            return thisState.compareTo(parentState) == -1 ? parentState : thisState;
        }
        
        boolean isMute() {
            return mute;
        }
        
        void setMute(boolean isMute) {
            mute = isMute;
        }
        
        boolean isIgnore() {
            return ignore;
        }
        
        void setIgnore(boolean isIgnored) {
            ignore = isIgnored;
        }

        @Override
        public String toString() {
            return "<html>"+ toColoredString(super.toString(), getAlertLevel()) +"</html>";
        }
    }
    
    static final class IDNode extends AlertNode {
        
        private RaisedAlertHistory history;
        
        IDNode(RaisedAlertHistory history) {
            super(history.getLatestAlert().getAlertId());
            setHistory(history);
        }

        void setHistory(RaisedAlertHistory history) {
            this.history = history;
        }
        
        RaisedAlertHistory getHistory() {
            return history;
        }
        
        String getID() {
            return getUserObject();
        }

        @Override
        public String toString() {
            return "<html>"+ toColoredString(getLastIdComponent(), getAlertLevel()) +"</html>";
        }
        
        String getLastIdComponent() {
            String[] ss = getID().split("/");
            int i = ss.length;
            while (i > 0) {
                String s = ss[--i].trim();
                if (!s.isEmpty()) return s;
            }
            return getID();
        }
        
        @Override
        String getInfo() {
            Alert alert = history.getLatestAlert();
            StringBuilder sb = new StringBuilder();
            sb.append("<html>Source: <b>").append(getSource()).append("</b><br>");
            sb.append("Alert ID: <b>").append(alert.getAlertId()).append("</b><br>");
            sb.append("Level at <b>").append(formatTimeStamp(history.getLatestAlertTimestamp())).append("</b> : <b>").append(toColoredString(null, history.getLatestAlertState())).append("<b><br>");
            sb.append("Highest level: ").append(toColoredString(null, history.getHighestAlertState()));
            sb.append("<p><b>Description:</b><p>").append(alert.getDescription()).append("</html>");
            return sb.toString();
        }
        
        @Override
        Instant getTimeStamp() {
            return getLatestAlertTimestamp(history);
        }
        
        @Override
        AlertState getAlertLevel() {
            return history.getHighestAlertState();
        }
        
    }
    
    private class AlertTreeRenderer extends DefaultTreeCellRenderer {

        @Override
        public Component getTreeCellRendererComponent(JTree tree, Object value, boolean sel, boolean expanded, boolean leaf, int row, boolean hasFocus) {
            return super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);
        }
        
    }
    
}
