package org.lsst.ccs.gconsole.plugins.trending;

import com.sun.jersey.api.client.WebResource;
import java.awt.Component;
import java.util.List;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import org.freehep.jas.plugin.tree.DefaultFTreeNodeAdapter;
import org.freehep.jas.plugin.tree.FTreeNode;
import org.freehep.jas.plugin.tree.FTreeNodeTransferable;
import org.freehep.jas.services.PlotRegionDropHandler;
import org.freehep.jas.services.Plotter;
import org.freehep.util.commanddispatcher.CommandProcessor;
import org.freehep.util.commanddispatcher.CommandSourceAdapter;

/**
 * Defines behavior for tree nodes corresponding to trending channels.
 * 
 * @author turri
 * @author onoprien
 */
final class DataChannelTreeAdapter extends DefaultFTreeNodeAdapter {
    
// -- Private parts : ----------------------------------------------------------

    private final LsstTrendingPlugin plugin;
    private final WebResource service;
    private final ChannelDropHandler dropHandler;
    
    private final Commands commands = new Commands();
    private final Plotter fakePlotter = new DummyPlotter();
    

// -- Construction : -----------------------------------------------------------

    DataChannelTreeAdapter(LsstTrendingPlugin plugin, WebResource service) {
        super(100);
        this.plugin = plugin;
        this.service = service;
        dropHandler = new ChannelDropHandler(plugin);
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    /** Returns the trending plugin singleton. */
    public LsstTrendingPlugin getPlugin() {
        return plugin;
    }
    
    /** Returns web resource associated with this trending data channels tree. */
    public WebResource getResource() {
        return service;
    }
    
    
// -- Implementing FTreeNodeAdapter : ------------------------------------------

    @Override
    public CommandProcessor commandProcessor(FTreeNode[] selectedNodes) {
        commands.setSelectedNodes(selectedNodes);
        return commands;
    }

    @Override
    public JPopupMenu modifyPopupMenu(FTreeNode[] nodes, JPopupMenu menu) {
        commands.setSelectedNodes(nodes);
        if (menu.getSubElements().length != 0) {
            menu.addSeparator();
        }
        JMenuItem item = new JMenuItem("Show");
        item.setToolTipText("Plot in the region previously used for this channel, if any, or on a new page.");
        plugin.getApplication().getCommandTargetManager().add(new CommandSourceAdapter(item));
        menu.add(item);
        item = new JMenuItem("Plot");
        item.setToolTipText("Plot in the currently selected region.");
        plugin.getApplication().getCommandTargetManager().add(new CommandSourceAdapter(item));
        menu.add(item);
        item = new JMenuItem("Overlay");
        item.setToolTipText("Overlay on the currently selected plot.");
        plugin.getApplication().getCommandTargetManager().add(new CommandSourceAdapter(item));
        menu.add(item);
        item = new JMenuItem("Add");
        plugin.getApplication().getCommandTargetManager().add(new CommandSourceAdapter(item));
        menu.add(item);
        item = new JMenuItem("New Plot");
        item.setToolTipText("Plot in a new region.");
        plugin.getApplication().getCommandTargetManager().add(new CommandSourceAdapter(item));
        menu.add(item);
        item = new JMenuItem("New Page");
        item.setToolTipText("Plot on a new page.");
        plugin.getApplication().getCommandTargetManager().add(new CommandSourceAdapter(item));
        menu.add(item);
        item = new JMenuItem("Refresh");
        item.setToolTipText("Refresh plots for this channel.");
        plugin.getApplication().getCommandTargetManager().add(new CommandSourceAdapter(item));
        menu.add(item);
        menu.addSeparator();
        return menu;
    }

    @Override
    public boolean doubleClick(FTreeNode node) {
        commands.setSelectedNodes(new FTreeNode[]{node});
        switch (plugin.getPreferences().doubleClick()) {
            case -1:
                commands.onShow(); break;
            case -2:
                commands.onPlot(); break;
            case -3:
                commands.onNewPlot(); break;
            case -4:
                commands.onNewPage(); break;
            default:
                commands.onShow();
        }
        return true;
    }

    @Override
    public FTreeNodeTransferable modifyTransferable(FTreeNode[] nodes, FTreeNodeTransferable transferable) {
        transferable.addDataForClass(PlotRegionDropHandler.class, dropHandler);
        transferable.addDataForClass(Plotter.class, fakePlotter);
        return transferable;
    }
    
    
// -- Command processor class : ------------------------------------------------

    public class Commands extends CommandProcessor {

        private volatile FTreeNode[] selectedNodes;

        void setSelectedNodes(FTreeNode[] selectedNodes) {
            this.selectedNodes = selectedNodes;
        }

        public void onShow() {
            DataChannelHandler chan = (DataChannelHandler) selectedNodes[0].objectForClass(DataChannelHandler.class);
            if (chan == null) return;
            plugin.getPlotMaker().plot(chan, null, true, true, false, false);
        }

        public void onOverlay() {
            DataChannelHandler chan = (DataChannelHandler) selectedNodes[0].objectForClass(DataChannelHandler.class);
            if (chan == null) return;
            plugin.getPlotMaker().plot(chan, null, false, false, false, true);
        }

        public void onNewPlot() {
            DataChannelHandler chan = (DataChannelHandler) selectedNodes[0].objectForClass(DataChannelHandler.class);
            if (chan == null) return;
            plugin.getPlotMaker().plot(chan, null, false, false, true, false);
        }

        public void onPlot() {
            DataChannelHandler chan = (DataChannelHandler) selectedNodes[0].objectForClass(DataChannelHandler.class);
            if (chan == null) return;
            plugin.getPlotMaker().plot(chan, null, false, false, false, false);
        }

        public void onNewPage() {
            DataChannelHandler chan = (DataChannelHandler) selectedNodes[0].objectForClass(DataChannelHandler.class);
            if (chan == null) return;
            plugin.getPlotMaker().plot(chan, null, false, true, false, false);
        }

        public void onRefresh() {
            DataChannelHandler chan = (DataChannelHandler) selectedNodes[0].objectForClass(DataChannelHandler.class);
            if (chan == null) return;
            plugin.getPlotMaker().refresh(chan, null);
        }

    }
    

// -- Dummy plotter for hoodwinking DefaultPage.DefaultRegion into accepting drop : ---
    
    public class DummyPlotter implements Plotter {

        @Override
        public void plot(Object data, int mode) {
        }

        @Override
        public void plot(Object data, int mode, Object style, String options) {
        }

        @Override
        public void remove(Object data) {
        }

        @Override
        public void clear() {
        }

        @Override
        public Component viewable() {
            return null;
        }

        @Override
        public List getData() {
            return null;
        }
        
    }
}
