package org.lsst.ccs.plugin.jas3.trending;

import hep.aida.IAxisStyle;
import hep.aida.IBaseHistogram;
import hep.aida.IDataStyle;
import hep.aida.IPlotterStyle;
import hep.aida.ref.plotter.PlotterFactory;
import java.awt.Component;
import java.awt.Point;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;
import javax.swing.SwingUtilities;
import org.freehep.jas.plugin.plotter.DefaultRegion;
import org.freehep.jas.services.PlotFactory;
import org.freehep.jas.services.PlotPage;
import org.freehep.jas.services.PlotRegion;
import org.freehep.jas.services.Plotter;
import org.freehep.swing.popup.HasPopupItems;
import org.lsst.ccs.plugin.jas3.trending.timeselection.TimeWindow;
import static org.lsst.ccs.plugin.jas3.trending.PlotMaker.Options.*;

/**
 * Machinery for making and updating plots.
 * <p>
 * Public methods of this class can be called from any thread, the actual plotting is 
 * done on the AWT event processing thread. If called on the event processing thread, these methods
 * execute synchronously and return when the plotting is finished. Otherwise, they prepare the data
 * for plotting, schedule a plotting task on the event processing thread, and return immediately.
 * 
 * @author onoprien
 */
public class PlotMaker {
    
// -- Private parts : ----------------------------------------------------------
    
    private final LsstTrendingPlugin plugin;
    private final PlotterFactory plotterStyleFactory;
    private final EnumMap<AuxData.Type,IPlotterStyle> auxStyles;
    
    private PlotFactory plotFactory;
    
    private ThreadPoolExecutor executor;
    private final int DATA_FETCHER_MAX_THREADS = 3;
    private final int DATA_FETCHER_KEEP_ALIVE = 120;
    
    
// -- Refresh options : --------------------------------------------------------
    
    public enum Options {REFRESH_ALL, REFRESH_NONE, FORCE_REPLOT}
    
// -- Construction and initialization : ----------------------------------------
    
    PlotMaker(LsstTrendingPlugin trendingPlugin) {
        plugin = trendingPlugin;
        plotterStyleFactory = new PlotterFactory();
        auxStyles = new EnumMap<>(AuxData.Type.class);
        
        IPlotterStyle style = plotterStyleFactory.createPlotterStyle();
        IDataStyle dataStyle = style.dataStyle();
        dataStyle.showInLegendBox(false);
        dataStyle.showInStatisticsBox(false);
        dataStyle.outlineStyle().setParameter("color","red");
        dataStyle.outlineStyle().setLineType("dashed");
        dataStyle.markerStyle().setVisible(false);
        auxStyles.put(AuxData.Type.ALARM, style);
        
        style = plotterStyleFactory.createPlotterStyle();
        dataStyle = style.dataStyle();
        dataStyle.showInLegendBox(false);
        dataStyle.showInStatisticsBox(false);
        dataStyle.outlineStyle().setParameter("color","orange");
        dataStyle.outlineStyle().setLineType("dashed");
        dataStyle.markerStyle().setVisible(false);
        auxStyles.put(AuxData.Type.WARNING, style);

        ThreadFactory tFactory = new ThreadFactory() {
            ThreadFactory delegate = Executors.defaultThreadFactory();
            int id = 0;
            public Thread newThread(Runnable r) {
                Thread thread = delegate.newThread(r);
                thread.setName("Trending Data Fetcher " + id++);
                return thread;
            }
        };
        executor = new ThreadPoolExecutor(DATA_FETCHER_MAX_THREADS, DATA_FETCHER_MAX_THREADS, 
                                          DATA_FETCHER_KEEP_ALIVE, TimeUnit.SECONDS, 
                                          new LinkedBlockingQueue<>(), tFactory);
        executor.allowCoreThreadTimeOut(true);
    }

    
// -- Operations : -------------------------------------------------------------
    
    /**
     * Plots time history for the specified channel.
     * 
     * @param channel Data channel to be plotted.
     * @param timeWindow Time window. If <tt>null</tt>, currently selected time window is used.
     * @param useExistingPlot If <tt>true</tt>, the new plot will replace an existing plot for the same channel if possible.
     *                        See {@link DataChannelHandler#findPlot DataChannelHandler.findPlot(...)} for details on how the plot to be replaced is chosen. 
     * @param newPage If <tt>true</tt>, the plot will be shown in a new page unless an existing plot is being replaced.
     * @param newPlot If <tt>true</tt>, the plot will be shown in a new region on the current page unless an existing plot is being replaced.
     * @param overlay If <tt>true</tt>, the plot will be overlaid on top of the current region.
     */
    public void plot(DataChannelHandler channel, TimeWindow timeWindow, boolean useExistingPlot, boolean newPage, boolean newPlot, boolean overlay) {
        
        if (channel == null) return;
        
        if (plotFactory == null) {
            plotFactory = (PlotFactory) plugin.getApplication().getLookup().lookup(PlotFactory.class);
            if (plotFactory == null) {
                plugin.getApplication().error("No plot factory available");
            }
        }
        
        // fetch currently selected time window in none is specified
        
        if (timeWindow == null) {
            timeWindow = plugin.getTimeWindowSelector().getSelectedTimeWindow();
            if (timeWindow == null) return;
        }
        
        // if we're on event processing thread, move to executor : -------------
        
        if (SwingUtilities.isEventDispatchThread()) {
            TimeWindow tw = timeWindow;
//            System.out.println("plot(...) is moving to executor");
            executor.execute(() -> {
                plot(channel, tw, useExistingPlot, newPage, newPlot, overlay);
            });
            return;
        }
        
        // if using existing plot for the specified channel is requested, find it
 
        PlotRegion region = null;
        PlotPage page = null;
        Plot existingPlot = null;
        if (useExistingPlot) {
            existingPlot = channel.findPlot(timeWindow);
            if (existingPlot != null) {
                page = existingPlot.getPage();
                region = existingPlot.getRegion();
            }
        }
        
        // find or create target page
        
        if (page == null) {
            if (!newPage) {
                page = plotFactory.currentPage();
            }
            if (page == null) {
                page = plotFactory.createPage(channel.getPath());
                if (page == null) {
                    plugin.getApplication().error("Unable to create a plot page");
                    return;
                } else {
                    page.createRegions(1, 1);
                    region = page.currentRegion();
                }
            }
        }
        
        // find or create target region
        
        if (region == null) {
            if (newPlot) {
                int n = page.numberOfRegions();
                for (int i=0; i<n; i++) {
                    PlotRegion r = page.region(i);
                    if (r.currentPlot() == null) {
                        region = r;
                        break;
                    }
                }
            } else {
                region = page.currentRegion();
            }
            if (region == null) {
                region = page.addRegion();
            }
        }
        
        // create data point set

        Plot plot = channel.makePlot(timeWindow);
        plot.setVisible(AuxData.Type.ALARM, plugin.getPreferences().drawAlarm());
        plot.setVisible(AuxData.Type.WARNING, plugin.getPreferences().drawAlarm());
        
        plot.setPage(page);
        plot.setRegion(region);
        if (existingPlot != null) {
            plot.setVisible(existingPlot);
        }
        
        // make region current and show page
        
        page.setCurrentRegion(region);
        page.showPage();
        
        // plot
        
        plotData(Collections.singletonList(plot), page, region, overlay);
    }
    
    /**
     * Refreshes all regions where the specified channel data is plotted.
     * 
     * @param channel Data channel.
     * @param timeWindow Time window for the new plots. 
     *                   If <tt>null</tt>, time windows already associated with plots being refreshed are used.
     */
    public void refresh(DataChannelHandler channel, TimeWindow timeWindow) {
        getRegions(channel).forEach(region -> refresh(region, timeWindow));
    }
    
    /**
     * Refreshes all regions on the specified page.
     * 
     * @param page Plot page.
     * @param timeWindow Time window for the new plots. 
     *                   If <tt>null</tt>, time windows already associated with plots being refreshed are used.
     */
    public void refresh(PlotPage page, TimeWindow timeWindow) {
        int n = page.numberOfRegions();
        try {
            for (int i = 0; i < n; i++) {
                refresh(page.region(i), timeWindow);
            }
        } catch (IndexOutOfBoundsException x) {
            // can happen if page is modified while we're refreshing it
        }
    }
    
    /**
     * Refreshes plots in the specified region.
     * 
     * @param region Plot region.
     * @param timeWindow Time window for the new plots. 
     *                   If <tt>null</tt>, time windows already associated with plots being refreshed are used.
     * @param options Additional options. If no options are given, this method will only refresh 
     *                plots with sliding time windows, and re-plot regions with refreshed plots.
     *                Specifying REFRESH_ALL or REFRESH_NONE options results in re-fetching data for all
     *                or none of the plots, respectively. FORCE_REPLOT option guarantees that the region is
     *                redrawn whether or not any plots it contains have been refreshed.
     */
    public void refresh(PlotRegion region, TimeWindow timeWindow, Options... options) {
        
        // check if region is still valid
        
        try {
            if (((JComponent)region).getTopLevelAncestor() == null) return;
        } catch (ClassCastException x) {
        }
        
        // if we're on event processing thread and fetching data from database is needed, move to executor
        
        PlotPage page = null;
        Plotter plotter = region.currentPlot();
        if (plotter == null) return;
        EnumSet<Options> opt = options.length == 0 ? EnumSet.noneOf(Options.class) : EnumSet.copyOf(Arrays.asList(options));
        
        if (SwingUtilities.isEventDispatchThread()) {
            boolean refreshReady = timeWindow == null;
            if (refreshReady) {
                refreshReady = opt.contains(REFRESH_NONE) || plotter.getData().stream()
                       .filter(po -> po instanceof Plot)
                       .allMatch(po -> {
                           Plot plot = (Plot)po;
                           return plot.isRefreshReady() || (!opt.contains(REFRESH_ALL) && plot.getTimeWindow().isFixed());
                       });
            }
            if (!refreshReady) {
//                System.out.println("refresh(...) is moving to executor");
                executor.execute(() -> {
                    refresh(region, timeWindow, options);
                });
                return;
            }
        }
        
        boolean replot = opt.contains(FORCE_REPLOT);
        ArrayList<Object> data = new ArrayList<>(plotter.getData().size());
        for (Object plottedObject : plotter.getData()) {
            if (plottedObject instanceof Plot) {
                Plot plot = (Plot) plottedObject;
                page = plot.getPage();
                TimeWindow tw = plot.getTimeWindow();
                if (timeWindow == null) { // refreshing
                    if (!opt.contains(REFRESH_NONE) && (opt.contains(REFRESH_ALL) || !tw.isFixed())) {
                        replot = true;
                        if (!plot.isRefreshReady()) {
                            plot.refresh();
                        }
                        plot.commitRefresh();
                    }
                } else { // changing time window
                    replot = true;
                    Plot oldPlot = plot;
                    plot = oldPlot.getChannel().makePlot(timeWindow);
                    plot.setVisible(oldPlot);
                    plot.setRegion(region);
                }
                data.add(plot);
            } else if (!(plottedObject instanceof AuxData)) {
                data.add(plottedObject);
            }
        }
        if (replot && page != null) {
            plotData(data, page, region, false);
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void plotData(List<Object> data, PlotPage page, PlotRegion region, boolean overlay) {
        
        // make sure we're on event processing thread
        
        if (!SwingUtilities.isEventDispatchThread()) {
            final List<Object> dataFinal = data;
            boolean overlayFinal = overlay;
//            System.out.println("plotData(...) is moving from thread "+ Thread.currentThread().getName());
            SwingUtilities.invokeLater(() -> plotData(dataFinal, page, region, overlayFinal));
            return;
        }
        
        // check that the region is still valid

        if (data.isEmpty()) return;
        try {
            if (((JComponent)region).getTopLevelAncestor() == null) return;
        } catch (ClassCastException x) {
        }

        // prepare plotter
        
        Plotter plotter = region.currentPlot();
        if (plotter == null) {
            plotter = plotFactory.createPlotterFor(IBaseHistogram.class);
            overlay = false;
        } else if (!overlay) {
            synchronized(plotter) {
                plotter.clear();
            }
        } else {
            ArrayList<Object> oldData = hideAuxData(plotter);
            if (oldData != null) {
                oldData.addAll(data);
                data = oldData;
                overlay = false;
            }
        }
        
        // prepare style

        IPlotterStyle style = plotterStyleFactory.createPlotterStyle();
        final IAxisStyle xAxisStyle = style.xAxisStyle();
        xAxisStyle.setParameter("type", "date");

        boolean commonExtent = data.size() > 1 && !overlay;
        if (commonExtent) {
            long tLow = Long.MAX_VALUE;
            long tHigh = 0L;
            for (Object datum : data) {
                try {
                    Plot pd = (Plot) datum;
                    synchronized (pd) {
                        tLow = Math.min(tLow, (long) pd.lowerExtent(0));
                        tHigh = Math.max(tHigh, (long) pd.upperExtent(0));
                    }
                    xAxisStyle.setParameter("lowerLimit", String.valueOf(tLow / 1000));
                    xAxisStyle.setParameter("upperLimit", String.valueOf(tHigh / 1000));
                } catch (ClassCastException x) {
                }
            }
        }
        
        EnumMap<AuxData.Type, Boolean> stateAux = null;
        if (!overlay && data.size() == 1  &&  data.get(0) instanceof Plot) {
            stateAux = new EnumMap<>(AuxData.Type.class);
        }
        
        // add data to plotter

        synchronized (plotter) {
            for (int i = 0; i < data.size(); i++) {
                Object datum = data.get(i);
                if (datum instanceof Plot) {
                    Plot plot = (Plot) datum;
                    plot.setPage(page);
                    plot.setRegion(region);
                    plot.getChannel().addPlotData(plot);
                    if (!commonExtent) {
                        xAxisStyle.setParameter("lowerLimit", String.valueOf(plot.lowerExtent(0) / 1000));
                        xAxisStyle.setParameter("upperLimit", String.valueOf(plot.upperExtent(0) / 1000));
                    }
                    plotter.plot(plot, (i == 0 && !overlay) ? Plotter.NORMAL : Plotter.OVERLAY, style, "");
                    if (stateAux != null) {
                        for (AuxData.Type type : AuxData.Type.values()) {
                            if (plot.getData().isAvailable(type)) {
                                if (plot.isVisible(type)) {
                                    IPlotterStyle auxStyle = auxStyles.get(type);
                                    if (auxStyle == null) {
                                        auxStyle = plotterStyleFactory.createPlotterStyle();
                                    }
                                    auxStyle.setAxisStyleX(xAxisStyle);
                                    for (AuxData auxData : plot.getData().get(type)) {
                                        plotter.plot(auxData, Plotter.OVERLAY, auxStyle, "");
                                    }
                                    stateAux.put(type, true);
                                } else {
                                    stateAux.put(type, false);
                                }
                            }
                        }
                    }
                } else {
                    plotter.plot(datum, (i == 0 && !overlay) ? Plotter.NORMAL : Plotter.OVERLAY, style, "");
                }
            }
        }
        
        // update popup menu
        
        try {
            DefaultRegion reg = (DefaultRegion) region;
            reg.removePopupItems(null, it -> it instanceof Popup);
            reg.addPopupItems(new Popup(region, stateAux));
        } catch (ClassCastException x) {
        }
        
        // plot
        
        region.showPlot(plotter);
    }
    
    /**
     * If the given plotter contains auxiliary data, clear it and return the list of 
     * plotted objects excluding auxiliary data. Otherwise, do nothing and return null.
     */
    private ArrayList<Object> hideAuxData(Plotter plotter) {
        synchronized(plotter) {
            List<Object> data = plotter.getData();
            Predicate<Object> isNotAux = object -> !(object instanceof AuxData);
            boolean doNotReplot = data.stream().allMatch(isNotAux);
            if (doNotReplot) {
                return null;
            } else {
                ArrayList<Object> out = data.stream().filter(isNotAux).collect(Collectors.toCollection(ArrayList::new));
                plotter.clear();
                return out;
            }
        }
    }
    
    static List<PlotRegion> getRegions(DataChannelHandler channel) {
        return channel.getPlots().stream().map(Plot::getRegion).collect(Collectors.toList());
    }
    
    static Plot getPlot(PlotRegion region) {
        Plotter plotter = region.currentPlot();
        if (plotter == null) return null;
        for (Object po : getPlotterData(plotter)) {
            if (po instanceof Plot) {
                return (Plot) po;
            }
        }
        return null;
    }
    
    static List<Object> getPlotterData(Plotter plotter) {
        synchronized (plotter) {
            return new ArrayList<>(plotter.getData());
        }
    }


// -- Helper classes : ---------------------------------------------------------
    
    private class Popup implements HasPopupItems {
        
        private EnumMap<AuxData.Type, Boolean> state;
        private final PlotRegion region;
        
        Popup(PlotRegion region) {
            this.region = region;
        }
        
        Popup(PlotRegion region, EnumMap<AuxData.Type, Boolean> state) {
            this.region = region;
            this.state = (state == null || state.isEmpty()) ? null : state;
        }

        @Override
        public JPopupMenu modifyPopupMenu(JPopupMenu menu, Component componentt, Point point) {
            if (state == null) return menu;
            menu.insert(new JSeparator(), 0);
            JMenu extraMenu = new JMenu("Show extras");
            for (Map.Entry<AuxData.Type, Boolean> e : state.entrySet()) {
                if (e.getValue() != null) {
                    JCheckBoxMenuItem item = new JCheckBoxMenuItem(e.getKey().getLabel(), e.getValue());
                    item.addActionListener(evt -> {
                        Plot plot = getPlot(region);
                        if (plot != null) {
                            plot.setVisible(e.getKey(), item.getState());
                            PlotMaker.this.refresh(region, null, REFRESH_NONE, FORCE_REPLOT);
                        }
                    });
                    extraMenu.add(item);
                }
            }
            menu.insert(extraMenu, 0);
            return menu;
        }
        
        void set(AuxData.Type type, Boolean visible) {
            if (state == null) state = new EnumMap<>(AuxData.Type.class);
            state.put(type, visible);
        }

    }
    
}
