package org.lsst.ccs.plugin.jas3.console;

import jas.util.Application;
import java.awt.BorderLayout;
import java.awt.Component;
import static java.awt.Dialog.DEFAULT_MODALITY_TYPE;
import java.awt.Dimension;
import java.io.IOException;
import java.util.HashMap;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JEditorPane;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTree;
import javax.swing.WindowConstants;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

/**
 *
 * @author onoprien
 */
public class FilterChooser extends JDialog implements TreeSelectionListener {

// -- Private parts : ----------------------------------------------------------
    
    private final int VSPACE = 5;
    private final int HSPACE = 5;
    
    FilterRegistry registry;
    
    private JTree tree;
    private DefaultTreeModel treeModel;
    private JEditorPane descriptionPane;
    private JButton deleteButton;
    private JButton okButton;
    
    private String chosenName;

// -- Construction and initialization : ----------------------------------------
    
    public FilterChooser(Component parent, FilterRegistry registry) {
        
        super(FilterEditor.getWindow(parent), "Choose Filter", DEFAULT_MODALITY_TYPE.APPLICATION_MODAL);
        setDefaultCloseOperation(WindowConstants.HIDE_ON_CLOSE);
        this.registry = registry;
        
        DefaultMutableTreeNode root = new DefaultMutableTreeNode();
        createNodes(root);
        treeModel = new DefaultTreeModel(root);
        tree = new JTree(treeModel);
        tree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        tree.setRootVisible(true);
        tree.addTreeSelectionListener(this);
 
        descriptionPane = new JEditorPane();
        descriptionPane.setEditable(false);
        descriptionPane.setContentType("text/html");
 
        JScrollPane treeView = new JScrollPane(tree);
        JScrollPane descriptionView = new JScrollPane(descriptionPane);
        Dimension minimumSize = new Dimension(100, 50);
        treeView.setMinimumSize(minimumSize);
        descriptionView.setMinimumSize(minimumSize);
        JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, treeView, descriptionView);
        splitPane.setDividerLocation(200); 
        splitPane.setPreferredSize(new Dimension(500, 300));
        add(splitPane, BorderLayout.CENTER);
        
        Box controls = Box.createHorizontalBox();
        controls.setBorder(BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE));
        add(controls, BorderLayout.SOUTH);
        deleteButton = new JButton("Delete");
        deleteButton.setEnabled(false);
        deleteButton.addActionListener(e -> {
            String name = getSelectedName();
            Object selection = tree.getLastSelectedPathComponent();
            if (name == null || selection == null || !(selection instanceof Node)) return;
            Node node = (Node)selection;
            if (!node.canDelete()) return;
            try {
                registry.deleteFilter(name);
                while (node != null) {
                    TreeNode mother = node.getParent();
                    treeModel.removeNodeFromParent(node);
                    node = (mother.isLeaf() && mother != root) ? (Node) mother : null;
                }
            } catch (IOException x) {
                Application.getApplication().error("Unable to delete filter "+ name, x);
            }
        });
        controls.add(deleteButton);
        controls.add(Box.createHorizontalStrut(2*HSPACE));
        controls.add(Box.createHorizontalGlue());
        okButton = new JButton("OK");
        okButton.setEnabled(false);
        okButton.addActionListener(e -> {
            chosenName = getSelectedName();
            FilterChooser.this.setVisible(false);
        });
        controls.add(okButton);
        controls.add(Box.createHorizontalStrut(HSPACE));
        JButton button = new JButton("Cancel");
        button.addActionListener(e -> FilterChooser.this.setVisible(false));
        controls.add(button);
        
        pack();
        setLocationRelativeTo(getOwner());
    }
    
// -- Getters : ----------------------------------------------------------------
    
    public String getChosenName() {
        return chosenName;
    }
    
// -- Responding to selection changes : ----------------------------------------

    @Override
    public void valueChanged(TreeSelectionEvent e) {
        Object selection = tree.getLastSelectedPathComponent();
        Node node = selection instanceof Node ? (Node)selection : null;
        if (node == null) {
            deleteButton.setEnabled(false);
            okButton.setEnabled(false);
            descriptionPane.setText("");
        } else if (node.isLeaf()) {
            deleteButton.setEnabled(node.canDelete());
            okButton.setEnabled(true);
            descriptionPane.setText(node.description);
        } else {
            deleteButton.setEnabled(false);
            okButton.setEnabled(false);
            if (node.getLevel() == 1) {
                String s = node.toString()+"/";
                if (s.equals(FilterRegistry.BUILTIN)) {
                    descriptionPane.setText("Built-In Filters.");
                } else if (s.equals(FilterRegistry.LOCAL)) {
                    descriptionPane.setText("Filters registered with the Graphical Console lookup.");
                } else if (s.equals(FilterRegistry.USER)) {
                    descriptionPane.setText("User-defined filters.");
                } else {
                    descriptionPane.setText("");
                }
            } else {
                descriptionPane.setText("");
            }
        }
    }    
    
    
// -- Showing chooser dialog : -------------------------------------------------

    /**
     * Opens a filter chooser dialog.
     * 
     * @param parent Parent component.
     * @param registry Filter registry.
     * @return Full name of the selected filter, or <tt>null</tt> if the user cancels the dialog.
     */
    static public String showDialog(Component parent, FilterRegistry registry) {
        FilterChooser chooser = new FilterChooser(parent, registry);
        chooser.setVisible(true);
        String out = chooser.getChosenName();
        chooser.dispose();
        return out;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void createNodes(DefaultMutableTreeNode root) {
        HashMap<String,Node> nodes = new HashMap<>();
        registry.availableFilters().forEach((filter, description) -> {
            String[] tokens = filter.split("/");
            String path = "";
            DefaultMutableTreeNode parent = root;
            for (String s : tokens) {
                path += s;
                Node node = nodes.get(path);
                if (node == null) {
                    node = new Node(s);
                    nodes.put(path, node);
                    parent.add(node);
                }
                parent = node;
                path += "/";
            }
            ((Node)parent).description = description;
            ((Node)parent).local = path.startsWith(FilterRegistry.USER);
        });
    }
    
    private String getSelectedName() {
        TreePath path = tree.getSelectionPath();
        if (path == null) {
            return null;
        } else {
            StringBuilder sb = new StringBuilder();
            for (Object o : path.getPath()) {
                sb.append(o).append("/");
            }
            return sb.substring(1, sb.length() - 1);
        }
    }
    
// -- Local classes : ----------------------------------------------------------
    
    private class Node extends DefaultMutableTreeNode {
        Node(String name) {
            super(name);
        }
        String description;
        boolean local; // can be deleted
        boolean canDelete() {
            return local;
        }
    }
    
}
