package org.lsst.ccs.plugin.jas3.trending;

import hep.aida.IAnnotation;
import hep.aida.IDataPoint;
import hep.aida.IDataPointSet;
import java.util.EnumSet;
import org.freehep.jas.services.PlotPage;
import org.freehep.jas.services.PlotRegion;
import org.lsst.ccs.plugin.jas3.trending.AuxData.Type;
import org.lsst.ccs.plugin.jas3.trending.timeselection.TimeWindow;

/**
 * Plotted data for a single channel with additional bookkeeping references.
 * An object of this type can be passed to a plotter and will be plotted as a data point set.
 * It also contains additional plottable data that may or may not be displayed depending on
 * plotter settings, as well as references to components where it is plotted.
 * 
 * In addition to its current dataset, an instance of this class may keep a reference 
 * to an updated data point set.
 * During construction, the current data point set is supplied and the updated data point set
 * is <tt>null</tt>. Later, the updated data point set can be set through a call to
 * one of the <tt>refresh(PlotData, ...)</tt> methods. Calling {@link #commitRefresh()}
 * replaces current data point set with the updated one, creating the latter if it has not been
 * set since the previous call to <tt>refresh()</tt>.
 * 
 * Methods modifying data sets of the Plot instance are synchronized on its monitor.
 * 
 * @author onoprien
 */
public final class Plot implements IDataPointSet {
    
// -- Private parts : ----------------------------------------------------------
        
    private PlotData data;
    private double low = -1.;
    private double high = -1.;
    
    private final TimeWindow timeWindow;
    private final DataChannelHandler channel;
    private PlotPage page;
    private PlotRegion region;
    
    private volatile PlotData spareData;
    private double spareLow = -1.;
    private double spareHigh = -1.;
    
    private volatile long timestamp;
    private EnumSet<Type> isVisible = EnumSet.noneOf(Type.class);

// -- Constructors : -----------------------------------------------------------

    public Plot(PlotData data, TimeWindow timeWindow, DataChannelHandler channel) {
        this(data, timeWindow, channel, null, null);
    }

    public Plot(PlotData data, TimeWindow timeWindow, DataChannelHandler channel, PlotPage page, PlotRegion region) {
        this.data = data;
        this.timeWindow = timeWindow;
        this.channel = channel;
        this.page = page;
        this.region = region;
        timestamp = System.currentTimeMillis();
    }


// -- Type getters and setters : ----------------------------------------------
    
    /** *  Returns the time of construction or last call to <tt>refresh(...)</tt>. */
    public long getTimeStamp() {
        return timestamp;
    }
    
    /** Returns time window associated with this dataset. */
    public TimeWindow getTimeWindow() {
        return timeWindow;
    }

    /** Returns data channel used to populate this dataset. */
    public DataChannelHandler getChannel() {
        return channel;
    }

    /** Returns the page where this dataset is plotted. */
    public PlotPage getPage() {
        return page;
    }

    /** Associates a page with this dataset. */
    public void setPage(PlotPage page) {
        this.page = page;
    }

    /** Returns the region where this dataset is plotted. */
    public PlotRegion getRegion() {
        return region;
    }

    /** Associates a region with this dataset. */
    public void setRegion(PlotRegion region) {
        this.region = region;
    }
    
    /**
     * Sets the range of this data point set along time axis.
     * The edges of the range can be later retrieved with {@link #lowerExtent} and
     * {@link #upperExtent} methods. Setting the time range does not affect {@link TimeWindow}
     * object associated with this data point set.
     * 
     * @param begin Beginning of the range in milliseconds.
     * @param end End of the range in milliseconds.
     */
    public void setTimeRange(long begin, long end) {
        low = begin;
        high = end;
    }
    
    /** Returns average time interval between points in this dataset, in milliseconds. */
    public double getGranularity() {
        int n = size();
        if (n<2) {
            return (upperExtent(0) - lowerExtent(0)) / 100.;
        } else {
            return (upperExtent(0) - lowerExtent(0)) / n;
        }
    }
    
    /** Returns visibility flag for the specified auxiliary data type. */
    public boolean isVisible(Type what) {
        return isVisible.contains(what);
    }
    
    /** Sets visibility flag for the specified auxiliary data type. */
    public void setVisible(Type what, boolean visible) {
        if (visible) {
            isVisible.add(what);
        } else {
            isVisible.remove(what);
        }
    }
    
    /** Sets auxiliary data visibility flags to match those of the specified plot. */
    public void setVisible(Plot other) {
        isVisible = other.isVisible.clone();
    }
    
    /** Returns current data set associated with this plot. */
    public PlotData getData() {
        return data;
    }
    
    /** Returns true if updated data for this plot exists and can be committed. */
    public boolean isRefreshReady() {
        return spareData != null;
    }
    
    
// -- Changing data : ----------------------------------------------------------
    
    /**
     * Replaces current plot data with updated plot data, creating the latter if necessary.
     * 
     * @return True if the current data point set has changed as a result of this call.
     */
    synchronized public boolean commitRefresh() {
        if (spareData == null || data == spareData) return false;
        data = spareData;
        low = spareLow;
        high = spareHigh;
        spareData = null;
        spareLow = -1;
        spareHigh = -1;
        return true;
    }
    
    /**
     * Sets updated data for this plot.
     * Time range is calculated based on the content of the supplied data point set.
     */
    public void refresh(PlotData data) {
        refresh(data, -1L, -1L);
    }
    
    /**
     * Sets updated data for this plot.
     */
    synchronized public void refresh(PlotData data, long begin, long end) {
        spareData = data;
        spareLow = begin;
        spareHigh = end;
        timestamp = System.currentTimeMillis();
    }
    
    /**
     * Sets updated data for this plot after fetching it from the database.
     * @return True if new data is available and the update succeeds.
     */
    public boolean refresh() {
        return channel.refresh(this);
    }
    

// -- IDataPointSet implementation : -------------------------------------------

    @Override
    public IAnnotation annotation() {
        return data.get().annotation();
    }

    @Override
    public String title() {
        return channel.getPath();
    }

    @Override
    public void setTitle(String string) throws IllegalArgumentException {
        data.get().setTitle(string);
    }

    @Override
    public int dimension() {
        return data.get().dimension();
    }

    @Override
    public void clear() {
        data.get().clear();
    }

    @Override
    public int size() {
        return data.get().size();
    }

    @Override
    public IDataPoint point(int i) {
        return data.get().point(i);
    }

    @Override
    public void setCoordinate(int i, double[] doubles, double[] doubles1) throws IllegalArgumentException {
        data.get().setCoordinate(i, doubles, doubles1);
    }

    @Override
    public void setCoordinate(int i, double[] doubles, double[] doubles1, double[] doubles2) throws IllegalArgumentException {
        data.get().setCoordinate(i, doubles, doubles1, doubles2);
    }

    @Override
    public IDataPoint addPoint() throws RuntimeException {
        return data.get().addPoint();
    }

    @Override
    public void addPoint(IDataPoint idp) throws IllegalArgumentException {
        data.get().addPoint(idp);
    }

    @Override
    public void removePoint(int i) throws IllegalArgumentException {
        data.get().removePoint(i);
    }

    @Override
    public double lowerExtent(int i) throws IllegalArgumentException {
        return (i==0 && low > 0.) ? low : data.get().lowerExtent(i);
    }

    @Override
    public double upperExtent(int i) throws IllegalArgumentException {
        return (i==0 && high > 0.) ? high : data.get().upperExtent(i);
    }

    @Override
    public void scale(double d) throws IllegalArgumentException {
        data.get().scale(d);
    }

    @Override
    public void scaleValues(double d) throws IllegalArgumentException {
        data.get().scaleValues(d);
    }

    @Override
    public void scaleErrors(double d) throws IllegalArgumentException {
        data.get().scaleErrors(d);
    }
  
}
