package org.lsst.ccs.plugin.jas3.trending;

import com.sun.jersey.api.client.WebResource;
import hep.aida.IDataPoint;
import hep.aida.ref.histogram.DataPointSet;
import java.awt.Container;
import java.lang.ref.WeakReference;
import java.util.*;
import javax.swing.JComponent;
import javax.ws.rs.core.MediaType;
import org.freehep.application.Application;
import org.freehep.application.studio.Plugin;
import org.freehep.application.studio.PluginInfo;
import org.freehep.application.studio.Studio;
import org.freehep.jas.services.PlotPage;
import org.freehep.jas.services.PlotRegion;
import org.freehep.jas.services.Plotter;
import org.lsst.ccs.localdb.statusdb.server.ChannelMetaData;
import org.lsst.ccs.localdb.statusdb.server.Data;
import org.lsst.ccs.localdb.statusdb.server.DataChannel;
import org.lsst.ccs.localdb.statusdb.server.TrendingData;
import org.lsst.ccs.localdb.statusdb.server.TrendingResult;
import org.lsst.ccs.plugin.jas3.rest.LsstRestPlugin;
import org.lsst.ccs.plugin.jas3.trending.timeselection.TimeWindow;

/**
 * Handles retrieval and caching of trending data associated with a specific {@link DataChannel}.
 * Also holds references to plotted data point sets for this channel.
 *
 * @author onoprien
 */
public class DataChannelHandler {
    
// -- Private parts : ----------------------------------------------------------
    
    private final DataChannel channel;
    private final DataChannelTreeAdapter adapter;
    private ArrayList<WeakReference<Plot>> plots;
    
    
// -- Construction and initialization : ----------------------------------------
    
    DataChannelHandler(DataChannel dataChannel, DataChannelTreeAdapter treeAdapter) {
        channel = dataChannel;
        adapter = treeAdapter;
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    /** Returns the path associated with the underlying {@link DataChannel}. */
    public String getPath() {
        return channel.getPathAsString();
    }
    
    /** Returns the underlying <tt>DataChannel</tt>. */
    public DataChannel getDataChannel() {
        return channel;
    }
    
    /**
     * Returns a list of plotted data point sets for this channel.
     * The list is not modifiable. 
     * Calling this method purges the list of plotted dataset, removing those associated with deleted regions.
     */
    public synchronized List<Plot> getPlots() {
        if (plots == null) return Collections.emptyList();
        ArrayList<Plot> out = new ArrayList<>(plots.size());
        ListIterator<WeakReference<Plot>> it = plots.listIterator();
        while (it.hasNext()) {
            WeakReference<Plot> ref = it.next();
            Plot pd = ref.get();
            if (pd == null || !isDataPlotted(pd)) {
                it.remove();
            } else {
                out.add(pd);
            }
        }
        if (out.isEmpty()) {
            plots = null;
        }
        return out;
    }
    
    /**
     * Returns an existing plot for this channel.
     * If there are several such plots, returns the one with 
     * the specified time window, if it exist. Otherwise, returns one of plots
     * displayed on pages with minimal number of regions. If there are no plots for this
     * channel, returns <tt>null</tt>. Only single plot regions are considered.
     */
    public Plot findPlot(TimeWindow timeWindow) {
        if (plots == null) return null;
        Plot out = null;
        for (Plot pd : getPlots()) {
            if (pd.getRegion().currentPlot().getData().stream().filter(e -> !(e instanceof AuxData)).count() == 1) {
                if (pd.getTimeWindow() == timeWindow) {
                    return pd;
                } else if (out == null || (pd.getPage().numberOfRegions() < out.getPage().numberOfRegions())) {
                    out = pd;
                }
            }
        }
        return out;
    }
    
// -- Operations : -------------------------------------------------------------
    
    /**
     * Creates and returns a data point set for the specified time window.
     */
    public Plot makePlot(TimeWindow timeWindow) {
        long begin = timeWindow.getLowerEdge();
        long end = timeWindow.getUpperEdge();
        PlotData dps = makePlotData(begin, end, 1);
        Plot pd = new Plot(dps, timeWindow, this);
        pd.setTimeRange(begin, end);
        return pd;
    }
    
    /**
     * Prepares the specified plot for refreshing.
     * Creates updated data point sets and stores them in the specified Plot object.
     * Subsequent call to {@link Plot#commitRefresh} on that instance will make the new data current.
     * 
     * @param plot Plot to be prepared for refreshing.
     * @param time Current time in milliseconds.
     * @return true if the the data has been updated.
     */
    public boolean refresh(Plot plot, long time) {
        TimeWindow timeWindow = plot.getTimeWindow();
        long begin = timeWindow.getLowerEdge(time);
        long end = timeWindow.getUpperEdge(time);
        try {
            PlotData data = makePlotData(begin, end, 2);
            plot.refresh(data, begin, end);
        } catch (RuntimeException x) {
            return false;
        }
        return true;
    }
    
    /**
     * Prepares the specified plot for refreshing.
     * Creates updated data point sets and stores them in the specified Plot object.
     * Subsequent call to {@link Plot#commitRefresh} on that instance will make the new data current.
     * 
     * @param plot Plot to be prepared for refreshing.
     * @return true if the the data has been updated.
     */
    public boolean refresh(Plot plot) {
        return refresh(plot, System.currentTimeMillis());
    }
    
    /**
     * Adds the specified data point set to the list of plotted data for this channel.
     */
    public synchronized void addPlotData(Plot plot) {
        if (plot == null) return;
        boolean out = true;
        if (plots == null) {
            plots = new ArrayList<>(1);
        } else {
            ListIterator<WeakReference<Plot>> it = plots.listIterator();
            while (it.hasNext()) {
                WeakReference<Plot> ref = it.next();
                Plot pd = ref.get();
                if (pd == null) {
                    it.remove();
                } else if (pd == plot) {
                    out = false;
                }
            }
        }
        if (out) {
            plots.add(new WeakReference<>(plot));
        }
    }
    
    /**
     * Removes the specified data point set from the list of plotted data for this channel.
     */
    public synchronized void removePlotData(Plot plot) {
        if (plot == null || plots == null) return;
        ListIterator<WeakReference<Plot>> it = plots.listIterator();
        while (it.hasNext()) {
            WeakReference<Plot> ref = it.next();
            Plot pd = ref.get();
            if (pd == null || pd == plot) {
                it.remove();
            }
        }
        if (plots.isEmpty()) {
            plots = null;
        }
    }

    
// -- Local methods : ----------------------------------------------------------
    
    private PlotData makePlotData(long begin, long end, int retry) {
        while (true) {
            try {
                PlotData pd = makePlotData(begin, end);
                return pd;
            } catch (RuntimeException x) {
                System.out.println("Exception thrown while fetching data "+ x);
                System.out.println("Time: "+ new Date());
//                Application.getApplication().error("Exception thrown while fetching data", x);
                    List plugins = ((Studio) (Application.getApplication())).getPlugins();
                    for (Object o : plugins) {
                        Plugin plugin = ((PluginInfo) o).getPlugin();
                        if (plugin != null && plugin instanceof LsstRestPlugin) {
                            ((LsstRestPlugin) plugin).refreshConnector();
                            System.gc();
                            break;
                        }
                    }
                if (--retry < 0) {
                    return null;
                }
            }
        }
    }

    
//    private Random random = new Random();
    private PlotData makePlotData(long begin, long end) {
        
//        int delay = random.nextInt(20);
//        System.out.println(channel.getPathAsString() +" will procrastinate for "+ delay +" seconds on thread "+ Thread.currentThread().getName());
//        try {
//            Thread.sleep(delay * 1000);
//        } catch (Exception x) {
//        }
        
        WebResource resource = adapter.getResource().path("data").path(String.valueOf(channel.getId()));
        resource = resource.queryParam("t1", String.valueOf(begin)).queryParam("t2", String.valueOf(end));
        TrendingPreferences pref = adapter.getPlugin().getPreferences();
        int nBins = pref.nBins();
        if (nBins > 0) {
            if (!pref.useRawData()) {
                resource = resource.queryParam("flavor", "stat");
            }
            resource = resource.queryParam("n", Integer.toString(nBins));
        } else {
            resource = resource.queryParam("flavor", pref.useRawData() ? "raw" : "stat");
        }
        Data data = resource.accept(MediaType.TEXT_XML).get(Data.class);
        TrendingResult result = data.getTrendingResult();
        
        DataPointSet dps = new DataPointSet("", "", 2);
        TrendingData[] dataArray = result.getTrendingDataArray();
        if (dataArray != null) {
            boolean drawRms = pref.drawRms();
            for (TrendingData d : dataArray) {
                IDataPoint p = dps.addPoint();
                p.coordinate(0).setValue(d.getAxisvalue().getValue() / 1000.);
                p.coordinate(1).setValue(d.getValue("value"));
                if (drawRms) {
                    double rms = d.getValue("rms");
                    if (!Double.isNaN(rms)) {
                        p.coordinate(1).setErrorMinus(rms);
                        p.coordinate(1).setErrorPlus(rms);
                    }
                }
            }
        }
        PlotData pd = new PlotData(dps);
        
        List<ChannelMetaData> metaList = result.getChannelMetadata();
        HashMap<String,AuxDataPointSet> metaMap = new HashMap<>();
        for (ChannelMetaData meta : metaList) {
            try {
                String name = meta.getName();
                AuxDataPointSet ps = metaMap.get(name);
                if (ps == null) {
                    ps = new AuxDataPointSet(null, name);
                    metaMap.put(name, ps);
                }
                double value = Double.parseDouble(meta.getValue());
                long start = meta.getTstart();
                if (start < begin) start = begin;
                IDataPoint p = ps.addPoint();
                p.coordinate(0).setValue(start / 1000.);
                p.coordinate(1).setValue(value);
                long stop = meta.getTstop();
                if (stop > end || stop == -1) stop = end;
                p = ps.addPoint();
                p.coordinate(0).setValue(stop / 1000.);
                p.coordinate(1).setValue(value);
            } catch (NumberFormatException x) {
            }
        }
        
        metaMap.values().forEach(d -> pd.add(d));        
        
        return pd;
    }
    
    private boolean isDataPlotted(Plot plotData) {
        
        PlotPage page = plotData.getPage();
        PlotRegion region = plotData.getRegion();
        if (page == null || region == null) return false;
        
        // check whether the region still belongs to the component hierarchy
        
        try {
            Container c = ((JComponent)region).getTopLevelAncestor();
            if (c == null) return false;
        } catch (ClassCastException x) {
        }
        
        // check whether the region references this data
        
        Plotter p = region.currentPlot();
        if (p == null) return false;
        List dataList = p.getData();
        if (dataList == null || dataList.indexOf(plotData) == -1) return false;
        
        // passed all checks
        
        return true;
    }

    
}
