package org.lsst.ccs.plugin.jas3.console;

import java.io.IOException;
import java.io.PrintWriter;
import java.net.URL;
import org.freehep.application.studio.Studio;
import org.freehep.jas.extensions.jconsole.JConsoleService;
import org.freehep.jas.extensions.jconsole.JLineConsole;
import org.freehep.jas.extensions.jconsole.JLineConsoleProvider;
import org.freehep.jas.plugin.console.Console;
import org.freehep.jas.plugin.console.ConsoleOutputStream;
import org.freehep.jas.plugin.console.ConsoleService;
import org.freehep.util.FreeHEPLookup;
import org.freehep.util.commanddispatcher.CommandProcessor;
import org.freehep.xml.menus.XMLMenuBuilder;
import org.lsst.ccs.plugin.jas3.LsstPlugin;
import org.lsst.ccs.shell.JLineShell;
import org.lsst.ccs.subsystem.shell.ConsoleCommandShell;
import org.xml.sax.SAXException;

/**
 * This plugin deals with creating LSST CCS tracers and consoles.
 */
public class LsstConsolePlugin extends LsstPlugin {

    private final String defaultConsoleName = "LSST Console";
    private final String defaultTracerName = "LSST Tracer";

    @Override
    protected void init() throws SAXException, IOException {

        Studio studio = getApplication();
        FreeHEPLookup lookup = studio.getLookup();
        lookup.add(this);

        studio.getCommandTargetManager().add(new ConsoleCommands());

        XMLMenuBuilder builder = studio.getXMLMenuBuilder();
        URL xml = getClass().getResource("LsstConsolePlugin.menus");
        builder.build(xml);
    }

    @Override
    protected void postInit() {
        super.postInit();
        JConsoleService cs = (JConsoleService) getApplication().getLookup().lookup(JConsoleService.class);
        if (cs == null) throw new IllegalStateException("JConsole service is not available");
        cs.addConsoleProvider(JLineConsole.class, new JLineConsoleProvider());
        getMessagingAccess().addStatusMessageListener(new AlarmListener(getApplication()));
    }
    
   
    private JLineConsole createCommandConsole(String name) {
        Studio studio = getApplication();
        try {
            JConsoleService cs = (JConsoleService) getApplication().getLookup().lookup(JConsoleService.class);
            if (cs != null) {
                JLineConsole console = (JLineConsole) cs.createConsole(JLineConsole.class, name, null, null);
                JLineShell shell = ConsoleCommandShell.createJLineShell(getMessagingAccess(), console.getConsoleReader());
                Thread t = new Thread("ConsoleInput-" + name) {
                    @Override
                    public void run() {
                        try {
                            shell.run();
                        } catch (IOException x) {
                            getApplication().error("Error in command shell", x);
                        } finally {
                        }
                    }
                };
                t.start();
                return console;
            }
        } catch (Exception x) {
            studio.error("Failed to start Console: " + name, x);
        }
        return null;
    }

    /** Creates JLineConsole based tracer. Not used at the moment. */
    private JLineConsole createJTracerConsole(String name) {
        FreeHEPLookup lookup = getApplication().getLookup();
        JConsoleService cs = (JConsoleService) lookup.lookup(JConsoleService.class);
        if (cs != null) {
            Tracer tracer = new Tracer(getMessagingAccess());
            JLineConsole console = (JLineConsole) cs.createConsole(JLineConsole.class, name, null, tracer);
            PrintWriter pw = new PrintWriter(console.getConsoleReader().getOutput(), true);
            cs.getPageContextForConsole(console).addPageListener(tracer);
            tracer.start(pw); // This returns immediately
            return console;
        }
        return null;
    }

    private Console createTracerConsole(String name) {
        FreeHEPLookup lookup = getApplication().getLookup();
        ConsoleService cs = (ConsoleService) lookup.lookup(ConsoleService.class);
        if (cs != null) {
            Tracer tracer = new Tracer(getMessagingAccess());
            Console console = cs.createConsole(name, null, tracer);
            ConsoleOutputStream out = console.getOutputStream(null);
            PrintWriter pw = new PrintWriter(out, true);
            cs.getPageContextForConsole(console).addPageListener(tracer);
            tracer.start(pw); // This returns immediately
            return console;
        }
        return null;
    }

    public class ConsoleCommands extends CommandProcessor {

        public void onLsstConsole() {
            createCommandConsole(defaultConsoleName);
        }

        public void onLsstTracer() {
            createTracerConsole(defaultTracerName);
        }
    }
}
