package org.lsst.ccs.plugin.jas3.dictionary;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.net.URL;
import java.util.concurrent.ExecutionException;
import java.util.logging.Level;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.SwingWorker;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableModel;
import javax.swing.tree.DefaultTreeModel;
import org.lsst.ccs.bus.AgentPresenceListener;
import org.lsst.ccs.bus.Agent;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.framework.DictionaryContext;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.structs.TreeBranch;

/**
 *
 * @author emarin
 */
public class DictionaryTreeMainPanel extends JPanel implements TableModelListener, ListSelectionListener, AgentPresenceListener, ActionListener {

   // private ModelSubsystemBusMaster busMaster;
    private CommandListener commandListener;

    private Logger logger = Logger.getLogger("org.lsst.ccs.subsystems.model.dictionaryTreeGui");
    private CommandListPanel cmdList;

    private AgentTable agentTable;

    private DictionaryTreePanel dictionaryTree;
    private ArgInputPanel argInputPanel;
    private HTMLTextPane resultPane;
    private JButton sendCmdButton;
    private JButton refreshAgentsButton;
    private JPanel topPane;

    //Optionally play with line styles.  Possible values are
    //"Angled" (the default), "Horizontal", and "None".
    private static boolean playWithLineStyle = false;
    private static String lineStyle = "Horizontal";

//    public void initGUI() {
//        //setName("ModelGUIModule");
//        busMaster.setListenToStatus(true);
//        busMaster.setStatusBroadcastPeriod(0);
//        busMaster.start();
//    }
    //Optionally set the look and feel.
    public DictionaryTreeMainPanel(CommandListener cmdListener) {
        super(new BorderLayout());
        commandListener = cmdListener;
        //   busMaster = new ModelSubsystemBusMaster("model-subsystem-gui");

        // Agent Table creation
        agentTable = new AgentTable(commandListener);
        agentTable.getModel().addTableModelListener(this);

        // Dictionary Tree creation
        dictionaryTree = new DictionaryTreePanel(commandListener);
        agentTable.getSelectionModel().addListSelectionListener(this);
//        agentJTable.getModel().addTableModelListener(dictionaryJTree);

        // Create a scroll pane and adds the subsystem list to it
        JScrollPane agentsView = new JScrollPane(agentTable);
        agentTable.setFillsViewportHeight(true);

        if (playWithLineStyle) {
            System.out.println("line style = " + lineStyle);
            dictionaryTree.putClientProperty("JTree.lineStyle", lineStyle);
        }

        //Create the scroll pane and add the dictionaryJTree to it. 
        JScrollPane treeView = new JScrollPane(dictionaryTree);

        // Create the command list viewing pane.
        cmdList = new CommandListPanel();
        dictionaryTree.addTreeSelectionListener(cmdList);
        agentTable.getSelectionModel().addListSelectionListener(cmdList);
        //agentJTable.getModel().addTableModelListener(cmdList);

        cmdList.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent e) {
                if (e.getClickCount() == 2) {
                    DictionaryCommand cmd = cmdList.getSelectedValue();
                    // Double click is only enabled for command without arguments
                    if (cmd.getArguments().length == 0) {
                        sendCommand();
                    }
                }
            }
        });

        JScrollPane cmdListView = new JScrollPane(cmdList);

        //Add the views to the TOP panel
        topPane = new JPanel(new GridLayout(1, 3));

        // Add subsystemview to the main Panel (not the top panel)
        JPanel leftPane = new JPanel(new BorderLayout());
        leftPane.setPreferredSize(new Dimension(300,800));

        // Refresh agents button
        URL url = this.getClass().getResource("/org/lsst/ccs/plugin/jas3/dictionary/refresh_24dp.png");
        refreshAgentsButton = new JButton();
        refreshAgentsButton.setIcon(new ImageIcon(url));
        refreshAgentsButton.setToolTipText("Refresh");
        refreshAgentsButton.addActionListener(this);

        leftPane.add(agentsView, BorderLayout.CENTER);
        leftPane.add(refreshAgentsButton, BorderLayout.SOUTH);

        //leftPane.setPreferredSize(new Dimension(300, 800));
        add(leftPane, BorderLayout.WEST);

        // Add TreeView to the top panel
        topPane.add(treeView);

        //Add command list view to the top panel
        topPane.add(cmdListView);

        // Result Panel
        // create and add the argInputPanel to the top panel
        argInputPanel = new ArgInputPanel();
        cmdList.addListSelectionListener(argInputPanel);

        JPanel buttonPane = new JPanel();
        buttonPane.setLayout(new FlowLayout(FlowLayout.RIGHT));
        sendCmdButton = new JButton(new ImageIcon(this.getClass().getResource("/org/lsst/ccs/plugin/jas3/dictionary/ic_send_black_24dp.png")));
        sendCmdButton.setText("Send");
        sendCmdButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                sendCommand();
            }
        });

        buttonPane.add(sendCmdButton);
        add(buttonPane, BorderLayout.SOUTH);

        JPanel topPaneRightPanel = new JPanel(new BorderLayout());
        topPaneRightPanel.add(argInputPanel, BorderLayout.CENTER);
        topPaneRightPanel.add(buttonPane, BorderLayout.SOUTH);

        topPane.add(topPaneRightPanel);

        // Console Pane
        resultPane = new HTMLTextPane();
        JScrollPane resultPaneView = new JScrollPane(resultPane);

        // Split Panel creation
        JSplitPane splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
        splitPane.setTopComponent(topPane);
        splitPane.setBottomComponent(resultPaneView);
        splitPane.setContinuousLayout(true);

        add(splitPane, BorderLayout.CENTER);
    }

    @Override
    public void tableChanged(TableModelEvent e) {
        if (e.getType() == TableModelEvent.UPDATE){
            if (agentTable.getSelectedRow() >= 0){
                String selectedSubsystem = ((Agent)agentTable.getValueAt(agentTable.getSelectedRow(), 0)).getName();
                boolean isEM = (boolean)agentTable.getValueAt(agentTable.getSelectedRow(), 1);
                String cmdName = isEM ? "switchToEngineeringMode" : "switchToNormalMode";
                resultPane.insertHTML("invoke " + selectedSubsystem + " " + cmdName +" ...");
                //Update Tree Panel
                dictionaryTree.updateRenderer(isEM);
                //Update List Command Panel
                cmdList.updateRenderer(isEM);
                new SendCommandWorker(cmdName, new Object[0], selectedSubsystem).execute();
            }
        }
    }
    
    // Agent selected has changed
    @Override
    public void valueChanged(ListSelectionEvent e) {
        if(agentTable.getSelectedRow() >=0){
            String selectedSubsystem = ((Agent)agentTable.getValueAt(agentTable.getSelectedRow(), 0)).getName();
            boolean isEM = (boolean)agentTable.getValueAt(agentTable.getSelectedRow(), 1);
            dictionaryTree.updateRendererAndData(isEM, selectedSubsystem);
        } else { // No agent selected : clear the tree panel
            dictionaryTree.setModel(new DefaultTreeModel(null));
        }
    }

    @Override
    public void connecting(Agent agent) {
        boolean isEM = commandListener.isInEngineeringMode(agent.getName());
        ((DefaultTableModel)agentTable.getModel()).addRow(new Object[]{agent,isEM});
        
    }

    @Override
    public void disconnecting(Agent agent) {
        ((AgentDefaultTableModel)agentTable.getModel()).removeRow(agent);
    }

    public void sendCommand(){
        TreeBranch<DictionaryContext> destinationNode = (TreeBranch<DictionaryContext>) dictionaryTree.getLastSelectedPathComponent();
                String destination = destinationNode.toString();
                String rootNode = dictionaryTree.getModel().getRoot().toString();

                if (!destination.equals(rootNode)) {
                    destination = rootNode + "/" + destination;
                }

                String s = "";
                DictionaryCommand cmd = cmdList.getSelectedValue();
                if (cmd != null) {
                    int n = cmd.getArguments().length;
                    Object[] p = new Object[n];
                    for (int i = 0; i < n; i++) {
                        p[i] = parse(argInputPanel.getArgInput()[i].getText());
                        s += p[i] + " ";
                    }

                    resultPane.insertHTML("invoke " + destination + " " + cmd.getCommandName() + " " + s + "...");
                    new SendCommandWorker(cmd.getCommandName(), p, destination).execute();
                }
        
    }

    @Override
    public void actionPerformed(ActionEvent e) {
        updateAgents();
    }
    
    public synchronized void updateAgents(){
        new GetCommandAgentWorker().execute();
    }
    
    private class GetCommandAgentWorker extends SwingWorker<Object[][], String> {

        @Override
        protected Object[][] doInBackground() throws Exception {
            return commandListener.getConnectedAgents();
            
        }

        @Override
        public void done() {
            try {
                (((AgentDefaultTableModel)agentTable.getModel())).update(get());
            
            } catch (InterruptedException ex) {
                java.util.logging.Logger.getLogger(AgentTable.class.getName()).log(Level.SEVERE, null, ex);
            } catch (ExecutionException ex) {
                java.util.logging.Logger.getLogger(AgentTable.class.getName()).log(Level.SEVERE, null, ex);
            }
            

        }
    }
    
    private class SendCommandWorker extends SwingWorker<String, String> {

        private String cmdName;
        private Object[] cmdArgs;
        private String destination;

        SendCommandWorker(String cmdName, Object[] cmdArgs, String destination) {
            this.cmdName = cmdName;
            this.cmdArgs = cmdArgs;
            this.destination = destination;
        }

        @Override
        protected String doInBackground() throws Exception {
            return commandListener.sendCommand(cmdName, cmdArgs, destination);
        }

        @Override
        public void done() {
            try {
                String result = get();
                resultPane.insertHTML(result);
            } catch (InterruptedException ex) {
                Logger.getLogger(DictionaryBusMaster.class.getName()).log(Level.SEVERE, null, ex);
            } catch (ExecutionException ex) {
                Logger.getLogger(DictionaryBusMaster.class.getName()).log(Level.SEVERE, null, ex);
            }
        }

    }
    private class UniqueWorker extends SwingWorker<Object,Object> {

        @Override
        protected Object doInBackground() throws Exception {
            throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
        }
        
        @Override
        public void done() {
            try {
                String result =(String) get();
                resultPane.insertHTML(result);
            } catch (InterruptedException ex) {
                Logger.getLogger(DictionaryBusMaster.class.getName()).log(Level.SEVERE, null, ex);
            } catch (ExecutionException ex) {
                Logger.getLogger(DictionaryBusMaster.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        
    }
    private Object parse(String s) {
        //quoting mechanism
        if (s.startsWith("\"")) {
            // assuming there is a quote at the end .....
            return s.substring(1, s.length() - 1);
        }
        if (s.matches("-?[0-9]+")) {
            return Integer.parseInt(s);
        } else if (s.matches("-?[0-9]*[.][0-9]*|-?[0-9]*[.][0-9]*e-?[0-9]+")) {
            return Double.parseDouble(s);
        } else {
            return s;
        }
    }
}