package org.lsst.ccs.plugin.jas3.trending;

import hep.aida.IAxisStyle;
import hep.aida.IBaseHistogram;
import hep.aida.IPlotterStyle;
import hep.aida.ref.plotter.PlotterFactory;
import java.util.*;
import javax.swing.JComponent;
import org.freehep.application.studio.Studio;
import org.freehep.jas.services.PlotFactory;
import org.freehep.jas.services.PlotPage;
import org.freehep.jas.services.PlotRegion;
import org.freehep.jas.services.Plotter;
import org.lsst.ccs.plugin.jas3.trending.timeselection.TimeWindow;
import org.lsst.ccs.plugin.jas3.trending.timeselection.TimeWindowComboBox;

/**
 * Machinery for making and updating plots.
 * 
 * @author onoprien
 */
public class PlotMaker {
    
// -- Private parts : ----------------------------------------------------------
    
    private final Studio app;
    private final TimeWindowComboBox timeRangeSelector;
    private final PlotterFactory plotterStyleFactory = new PlotterFactory();
    private PlotFactory plotFactory;
    private volatile AutoRefresh autoRefresh;
    
// -- Construction and initialization : ----------------------------------------
    
    public PlotMaker(Studio application, TimeWindowComboBox timeWindowSelector) {
        app = application;
        timeRangeSelector = timeWindowSelector;
    }

    
// -- Operations : -------------------------------------------------------------
    
    public void setAutorefresh(AutoRefresh autoRefresh) {
        this.autoRefresh = autoRefresh;
    }
    
    public void plot(DataChannelHandler channel, TimeWindow timeWindow, boolean useExistingPlot, boolean newPage, boolean newPlot, boolean overlay) {

        if (channel == null) return;
        
        if (plotFactory == null) {
            plotFactory = (PlotFactory) app.getLookup().lookup(PlotFactory.class);
            if (plotFactory == null) {
                app.error("No plot factory available");
            }
        }
        
        // fetch currently selected time window in none is specified
        
        if (timeWindow == null) {
            timeWindow = timeRangeSelector.getSelectedTimeWindow();
            if (timeWindow == null) return;
        }

        PlotRegion region = null;
        PlotPage page = null;
        
        // create data point set

        Plot plotData = channel.makePlotData(timeWindow);
        
        // if using existing plot for the specified channel is requested, find it
        
        if (useExistingPlot) {
            Plot dps = channel.findPlotData(timeWindow);
            if (dps != null) {
                page = dps.getPage();
                region = dps.getRegion();
            }
        }
        
        // find or create target page
        
        if (page == null) {
            if (!newPage) {
                page = plotFactory.currentPage();
            }
            if (page == null) {
                page = plotFactory.createPage(channel.getPath());
                if (page == null) {
                    app.error("Unable to create a plot page");
                    return;
                } else {
                    page.createRegions(1, 1);
                    region = page.currentRegion();
                }
            }
        }
        
        // find or create target region
        
        if (region == null) {
            if (newPlot) {
                int n = page.numberOfRegions();
                for (int i=0; i<n; i++) {
                    PlotRegion r = page.region(i);
                    if (r.currentPlot() == null) {
                        region = r;
                        break;
                    }
                }
            } else {
                region = page.currentRegion();
            }
            if (region == null) {
                region = page.addRegion();
            }
        }
        
        // make region current and show page
        
        plotData.setPage(page);
        plotData.setRegion(region);
        
        page.setCurrentRegion(region);
        page.showPage();
        
        // plot
        
        plotData(Collections.singletonList(plotData), page, region, overlay);
    }
    
    public void refresh(DataChannelHandler channel, TimeWindow timeWindow) {
        for (PlotRegion region : getRegions(channel)) {
            refresh(region, timeWindow);
        }
    }
    
    public void refresh(PlotPage page, TimeWindow timeWindow) {
        int n = page.numberOfRegions();
        for (int i=0; i<n; i++) {
            refresh(page.region(i), timeWindow);
        }        
    }
    
    public void refresh(PlotRegion region, TimeWindow timeWindow) {
        
        try {
            if (((JComponent)region).getRootPane() == null) return;
        } catch (ClassCastException x) {
        }
        
        PlotPage page = null;
        Plotter plotter = region.currentPlot();
        if (plotter == null) return;
        List<Object> data = new ArrayList<>(plotter.getData());
        ListIterator<Object> it = data.listIterator();
        while (it.hasNext()) {
            Object plottedObject = it.next();
            try {
                Plot pd = (Plot) plottedObject;
                TimeWindow tw = pd.getTimeWindow();
                if (timeWindow == null) { // refreshing
                    if (!tw.isFixed()) {
                        page = pd.getPage();
                        pd.refresh();
                    }
                } else { // changing time window
                    page = pd.getPage();
                    pd = pd.getChannel().makePlotData(timeWindow);
                    pd.setPage(page);
                    pd.setRegion(region);
                    it.set(pd);
                }
            } catch (ClassCastException x) {
            }
        }
        if (page != null) {
            plotData(data, page, region, false);
        }
    }
    
//    public void close(PlotPage page) {
//        int n = page.numberOfRegions();
//        for (int i=0; i<n; i++) {
//            PlotRegion region = page.region(i);
//            Plotter plotter = region.currentPlot();
//            if (plotter != null) {
//                for (Object plottedObject : plotter.getData()) {
//                    try {
//                        Plot pd = (Plot) plottedObject;
//                        pd.getChannel().removePlotData(pd);
//                    } catch (ClassCastException x) {
//                    }
//                }
//            }
//        }
//    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private List<PlotRegion> getRegions(DataChannelHandler channel) {
        List<Plot> pdList = channel.getPlotDataAll();
        if (pdList.isEmpty()) {
            return Collections.emptyList();
        } else {
            List<PlotRegion> out = new ArrayList<>(pdList.size());
            for (Plot pd : pdList) {
                out.add(pd.getRegion());
            }
            return out;
        }
    }
    
    private void plotData(List<Object> data, PlotPage page, PlotRegion region, boolean overlay) {

        if (data.isEmpty()) return;

        Plotter plotter = region.currentPlot();
        if (plotter == null) {
            plotter = plotFactory.createPlotterFor(IBaseHistogram.class);
        } else if (!overlay) {
            plotter.clear();
        }

        IPlotterStyle style = plotterStyleFactory.createPlotterStyle();
        final IAxisStyle xAxisStyle = style.xAxisStyle();
        xAxisStyle.setParameter("type", "date");

        boolean commonExtent = data.size() > 1 && !overlay;
        if (commonExtent) {
            long tLow = Long.MAX_VALUE;
            long tHigh = 0L;
            for (Object datum : data) {
                try {
                    Plot pd = (Plot) datum;
                    tLow = Math.min(tLow, (long) pd.lowerExtent(0));
                    tHigh = Math.max(tHigh, (long) pd.upperExtent(0));
                    xAxisStyle.setParameter("lowerLimit", String.valueOf(tLow / 1000));
                    xAxisStyle.setParameter("upperLimit", String.valueOf(tHigh / 1000));
                } catch (ClassCastException x) {
                }
            }
        }

        boolean needRefresh = false;
        for (int i = 0; i < data.size(); i++) {
            Object datum = data.get(i);
            try {
                Plot pd = (Plot) datum;
                needRefresh = !pd.getTimeWindow().isFixed() || needRefresh;
                pd.setPage(page);
                pd.setRegion(region);
                pd.getChannel().addPlotData(pd);
                if (!commonExtent) {
                    xAxisStyle.setParameter("lowerLimit", String.valueOf(pd.lowerExtent(0) / 1000));
                    xAxisStyle.setParameter("upperLimit", String.valueOf(pd.upperExtent(0) / 1000));
                }
            } catch (ClassCastException x) {
            }
            plotter.plot(data.get(i), (i == 0 && !overlay) ? Plotter.NORMAL : Plotter.OVERLAY, style, "");
        }
        
        region.showPlot(plotter);
        
        if (needRefresh && autoRefresh != null) {
            autoRefresh.schedule(region);
        }
    }
    
    
}
