package org.lsst.ccs.plugin.jas3.trending;

import com.sun.jersey.api.client.WebResource;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.IOException;
import java.net.URL;
import java.util.Collection;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.BorderFactory;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JToolBar;
import org.freehep.application.mdi.PageContext;
import org.freehep.application.mdi.PageManager;
import org.freehep.application.studio.Plugin;
import org.freehep.jas.plugin.tree.FTree;
import org.freehep.jas.plugin.tree.FTreeNode;
import org.freehep.jas.plugin.tree.FTreeNodeAddedNotification;
import org.freehep.jas.plugin.tree.FTreeNodeRemovedNotification;
import org.freehep.jas.plugin.tree.FTreePath;
import org.freehep.jas.plugin.tree.FTreeProvider;
import org.freehep.jas.services.PlotPage;
import org.freehep.util.FreeHEPLookup;
import org.freehep.xml.menus.XMLMenuBuilder;
import org.lsst.ccs.localdb.statusdb.server.DataChannel;
import org.lsst.ccs.plugin.jas3.rest.LsstRestPlugin;
import org.lsst.ccs.plugin.jas3.trending.timeselection.TimeWindowComboBox;
import org.openide.util.Lookup;
import org.openide.util.LookupEvent;
import org.openide.util.LookupListener;
import org.xml.sax.SAXException;

/**
 * This plugin is responsible for listening for added or removed WebResources
 * registered by the Rest plugin. On detecting changes it will add them to the
 * JAS Tree in the control area.
 *
 * @author tonyj
 * @author onoprien
 */
public class LsstTrendingPlugin extends Plugin {
    
// -- Private parts : ----------------------------------------------------------

    private static final Logger logger = Logger.getLogger(LsstRestPlugin.class.getName());
    private TrendingPreferences prefs;
    
    private TimeWindowComboBox timeRangeSelector;
    private PlotMaker plotMaker;
    
    private volatile AutoRefresh autoRefresh;
    
    
// -- Initialization : ---------------------------------------------------------

    @Override
    protected void init() throws SAXException, IOException {
        
        logger.finest("LsstTrendingPlugin created");
        final FreeHEPLookup lookup = getApplication().getLookup();
        addDataServerListener(lookup);
        timeRangeSelector = addTrendingToolbar();
        plotMaker = new PlotMaker(getApplication(), timeRangeSelector);

        final TrendingService trendingService = new TrendingService() {
            @Override
            public void show(String[] path) {
                FTreeProvider treeProvider = (FTreeProvider) lookup.lookup(FTreeProvider.class);
                if (treeProvider != null) {
                    final FTree tree = treeProvider.tree();
                    FTreeNode node = tree.findNode(new FTreePath(path));
                    tree.adapterForClass(DataChannelHandler.class).doubleClick(node);
                }
            }
        };
        lookup.add(trendingService);
        prefs = new TrendingPreferences(this);
        lookup.add(prefs);
    }

//    @Override
//    protected void postInit() {
//        studio.getPageManager().addPageListener(new PageListener() {
//            public void pageChanged(PageEvent pe) {
//                if (pe.getID() == PageEvent.PAGECLOSED) {
//                    Component c = pe.getPageContext().getPage();
//                    try {
//                        PlotPage page = (PlotPage) c;
//                        plotMaker.close(page);
//                    } catch (ClassCastException x) {
//                    }
//                }
//            }
//        });
//    }

    /**
     * Listens for data server restful interfaces to be registered/unregistered.
     * Calls handleWebResource when a change occurs.
     */
    private void addDataServerListener(FreeHEPLookup lookup) {

        Lookup.Template template = new Lookup.Template(WebResource.class, "dataserver", null);
        final Lookup.Result result = lookup.lookup(template);
        handleWebResource(result.allInstances());
        LookupListener ll = new LookupListener() {
            @Override
            public void resultChanged(LookupEvent le) {
                handleWebResource(result.allInstances());
            }
        };
        result.addLookupListener(ll);
    }

    /**
     * Adds a tool bar for selection of time period for new trending plots
     *
     * @throws SAXException
     * @throws IOException
     */
    private TimeWindowComboBox addTrendingToolbar() throws SAXException, IOException {
        
        XMLMenuBuilder builder = getApplication().getXMLMenuBuilder();
        URL xml = getClass().getResource("Trending.menus");
        builder.build(xml);
        
        JToolBar toolbar = builder.getToolBar("trending");
        
        TimeWindowComboBox box = new TimeWindowComboBox(getApplication());
        box.setEnabled(box.getItemCount() > 0);

        JLabel boxLabel = new JLabel("Trending Period: ");
        toolbar.add(boxLabel, 0);
        toolbar.add(box, 1);  
        
        // Apply button
        
        JMenuBar bar = new JMenuBar();
        bar.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(1, 1, 1, 1), BorderFactory.createRaisedBevelBorder()));
        bar.setAlignmentY(Component.CENTER_ALIGNMENT);
        JMenu menu = new JMenu("Apply");
//        menu.setFont(boxLabel.getFont());
        bar.add(menu);
        JMenuItem item = new JMenuItem("All");
        item.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                PageManager pm = getApplication().getPageManager();
                PageContext selectedPage = pm.getSelectedPage();
                List pcList = pm.pages();
                for (Object o : pcList) {
                    try {
                        PlotPage page = (PlotPage) ((PageContext)o).getPage();
                        if (page != null) plotMaker.refresh(page, timeRangeSelector.getSelectedTimeWindow());
                    } catch (NullPointerException|ClassCastException x) {
                    }
                }
                selectedPage.requestShow();
            }
        });
        menu.add(item);
        item = new JMenuItem("Page");
        item.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                try {
                    PlotPage page = (PlotPage) getApplication().getPageManager().getSelectedPage().getPage();
                    if (page != null) plotMaker.refresh(page, timeRangeSelector.getSelectedTimeWindow());
                } catch (NullPointerException|ClassCastException x) {
                }
            }
        });
        menu.add(item);
        item = new JMenuItem("Region");
        item.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                try {
                    PlotPage page = (PlotPage) getApplication().getPageManager().getSelectedPage().getPage();
                    plotMaker.refresh(page.currentRegion(), timeRangeSelector.getSelectedTimeWindow());
                } catch (NullPointerException|ClassCastException x) {
                }
            }
        });
        menu.add(item);
        toolbar.add(bar,2);
        
        
        // Resresh button
        
        bar = new JMenuBar();
        bar.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(1, 1, 1, 1), BorderFactory.createRaisedBevelBorder()));
        bar.setAlignmentY(Component.CENTER_ALIGNMENT);
        menu = new JMenu("Refresh");
//        menu.setFont(boxLabel.getFont());
        bar.add(menu);
        item = new JMenuItem("All");
        item.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                PageManager pm = getApplication().getPageManager();
                PageContext selectedPage = pm.getSelectedPage();
                List pcList = pm.pages();
                for (Object o : pcList) {
                    try {
                        PlotPage page = (PlotPage) ((PageContext)o).getPage();
                        if (page != null) plotMaker.refresh(page, null);
                    } catch (NullPointerException|ClassCastException x) {
                    }
                }
                selectedPage.requestShow();
            }
        });
        menu.add(item);
        item = new JMenuItem("Page");
        item.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                try {
                    PlotPage page = (PlotPage) getApplication().getPageManager().getSelectedPage().getPage();
                    if (page != null) plotMaker.refresh(page, null);
                } catch (NullPointerException|ClassCastException x) {
                }
            }
        });
        menu.add(item);
        item = new JMenuItem("Region");
        item.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                try {
                    PlotPage page = (PlotPage) getApplication().getPageManager().getSelectedPage().getPage();
                    plotMaker.refresh(page.currentRegion(), null);
                } catch (NullPointerException|ClassCastException x) {
                }
            }
        });
        menu.add(item);
        menu.addSeparator();
        JCheckBoxMenuItem cbItem = new JCheckBoxMenuItem("Auto");
        cbItem.setSelected(false);
        cbItem.addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                if (e.getStateChange() == ItemEvent.SELECTED) {
                    autoRefresh = new AutoRefresh(LsstTrendingPlugin.this);
                    autoRefresh.start();
                } else if (e.getStateChange() == ItemEvent.DESELECTED) {
                    if (autoRefresh != null) {
                        autoRefresh.stop();
                        autoRefresh = null;
                    }
                }
            }
        });
        menu.add(cbItem);
        toolbar.add(bar,3);
        
        getApplication().addToolBar(toolbar, toolbar.getName());
        return box;
    }

    private void handleWebResource(Collection<WebResource> allInstances) {
        for (WebResource service : allInstances) {
            handleDataServer(service);
            return;
        }
        handleDataServer(null);
    }

    private void handleDataServer(WebResource service) {
        FreeHEPLookup lookup = getApplication().getLookup();
        FTreeProvider treeProvider = (FTreeProvider) lookup.lookup(FTreeProvider.class);
        if (treeProvider != null) {
            final FTree tree = treeProvider.tree();

            if (service == null) {
                logger.log(Level.INFO, "WebResource removed");
                FTreeNode root = tree.root();
                // Clear old tree data
                for (FTreeNode node : (Collection<FTreeNode>) (root.childNodes())) {
                    tree.treeChanged(new FTreeNodeRemovedNotification(getApplication(), node.path()));
                }
            } else {
                logger.log(Level.INFO, "WebResource registered {0}", service);

                DataChannel.DataChannelList channelList = (DataChannel.DataChannelList) lookup.lookup(DataChannel.DataChannelList.class);

                DataChannelTreeAdapter treeAdapter = new DataChannelTreeAdapter(this, service);
                treeProvider.treeNodeAdapterRegistry().registerNodeAdapter(treeAdapter, DataChannelHandler.class);
                for (DataChannel chan : channelList.list) {
                    tree.treeChanged(new FTreeNodeAddedNotification(getApplication(), chan.getPathAsString(), new DataChannelHandler(chan, treeAdapter)));
                }
            }
        }
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    public TrendingPreferences getPreferences() {
        return prefs;
    }
    
    public PlotMaker getPlotMaker() {
        return plotMaker;
    }
    
    public AutoRefresh getAutoRefresh() {
        return autoRefresh;
    }
    
}
