package org.lsst.ccs.plugin.jas3.trending;

import com.sun.jersey.api.client.WebResource;
import hep.aida.IDataPoint;
import hep.aida.IDataPointSet;
import hep.aida.ref.histogram.DataPointSet;
import java.lang.ref.WeakReference;
import java.util.*;
import javax.ws.rs.core.MediaType;
import org.freehep.jas.services.PlotPage;
import org.freehep.jas.services.PlotRegion;
import org.freehep.jas.services.Plotter;
import org.lsst.ccs.localdb.statusdb.server.ChannelMetaData;
import org.lsst.ccs.localdb.statusdb.server.Data;
import org.lsst.ccs.localdb.statusdb.server.DataChannel;
import org.lsst.ccs.localdb.statusdb.server.TrendingData;
import org.lsst.ccs.localdb.statusdb.server.TrendingResult;
import org.lsst.ccs.plugin.jas3.trending.timeselection.TimeWindow;

/**
 * Handles retrieval and caching of trending data associated with a specific {@link DataChannel}.
 * Also holds references to plotted data point sets for this channel.
 *
 * @author onoprien
 */
public class DataChannelHandler {
    
// -- Private parts : ----------------------------------------------------------
    
    private final DataChannel channel;
    private final DataChannelTreeAdapter adapter;
    private ArrayList<WeakReference<Plot>> plots;
    
    
// -- Construction and initialization : ----------------------------------------
    
    DataChannelHandler(DataChannel dataChannel, DataChannelTreeAdapter treeAdapter) {
        channel = dataChannel;
        adapter = treeAdapter;
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    /** Returns the path associated with the underlying {@link DataChannel}. */
    public String getPath() {
        return channel.getPathAsString();
    }
    
    /** Returns the underlying <tt>DataChannel</tt>. */
    public DataChannel getDataChannel() {
        return channel;
    }
    
    /**
     * Returns a list of plotted data point sets for this channel.
     * The list is not modifiable. 
     * Calling this method purges the list of plotted dataset, removing those associated with deleted regions.
     */
    public synchronized List<Plot> getPlotDataAll() {
        if (plots == null) return Collections.emptyList();
        System.gc();
        boolean purge = false;
        ArrayList<Plot> out = new ArrayList<>(plots.size());
        ListIterator<WeakReference<Plot>> it = plots.listIterator();
        while (it.hasNext()) {
            WeakReference<Plot> ref = it.next();
            Plot pd = ref.get();
            if (pd == null || !isDataPlotted(pd)) {
                it.remove();
                purge = true;
            } else {
                out.add(pd);
            }
        }
        if (out.isEmpty()) {
            plots = null;
        }
        if (purge) {
            purge();
        }
        return out;
    }
    
    /**
     * Returns an existing plotted data point set for this channel.
     * If there are several plots for this channel, returns the data point set with 
     * the specified time window, if it exist. Otherwise, returns one of the data point sets
     * plotted on pages with minimal number of regions. If there are no plots for this
     * channel, returns <tt>null</tt>.
     */
    public Plot findPlotData(TimeWindow timeWindow) {
        if (plots == null) return null;
        Plot out = null;
        for (Plot pd : getPlotDataAll()) {
            if (pd.getRegion().currentPlot().getData().size() == 1) {
                if (pd.getTimeWindow() == timeWindow) {
                    return pd;
                } else if (out == null || (pd.getPage().numberOfRegions() < out.getPage().numberOfRegions())) {
                    out = pd;
                }
            }
        }
        return out;
    }
    
// -- Operations : -------------------------------------------------------------
    
    /**
     * Creates and returns a data point set for the specified time window.
     */
    public Plot makePlotData(TimeWindow timeWindow) {
        long begin = timeWindow.getLowerEdge();
        long end = timeWindow.getUpperEdge();
        PlotData dps = makeDataPointSet(begin, end);
        Plot pd = new Plot(dps, timeWindow, this);
        pd.setTimeRange(begin, end);
        return pd;
    }
    
    /**
     * Prepares the specified Plot object for prepareRefresh by constructing updated DataPointSet.
     * Subsequent call to {@link PlotData#update) will use this set to update plotted data.
     * @param plot Plot data to be prepared for the prepareRefresh.
     * @return true if the prepareRefresh will change the data.
     */
    public boolean refresh(Plot plot) {
        TimeWindow timeWindow = plot.getTimeWindow();
        if (timeWindow.isFixed()) return false;
        long begin = timeWindow.getLowerEdge();
        long end = timeWindow.getUpperEdge();
        plot.prepareRefresh(makeDataPointSet(begin, end), begin, end);
        return true;
    }
    
    /**
     * Adds the specified data point set to the list of plotted data for this channel.
     */
    public synchronized boolean addPlotData(Plot plot) {
        boolean out = true;
        if (plots == null) {
            plots = new ArrayList<>(1);
        } else {
            boolean purge = false;
            ListIterator<WeakReference<Plot>> it = plots.listIterator();
            while (it.hasNext()) {
                WeakReference<Plot> ref = it.next();
                Plot pd = ref.get();
                if (pd == null) {
                    it.remove();
                    purge = true;
                } else if (pd == plot) {
                    out = false;
                }
            }
            if (purge) {
                purge();
            }
        }
        if (out) {
            plots.add(new WeakReference<>(plot));
        }
        return out;
    }
    
    /**
     * Removes the specified data point set from the list of plotted data for this channel.
     */
    public synchronized boolean removePlotData(Plot plot) {
        if (plots == null) return false;
        boolean out = false;
        boolean purge = false;
        ListIterator<WeakReference<Plot>> it = plots.listIterator();
        while (it.hasNext()) {
            WeakReference<Plot> ref = it.next();
            Plot pd = ref.get();
            if (pd == null) {
                it.remove();
                purge = true;
            } else if (pd == plot) {
                it.remove();
                out = true;
            }
        }
        if (out || purge) {
            purge();
        }
        if (plots.isEmpty()) {
            plots = null;
        }
        return out;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private PlotData makeDataPointSet(long begin, long end) {
        
        WebResource resource = adapter.getResource().path("data").path(String.valueOf(channel.getId()));
        resource = resource.queryParam("t1", String.valueOf(begin)).queryParam("t2", String.valueOf(end));
        TrendingPreferences pref = adapter.getPlugin().getPreferences();
        int nBins = pref.nBins();
        if (nBins > 0) {
            if (!pref.useRawData()) {
                resource = resource.queryParam("flavor", "stat");
            }
            resource = resource.queryParam("n", Integer.toString(nBins));
        } else {
            resource = resource.queryParam("flavor", pref.useRawData() ? "raw" : "stat");
        }
        Data data = resource.accept(MediaType.TEXT_XML).get(Data.class);
        TrendingResult result = data.getTrendingResult();
        
        DataPointSet dps = new DataPointSet("", "", 2);
        TrendingData[] dataArray = result.getTrendingDataArray();
        if (dataArray != null) {
            boolean drawRms = adapter.getPlugin().getPreferences().drawRms();
            for (TrendingData d : dataArray) {
                IDataPoint p = dps.addPoint();
                p.coordinate(0).setValue(d.getAxisvalue().getValue() / 1000.);
                p.coordinate(1).setValue(d.getValue("value"));
                if (drawRms) {
                    double rms = d.getValue("rms");
                    if (!Double.isNaN(rms)) {
                        p.coordinate(1).setErrorMinus(rms);
                        p.coordinate(1).setErrorPlus(rms);
                    }
                }
            }
        }
        PlotData pd = new PlotData(dps);
        
        List<ChannelMetaData> metaList = result.getChannelMetadata();
        HashMap<String,DataPointSet> metaMap = new HashMap<>();
        for (ChannelMetaData meta : metaList) {
            try {
                dps = metaMap.get(meta.getName());
                if (dps == null) {
                    dps = new DataPointSet("", "", 2);
                    metaMap.put(meta.getName(), dps);
                }
                double value = Double.parseDouble(meta.getValue());
                long start = meta.getTstart();
                if (start < begin) start = begin;
                IDataPoint p = dps.addPoint();
                p.coordinate(0).setValue(start / 1000.);
                p.coordinate(1).setValue(value);
                long stop = meta.getTstop();
                if (stop > end || stop == -1) stop = end;
                p = dps.addPoint();
                p.coordinate(0).setValue(stop / 1000.);
                p.coordinate(1).setValue(value);
            } catch (NumberFormatException x) {
            }
        }
        
        DataPointSet alarmLow = metaMap.get("alarmLow");
        DataPointSet alarmHigh = metaMap.get("alarmHigh");
        if (alarmLow != null || alarmHigh != null) {
            pd.setAlarmLevels(new IDataPointSet[] {alarmLow, alarmHigh});
        }
        
        DataPointSet warningLow = metaMap.get("warningLow");
        DataPointSet warningHigh = metaMap.get("warningHigh");
        if (warningLow != null || warningHigh != null) {
            pd.setAlarmLevels(new IDataPointSet[] {warningLow, warningHigh});
        }
        
        return pd;
    }
    
    private boolean isDataPlotted(Plot plotData) {
        
        PlotPage page = plotData.getPage();
        PlotRegion region = plotData.getRegion();
        if (page == null || region == null) return false;
        
        // check whether the page is still managed by JAS3
        
//        Studio studio = (Studio) Studio.getApplication();
//        PageManager pm = studio.getPageManager();
//        if (!pm.pages().contains(page)) {
//            return false;
//        }
        
        // check whether the region references this data
        
        Plotter p = region.currentPlot();
        if (p == null) return false;
        List dataList = p.getData();
        if (dataList == null || dataList.indexOf(plotData) == -1) return false;
        
        // check whether the page still contains this region
        
        int n = page.numberOfRegions();
        for (int i=0; i<n; i++) {
            PlotRegion r = page.region(i);
            if (r == region) {
                return true;
            }
        }
        return false;
    }
    
    private void purge() {
//        System.out.println("purging");
    }

    
}
