package org.lsst.ccs.plugin.jas3.trending.timeselection;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Calendar;
import java.util.Date;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.SpinnerDateModel;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingUtilities;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 * GUI panel for entering a custom time window.
 *
 * @author onoprien
 */
public class TimeWindowPanel extends JPanel {

// -- Private parts : ----------------------------------------------------------
    
    private final int MAX_YEARS = 50;
    private final int SECONDS_IN_YEAR = 365 * 24 * 60 * 60;
    private final int SECONDS_IN_DAY = 24 * 60 * 60;
    private final int SECONDS_IN_HOUR = 60 * 60;
    private final int SECONDS_IN_MINUTE = 60;
    
    private final int VSPACE = 5;
    private final int HSPACE = 10;
    
    private final TimePanel fromPanel;
    private final TimePanel toPanel;
    private final JTextField nameField;
    private final JCheckBox saveCheck;
    
    static private TimeWindow range;
    static private JButton okButton;

    
// -- Construction : -----------------------------------------------------------
    
    private TimeWindowPanel() {
        
        setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
        
        fromPanel = new TimePanel(" Starting ");
        add(fromPanel);
        
        add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));
        
        toPanel = new TimePanel(" Ending ");
        add(toPanel);
        
        add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));
        
        Box saveBox = Box.createVerticalBox();
        saveBox.setBorder(BorderFactory.createCompoundBorder(
                    BorderFactory.createTitledBorder(" Remember As "), 
                    BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));
        nameField = new JTextField();
        nameField.setColumns(30);
        nameField.addCaretListener(new CaretListener() {
            public void caretUpdate(CaretEvent e) {
                saveCheck.setEnabled(!nameField.getText().trim().isEmpty());
            }
        });
        saveBox.add(nameField);
        saveBox.add(Box.createRigidArea(new Dimension(0, VSPACE)));
        saveCheck = new JCheckBox("Remember between sessions");
        saveBox.add(saveCheck);
        add(saveBox);

        if (range == null) {
            fromPanel.setTime("now - "+ SECONDS_IN_DAY);
            toPanel.setTime("now");
            saveCheck.setEnabled(false);
        } else {
            fromPanel.setTime(range.getLowerEdgeString());
            toPanel.setTime(range.getUpperEdgeString());
            nameField.setText(range.getName());
            saveCheck.setSelected(range.isPersistent());
            saveCheck.setEnabled(true);
        }
    }


// -- Displaying the GUI : -----------------------------------------------------
    
    /**
     * Opens trending time window creation/editing dialog.
     * 
     * @param parent Parent component.
     * @param seed Time window to be edited, or <tt>null</tt> if a new time window should be created.
     * @return Modified or created time window, or <tt>null</tt> if the dialog is canceled.
     */
    static public TimeWindow editTimeWindow(JComponent parent, TimeWindow seed) {
        range = seed;
        okButton = new JButton("  OK  ");
        final TimeWindowPanel tsp = new TimeWindowPanel();
        okButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                tsp.getRange();
                Window w = SwingUtilities.getWindowAncestor(tsp);
                if (w != null) w.setVisible(false);
            }
        });
        JButton cancelButton = new JButton("Cancel");
        cancelButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                range = null;
                Window w = SwingUtilities.getWindowAncestor(tsp);
                if (w != null) w.setVisible(false);
            }
        });
        JOptionPane.showOptionDialog(parent, tsp, "Select time range", JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE, null, new Object[] {okButton,cancelButton}, null);
        okButton = null;
        return range;
    }

    
// -- Local methods : ----------------------------------------------------------
    
    private void checkValidy() {
        okButton.setEnabled(fromPanel.isDataValid() && toPanel.isDataValid());
    }
    
    private TimeWindow getRange() {
        String name = nameField.getText().trim();
        boolean persistent = !name.isEmpty() && saveCheck.isSelected();
        String start =  fromPanel.getTime();
        String end = toPanel.getTime();
        if (range == null) {
            range = new TimeWindow(name, start, end, persistent);
        } else {
            range.setName(name);
            range.setLowerEdge(start);
            range.setUpperEdge(end);
            range.setPersistent(persistent);
        }
        return range;
    }
    
    
// -- Class for single time point selection : ----------------------------------
    
    private class TimePanel extends JPanel {
        
        final JTextField _field;
        final ButtonGroup _group;
        final JRadioButton _absButton;
        final JRadioButton _relButton;
        final JSpinner _timeSpinner;
        final JSpinner _yearSpinner, _daySpinner, _hourSpinner, _minSpinner, _secSpinner;
        
        
        final SpinnerDateModel _timeSpinnerModel;
        final SpinnerNumberModel _yearSpinnerModel, _daySpinnerModel, _hourSpinnerModel, _minSpinnerModel, _secSpinnerModel;
        
        boolean _listen;
        
        ChangeListener _absListener = new ChangeListener() {
            public void stateChanged(ChangeEvent e) {
                if (_listen) {
                    _listen = false;
                    updateFromAbsolute();
                    _listen = true;
                }
            }
        };
        ChangeListener _relListener = new ChangeListener() {
            public void stateChanged(ChangeEvent e) {
                if (_listen) {
                    _listen = false;
                    updateFromRelative();
                    _listen = true;
                }
            }
        };
        
        TimePanel(String title) {
            _listen = false;
            
            setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
            setBorder(BorderFactory.createCompoundBorder(
                    BorderFactory.createTitledBorder(title), 
                    BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));
            
            _field = new JTextField();
            _field.addCaretListener(new CaretListener() {
                public void caretUpdate(CaretEvent e) {
                    if (_listen) {
                        _listen = false;
                        updateFromField();
                        _listen = true;
                    }
                }
                
            });
            add(_field);
            
            add(Box.createRigidArea(new Dimension(0, VSPACE)));
            
            Box box = Box.createHorizontalBox();
            add(box);
            _group = new ButtonGroup();
            _absButton = new JRadioButton("Fixed: ");
            _absButton.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    if (_listen) {
                        _listen = false;
                        enableRelative(false);
                        updateFromAbsolute();
                        _field.setForeground(Color.BLACK);
                        _listen = true;
                    }
                }
            });
            _group.add(_absButton);
            box.add(_absButton);
            box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
            _timeSpinnerModel = new SpinnerDateModel(new Date(), null, null, Calendar.HOUR);
            _timeSpinner = new JSpinner(_timeSpinnerModel);
            _timeSpinner.setEditor(new JSpinner.DateEditor(_timeSpinner, " "+ TimeWindow.DATE_PATTERN +" "));
            _timeSpinner.addChangeListener(_absListener);
            box.add(_timeSpinner);
            box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
            box.add(Box.createHorizontalGlue());
            
            add(Box.createRigidArea(new Dimension(0, VSPACE)));
            
            box = Box.createHorizontalBox();
            add(box);
            _relButton = new JRadioButton();
            _relButton.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    if (_listen) {
                        _listen = false;
                        enableRelative(true);
                        updateFromRelative();
                        _field.setForeground(Color.BLACK);
                        _listen = true;
                    }
                }
            });
            _group.add(_relButton);
            box.add(_relButton);
            _yearSpinnerModel = new SpinnerNumberModel(0, 0, MAX_YEARS, 1);
            _yearSpinner = new JSpinner(_yearSpinnerModel);
            _yearSpinner.addChangeListener(_relListener);
            box.add(_yearSpinner);
            box.add(Box.createRigidArea(new Dimension(HSPACE/2, 0)));
            box.add(new JLabel("years"));
            box.add(Box.createRigidArea(new Dimension(HSPACE/2, 0)));
            _daySpinnerModel = new SpinnerNumberModel(0, 0, 999, 1);
            _daySpinner = new JSpinner(_daySpinnerModel);
            _daySpinner.addChangeListener(_relListener);
            box.add(_daySpinner);
            box.add(Box.createRigidArea(new Dimension(HSPACE/2, 0)));
            box.add(new JLabel("days"));
            box.add(Box.createRigidArea(new Dimension(HSPACE/2, 0)));
            _hourSpinnerModel = new SpinnerNumberModel(0, 0, 99, 1);
            _hourSpinner = new JSpinner(_hourSpinnerModel);
            _hourSpinner.addChangeListener(_relListener);
            box.add(_hourSpinner);
            box.add(Box.createRigidArea(new Dimension(HSPACE/2, 0)));
            box.add(new JLabel("hours"));
            box.add(Box.createRigidArea(new Dimension(HSPACE/2, 0)));
            _minSpinnerModel = new SpinnerNumberModel(0, 0, 99, 1);
            _minSpinner = new JSpinner(_minSpinnerModel);
            _minSpinner.addChangeListener(_relListener);
            box.add(_minSpinner);
            box.add(Box.createRigidArea(new Dimension(HSPACE/2, 0)));
            box.add(new JLabel("minutes"));
            box.add(Box.createRigidArea(new Dimension(HSPACE/2, 0)));
            _secSpinnerModel = new SpinnerNumberModel(0, 0, 9999, 1);
            _secSpinner = new JSpinner(_secSpinnerModel);
            _secSpinner.addChangeListener(_relListener);
            box.add(_secSpinner);
            box.add(Box.createRigidArea(new Dimension(HSPACE/2, 0)));
            box.add(new JLabel("seconds  in the past"));
            box.add(Box.createRigidArea(new Dimension(HSPACE, 0)));
            box.add(Box.createHorizontalGlue());
            
            add(Box.createRigidArea(new Dimension(0, VSPACE)));
            
            _listen = true;
        }
        
        boolean isDataValid() {
            return _field.getForeground() == Color.BLACK;
        }
        
        String getTime() {
            if (_absButton.isSelected()) {
                updateFromAbsolute();
            } else {
                updateFromRelative();
            }
            return _field.getText().trim();
        }
        
        void setTime(String time) {
            _field.setText(time);
            updateFromField();
        }
        
        private void setValid(boolean isValid) {
            if (isValid) {
                _field.setForeground(Color.BLACK);
            } else {
                _field.setForeground(Color.RED);
            }
            TimeWindowPanel.this.checkValidy();
        }
        
        private void updateFromAbsolute() {
            _field.setText(TimeWindow.DATE_FORMAT.format(_timeSpinnerModel.getDate()));
        }
        
        private void updateFromRelative() {
            int seconds = SECONDS_IN_YEAR * _yearSpinnerModel.getNumber().intValue() +
                          SECONDS_IN_DAY * _daySpinnerModel.getNumber().intValue() +
                          SECONDS_IN_HOUR * _hourSpinnerModel.getNumber().intValue() +
                          SECONDS_IN_MINUTE * _minSpinnerModel.getNumber().intValue() +
                          _secSpinnerModel.getNumber().intValue();
            _field.setText("now"+ (seconds > 0 ? (" - "+ seconds) : ""));
        }
        
        private void updateFromField() {
            String data = _field.getText();
            try {
                long edge = TimeWindow.parseEdge(data);
                if (edge > 0L) {
                    Date date = new Date(edge);
                    _timeSpinnerModel.setValue(date);
                    enableRelative(false);
                } else {
                    long seconds = -edge/1000;
                    _yearSpinnerModel.setValue(seconds / SECONDS_IN_YEAR);
                    seconds %= SECONDS_IN_YEAR;
                    _daySpinnerModel.setValue(seconds / SECONDS_IN_DAY);
                    seconds %= SECONDS_IN_DAY;
                    _hourSpinnerModel.setValue(seconds / SECONDS_IN_HOUR);
                    seconds %= SECONDS_IN_HOUR;
                    _minSpinnerModel.setValue(seconds / SECONDS_IN_MINUTE);
                    seconds %= SECONDS_IN_MINUTE;
                    _secSpinnerModel.setValue(seconds);
                    enableRelative(true);
                }
                setValid(true);
            } catch (IllegalArgumentException x) {
                setValid(false);
            }
        }
                
        private void enableRelative(boolean relative) {
            _timeSpinner.setEnabled(!relative);
            _yearSpinner.setEnabled(relative);
            _daySpinner.setEnabled(relative);
            _hourSpinner.setEnabled(relative);
            _minSpinner.setEnabled(relative);
            _secSpinner.setEnabled(relative);
            if (relative) {
                _relButton.setSelected(true);
            } else {
                _absButton.setSelected(true);
            }
        }
       
    }
        
}
