package org.lsst.ccs.plugin.jas3.trending;

import java.lang.ref.WeakReference;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;
import javax.swing.SwingUtilities;
import org.freehep.application.mdi.PageContext;
import org.freehep.application.mdi.PageManager;
import org.freehep.application.studio.Studio;
import org.freehep.jas.services.PlotPage;
import org.freehep.jas.services.PlotRegion;
import org.freehep.jas.services.Plotter;

/**
 * Handles auto-refreshing trending plots.
 * The class is thread-safe, all public methods can be called from any thread.
 *
 * @author onoprien
 */
final public class AutoRefresh {

// -- Private parts : ----------------------------------------------------------
    
    private final PlotMaker plotMaker;
    
    private final DelayQueue<RegionUpdate> queue = new DelayQueue<>();
    private final Set<PlotRegion> set = Collections.synchronizedSet(Collections.newSetFromMap(new WeakHashMap<PlotRegion,Boolean>()));
    private final Runnable dataFetcher = new DataFetcher();
    private Thread dataFetcherThread;
    
    private volatile long[] delayBounds;
    

// -- Construction and initialization : ----------------------------------------
    
    public AutoRefresh(LsstTrendingPlugin plugin) {
        plotMaker = plugin.getPlotMaker();
        TrendingPreferences pref = plugin.getPreferences();
        setDelay(pref.refreshMin(), pref.refreshMax(), TimeUnit.SECONDS);
    }
    
// -- Operation : --------------------------------------------------------------
    
    /**
     * Sets the allowed range of prepareRefresh frequency.
     * 
     * @param min Minimum prepareRefresh interval.
     * @param max Maximum prepareRefresh interval.
     * @param unit Time unit used to specify minimum and maximum intervals.
     */
    public void setDelay(long min, long max, TimeUnit unit) {
        if (min > max) {
            throw new IllegalArgumentException();
        } else if (min == max) {
            delayBounds = new long[] {TimeUnit.MILLISECONDS.convert(min, unit)};
        } else {
            delayBounds = new long[] {TimeUnit.MILLISECONDS.convert(min, unit), TimeUnit.MILLISECONDS.convert(max, unit)};
        }
    }
    
    /**
     * Starts auto-refreshing.
     * Compiles a list of all regions that need refreshing, puts them in the queue,
     * and starts data fetcher thread.
     */
    public synchronized void start() {
        
        stop();
        
        Studio studio = (Studio) Studio.getApplication();
        PageManager pm = studio.getPageManager();
        List pcList = pm.pages();
        for (Object o : pcList) {
            try {
                PlotPage page = (PlotPage) ((PageContext) o).getPage();
                for (int i=0; i<page.numberOfRegions(); i++) {
                    PlotRegion region = page.region(i);
                    schedule(region);
                }
            } catch (NullPointerException | ClassCastException x) {
            }
        }

        dataFetcherThread = new Thread(dataFetcher, "Auto-refresh");
        int priority = Math.min(Thread.currentThread().getPriority()-1, Thread.NORM_PRIORITY-1);
        dataFetcherThread.setPriority(Math.max(priority, Thread.MIN_PRIORITY));
        dataFetcherThread.start();
        plotMaker.setAutorefresh(this);
    }
    
    /**
     * Stops auto-refreshing.
     * Clears the auto-refreshing tasks queue and stops data fetcher thread.
     */
    public synchronized void stop() {
        plotMaker.setAutorefresh(null);
        if (dataFetcherThread != null) {
            dataFetcherThread.interrupt();
            dataFetcherThread = null;
        }
        queue.clear();
    }
    
    
// -- Scheduling updates : -----------------------------------------------------
    
    public void schedule(PlotRegion region) {

        
        if (region == null) return;
        Plotter p = region.currentPlot();
        if (p == null) return;
        
        PlotPage page = null;
        long delay = Long.MAX_VALUE;
        for (Object o : p.getData()) {
            try {
                Plot pd = (Plot) o;
                if (!pd.getTimeWindow().isFixed()) {
                    if (page == null) { // check if the region is still active
                        page = pd.getPage();
                        boolean isRegionDisplayed = false;
                        for (int i = 0; i < page.numberOfRegions(); i++) {
                            PlotRegion r = page.region(i);
                            if (r == region) {
                                isRegionDisplayed = true;
                                break;
                            }
                        }
                        if (!isRegionDisplayed) return;
                    }
                    delay = Math.min(delay, getDelay(pd));
                }
                
            } catch (ClassCastException x) {
            }
        }
        
        schedule(region, delay, TimeUnit.MILLISECONDS);
    }
    
    public void schedule(PlotRegion region, long delay, TimeUnit unit) {
        RegionUpdate ru = new RegionUpdate(TimeUnit.MILLISECONDS.convert(delay, unit), region);
        synchronized (set) {
            if (set.add(region)) {
                queue.add(ru);
            }
        }
    }
    
    
// -- Class to hold reference to updatable region : ----------------------------
    
    /**
     * Class to hold reference to a region that can be refreshed.
     * Note that natural ordering and hashCode() are inconsistent with equals - do not use in maps.
     */
    static private class RegionUpdate implements Delayed {
        
        long updateTime;
        WeakReference<PlotRegion> region;
        
        RegionUpdate(long delay, PlotRegion region) {
            this.updateTime = delay + System.currentTimeMillis();
            this.region = new WeakReference(region);
        }

        @Override
        public long getDelay(TimeUnit unit) {
            return unit.convert(updateTime - System.currentTimeMillis(), TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed o) {
            RegionUpdate other = (RegionUpdate) o;
            long delta = updateTime - other.updateTime;
            return delta<0L ? -1 : (delta>0 ? 1:0);
        }
        
        PlotRegion getRegion() {
            return region.get();
        }

        @Override
        public boolean equals(Object obj) {
            return obj instanceof RegionUpdate ? ((RegionUpdate)obj).region.get() == region : false;
        }
        
    }
    
// -- Code to run on data fetcher thread : -------------------------------------
    
    private class DataFetcher implements Runnable {

        public void run() {
            while (true) {
                try {
                    RegionUpdate ru = queue.take();
                    PlotRegion region = ru.getRegion();
                    set.remove(region);
                    if (region != null) {
                        Plotter plotter = region.currentPlot();
                        if (plotter != null) {
                            boolean needUpdate = false;
                            for (Object plottedObject : plotter.getData()) {
                                try {
                                    Plot pd = (Plot) plottedObject;
                                    needUpdate = pd.refresh() || needUpdate;
                                } catch (ClassCastException x) {
                                }
                            }
                            if (needUpdate) {
                                SwingUtilities.invokeLater(new Runnable() {
                                    public void run() {
                                        plotMaker.refresh(region, null);
                                    }
                                });
                            }
                        }
                    }
                } catch (InterruptedException x) {
                    break;
                } catch (Throwable x) {
                }
            }
        }

    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private long getDelay(Plot pd) {
        if (delayBounds.length == 1) {
            return delayBounds[0];
        } else {
            long delay = Math.round(pd.getGranularity()*1.1);
            return Math.min(delayBounds[1], Math.max(delay, delayBounds[0]));
        }
    }
    
}
