package org.lsst.ccs.plugin.jas3.trending.timeselection;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Comparator;
import java.util.Date;

/**
 * Time window for use by trending plots.
 *
 * @author turri
 */
public class TimeSelection {
    
// -- Public fields : ----------------------------------------------------------
    
    static public final String DATE_PATTERN = "MM/dd/yyyy HH:mm:ss";
    static public final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat(DATE_PATTERN);
    static private final String DELIMITER_RE = "\\$";
    static private final String DELIMITER = "$";
    
    /** Time window name. */
    public String name;
    /** String that defines time window start. */
    public String lowerEdgeString;
    /** String that defines time window end. */
    public String upperEdgeString;
    
// -- Private parts : ----------------------------------------------------------
    
    private boolean persist; // true if this time window should be persisted between sessions
    private long lastUsed; // time when this window was last selected
    
// -- Construction : -----------------------------------------------------------
    
    public TimeSelection(String name, String start, String end, boolean persistent) {
        this.name = name;
        lowerEdgeString = start;
        upperEdgeString = end;
        persist = persistent;
        lastUsed = System.currentTimeMillis();
    }
    
// -- Setters and getters : ----------------------------------------------------
    
    /** Returns the current value of this window start time in milliseconds. */
    public long getLowerEdge() {
        return getEdge(lowerEdgeString);
    }
    
    /** Returns the current value of this window end time in milliseconds. */
    public long getUpperEdge() {
        return getEdge(upperEdgeString);
    }
    
    private long getEdge(String edgeString) {
        try {
            Date d = DATE_FORMAT.parse(edgeString);
            return d.getTime();
        } catch (ParseException x) {
        }
        try {
            return (long)(1000*TimeSelectionUtils.getSecondsForString(edgeString));        
        } catch (Throwable t) {
            throw new RuntimeException(t);
        }
    }
    
    /** Returns <tt>true</tt> if this time window should be persisted between sessions. */
    public boolean isPersistent() {
        return persist;
    }
    
    /** Sets the flag that indicate whether or not this time window should be persisted between sessions. */
    public void setPersistent(boolean persistent) {
        persist = persistent;
    }
    
    /** Returns the time in milliseconds when this time window was last selected. */
    public long getLastUseTime() {
        return lastUsed;
    }
    
    /** Sets the time when this time window was last selected to the current system time. */
    public void touch() {
        lastUsed = System.currentTimeMillis();
    }

    /** Returns user-readable string representation of this time window. */
    @Override
    public String toString() {
        return name +" : "+ lowerEdgeString +" through "+ upperEdgeString;
    }
    
    /** Returns string representation of this time window used for saving it in application properties. */
    public String toCompressedString() {
        return name + DELIMITER + lowerEdgeString + DELIMITER + upperEdgeString;
    }
    
    /** Parses string representation of this time window used for saving it in application properties. */
    static public TimeSelection parseCompressedString(String s) {
        String[] tokens = s.split(DELIMITER_RE);
        try {
            TimeSelection ts = new TimeSelection(tokens[0], tokens[1], tokens[2], true);
            ts.getLowerEdge();
            ts.getUpperEdge();
            return ts;
        } catch (RuntimeException x) {
            throw new IllegalArgumentException(x);
        }
    
    }
    
// -- Comparators : ------------------------------------------------------------
    
    /** Returns comparator that compares time windows by names. */
    static public Comparator<TimeSelection> compareByName() {
        return new Comparator<TimeSelection>() {
            public int compare(TimeSelection o1, TimeSelection o2) {
                return o1.name.compareTo(o2.name);
            }
        };
    }
    
    /** Returns comparator that compares time windows by last use times, most recent first. */
    static public Comparator compareByTime() {
        return new Comparator() {
            public int compare(Object o1, Object o2) {
                return (int) Math.signum(((TimeSelection)o2).getLastUseTime() - ((TimeSelection)o1).getLastUseTime());
            }
        };
    }
    
    
}
