package org.lsst.ccs.plugin.jas3.rest;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.ws.rs.core.MediaType;
import org.freehep.application.studio.Studio;
import org.freehep.util.FreeHEPLookup;
import org.lsst.ccs.localdb.statusdb.server.DataChannel;

/**
 * This class is used to connect to a restful service. It can be run on a
 * background thread and does not directly interface to the GUI.
 *
 * @author tonyj
 */
class RestConnector implements Runnable, PropertyChangeListener {

    private static final Logger logger = Logger.getLogger(RestConnector.class.getName());
    private final RestPreferences prefs;
    private final Client client;
    private DataChannel.DataChannelList channelList;
    private final Studio studio;
    private WebResource service;

    RestConnector(Studio studio, RestPreferences restPrefs) {
        this.studio = studio;
        this.prefs = restPrefs;
        ClientConfig config = new DefaultClientConfig();
        client = Client.create(config);
        initialize();
    }

    private void initialize() {
        prefs.addPropertyChangeListener(this);
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        // Fired if the RestPreferences URL has been changed.
        Thread t = new Thread(this);
        t.start();
    }
    
    /**
     * The run method is called to establish a new connection to the rest
     * server, and register the channels obtained so they can be displayed to
     * the user. It can also be invoked if the rest URL is changed, in which
     * case it discards the old channel list if any, and obtains a new one.
     * 
     * Once obtained both the WebResource (service) and the channel list are 
     * registered with the lookup service, so that any plugin which wants to 
     * react to the change can do so.
     */
    @Override
    public synchronized void run() {
        final FreeHEPLookup lookup = studio.getLookup();
        
        if (service != null) {
            logger.log(Level.INFO, "Discarding old channel list");
            lookup.remove(channelList);
            lookup.remove(service,"dataserver");
            service = null;
        }
        
        String restFullUrl = prefs.getRestURL();
        logger.log(Level.INFO, "Connecting to rest server {0}", restFullUrl);
        WebResource service = client.resource(restFullUrl).path("dataserver");
        try {
            channelList = service.path("listchannels").accept(MediaType.TEXT_XML).get(DataChannel.DataChannelList.class);
            lookup.add(channelList);
            lookup.add(service, "dataserver");
            this.service = service;
            logger.log(Level.INFO, "Connection successful, {0} channels read", channelList.list.size());
        } catch (Exception e) {
            logger.log(Level.WARNING, "Connection unsuccessful", e);
        }
    }
}
