package org.lsst.ccs.plugin.jas3.console;

import java.io.IOException;
import java.io.PrintWriter;
import java.net.URL;
import org.freehep.application.studio.Plugin;
import org.freehep.application.studio.Studio;
import org.freehep.jas.plugin.console.Console;
import org.freehep.jas.plugin.console.ConsoleOutputStream;
import org.freehep.jas.plugin.console.ConsoleService;
import org.freehep.util.FreeHEPLookup;
import org.freehep.util.commanddispatcher.CommandProcessor;
import org.freehep.xml.menus.XMLMenuBuilder;
import org.lsst.ccs.bus.BusMessagingLayer;
import org.lsst.ccs.bus.MessagingFactory;
import org.xml.sax.SAXException;

/**
 * This plugin deals with creating LSST CCS tracers and consoles.
 * @author tonyj
 */
public class LsstConsolePlugin extends Plugin {

    private String defaultConsoleName = "LSST Console";
    private String defaultTracerName = "LSST Tracer";
    private MessagingFactory messagingFactory;

    @Override
    protected void init() throws SAXException, IOException {

        Studio studio = getApplication();
        FreeHEPLookup lookup = studio.getLookup();
        lookup.add(this);

        studio.getCommandTargetManager().add(new ConsoleCommands());

        XMLMenuBuilder builder = studio.getXMLMenuBuilder();
        URL xml = getClass().getResource("LsstConsolePlugin.menus");
        builder.build(xml);
    }

    @Override
    protected void postInit() {
        super.postInit();
        messagingFactory = MessagingFactory.getInstance().forSubsystem(BusMessagingLayer.ANONYMOUS_AGENT);
        messagingFactory.addStatusListener(new AlarmListener(getApplication()));
    }
    
   
    private Console createCommandConsole(String name) {
        Studio studio = getApplication();
        try {
            FreeHEPLookup lookup = studio.getLookup();
            ConsoleService cs = (ConsoleService) lookup.lookup(ConsoleService.class);
            if (cs != null) {
                Console console = cs.createConsole(name, null);
                final CommandConsole command = new CommandConsole(console);
                Thread t = new Thread("ConsoleInput-" + name) {
                    @Override
                    public void run() {
                        command.start();
                        // This will not return until the console is closed
                        // and the CommandConsole has been shutdown
                        command.runConsole();
                    }
                };
                t.start();
                return console;
            }
        } catch (Exception e) {
            studio.error("Failed to start Console: " + name, e);
        }
        return null;
    }

    private Console createTracerConsole(String name) {
        FreeHEPLookup lookup = getApplication().getLookup();
        ConsoleService cs = (ConsoleService) lookup.lookup(ConsoleService.class);
        if (cs != null) {
            Tracer tracer = new Tracer(messagingFactory);
            Console console = cs.createConsole(name, null, tracer);
            ConsoleOutputStream out = console.getOutputStream(null);
            PrintWriter pw = new PrintWriter(out, true);
            cs.getPageContextForConsole(console).addPageListener(tracer);
            tracer.start(pw); // This returns immediately
            return console;
        }
        return null;
    }

    public class ConsoleCommands extends CommandProcessor {

        public void onLsstConsole() {
            createCommandConsole(defaultConsoleName);
        }

        public void onLsstTracer() {
            createTracerConsole(defaultTracerName);
        }
    }
}
