package org.lsst.ccs.subsystem.vacuum;

import java.util.ArrayList;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.Measurement;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.subsystem.common.actions.MpmAction;
import org.lsst.ccs.subsystem.vacuum.constants.VacuumAlert;
import org.lsst.ccs.subsystem.vacuum.constants.ConditionState;
import org.lsst.ccs.subsystem.vacuum.constants.Conditions;
import org.lsst.ccs.subsystem.vacuum.constants.DeviceState;
import org.lsst.ccs.subsystem.vacuum.constants.Devices;
import org.lsst.ccs.subsystem.vacuum.constants.LatchState;
import org.lsst.ccs.subsystem.vacuum.constants.Latches;
import org.lsst.ccs.subsystem.vacuum.constants.MonitorControl;
import org.lsst.ccs.subsystem.vacuum.constants.PLCState;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchEnable;
import org.lsst.ccs.subsystem.vacuum.constants.Switches;
import org.lsst.ccs.subsystem.vacuum.constants.SwitchState;
import org.lsst.ccs.subsystem.vacuum.constants.VacuumAgentProperties;
import org.lsst.ccs.subsystem.vacuum.constants.CryoVacuumState;
import org.lsst.ccs.subsystem.vacuum.constants.HxVacuumState;
import org.lsst.ccs.subsystem.vacuum.data.VacuumException;
import org.lsst.ccs.subsystem.vacuum.data.VacSysState;

/**
 * The camera vacuum system
 *
 * @author The LSST CCS Team
 */
public class VacuumMain extends Subsystem implements HasLifecycle, AlertListener {

    static class SensorData {
        Channel channel;
        int maxErrors;
        int numErrors = 0;
        boolean getLimits = false;
        double value = 0.0;
        double goodValue = Double.NaN;
        double lowLimit;
        double highLimit;

        private SensorData(Channel channel, int maxErrors, boolean getLimits) {
            this.channel = channel;
            this.maxErrors = maxErrors;
            this.getLimits = getLimits;
        }

        private SensorData(Channel channel, int maxErrors) {
            this(channel, maxErrors, false);
        }
    }

    private static final double
        PRESS_ATMOS = 759.0,
        SPEED_CRYO_TURBO_MAX = 60000,
        SPEED_HX_TURBO_MAX = 81000;
    private static final int
        MAX_PRESS_ERRORS = 2,
        MAX_SPEED_ERRORS = 0,
        MAX_TEMP_ERRORS = 0,
        MAX_POWER_ERRORS = 0,
        MAX_AIR_ERRORS = 0,
        DELAY_ION_OFF = 10000,
        DELAY_VACUUM_BAD = 10000;
    private static final Map<Integer, VacuumAlert> latchAlertMap = new HashMap<>();
    static {
        latchAlertMap.put(Latches.LATCH_CR_VACUUM, VacuumAlert.CRYO_VACUUM_BAD);
        latchAlertMap.put(Latches.LATCH_CR_GATE_NFC, VacuumAlert.CRYO_GATE_FORCED_SHUT);
        latchAlertMap.put(Latches.LATCH_CR_GATE_AO, VacuumAlert.CRYO_GATE_CANNOT_OPEN);
        latchAlertMap.put(Latches.LATCH_CR_PUMP, VacuumAlert.CRYO_TURBO_PUMP_BAD);
        latchAlertMap.put(Latches.LATCH_HX_VACUUM, VacuumAlert.HX_VACUUM_BAD);
        latchAlertMap.put(Latches.LATCH_HX_GATE_NFC, VacuumAlert.HX_GATE_FORCED_SHUT);
        latchAlertMap.put(Latches.LATCH_HX_GATE_AO, VacuumAlert.HX_GATE_CANNOT_OPEN);
        latchAlertMap.put(Latches.LATCH_HX_PUMP, VacuumAlert.HX_TURBO_PUMP_BAD);
    }
    private static final Map<String, Integer> revLatchAlertMap = new HashMap<>();
    static {
        for (int cond : latchAlertMap.keySet()) {
            revLatchAlertMap.put(latchAlertMap.get(cond).getId(), cond);
        }
    }
    private static final List<Integer> cryoIonPumps = new ArrayList<>();
    static {
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP1);
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP2);
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP3);
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP4);
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP5);
        cryoIonPumps.add(Switches.SW_CRYO_ION_PUMP6);
    }
    private static final List<Integer> hxIonPumps = new ArrayList<>();
    static {
        hxIonPumps.add(Switches.SW_HX_ION_PUMP1);
        hxIonPumps.add(Switches.SW_HX_ION_PUMP2);
    }
    private static final List<Integer> utSwitches = new ArrayList<>();
    static {
        utSwitches.add(Switches.SW_INST_FTH_VALVE);
        utSwitches.add(Switches.SW_INST_FTPP_VALVE);
        utSwitches.add(Switches.SW_INST_L3H_VALVE);
        utSwitches.add(Switches.SW_INST_L3LF_VALVE);
        utSwitches.add(Switches.SW_INST_SCROLL_PUMP);
        utSwitches.add(Switches.SW_CRYO_SCROLL_PUMP);
        utSwitches.add(Switches.SW_HX_SCROLL_PUMP);
    }

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;

    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private final List<SwitchDevice> switchDevcs = new ArrayList<>();

    // From Groovy file
    private Channel cryoVacPressure, cryoTurboPressure, cryoTurboTemp, cryoTurboPower, cryoFlinePressure, cryoTurboSpeed,
                    hxVacPressure, hxTurboPressure, hxTurboTemp, hxTurboPower, hxFlinePressure, hxTurboSpeed, airPressure;

    // Configuration parameters
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("0.0..0.5"), units = "Torr")
    private volatile double pressTurboLow; // 0.5 max
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("0.0..0.5"), units = "Torr")
    private volatile double pressForelineLow; // 0.5 max
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("0.00..0.09"), units = "Torr")
    private volatile double pressDiffLow; // 0.09 max
    @ConfigurationParameter(category="Vacuum", isFinal=true)
    private volatile double pressDiffHigh; // 20.0
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("1.0e-6..1.1e-5"), units = "Torr")
    private volatile double pressIonOff; // 1.1e-5 max
    @ConfigurationParameter(category="Vacuum", isFinal=true)
    private volatile double pressIonEnable; // 1.0e-6
    @ConfigurationParameter(category="Vacuum", isFinal=true)
    private volatile double pressVacuum; // 1.0e-7
    @ConfigurationParameter(category="Vacuum", isFinal=true)
    private volatile double pressRefrigOk; // 9.0e-3
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("0.0..0.1"))
    private volatile double speedFractTurboLow; // 0.1
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("6.0e-5..2.0e-4"), units = "Torr")
    private volatile double pressCCEnable; // 2.0e-4
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("1.2e-5..1.2e-4"), units = "Torr")
    private volatile double pressCCOff; // 2e-5
    @ConfigurationParameter(category="Vacuum", isFinal=false, range = ("0..300000"), units = "ms")
    private volatile double delayCCOff; // ~20000 ops, 120000 pump-down
    @ConfigurationParameter(category="Vacuum", isFinal=true, units = "ms")
    private volatile long tranTimeCryoGate; // 2000
    @ConfigurationParameter(category="Vacuum", isFinal=true, units = "ms")
    private volatile long tranTimeHxGate; // 2000

    // General
    private static final Logger LOG = Logger.getLogger(VacuumMain.class.getName());
    private double speedCryoTurboLow, speedHxTurboLow;
    private MonitorTaskControl monitorControl;
    private final SwitchDevice[] switchDevices = new SwitchDevice[Devices.NUM_DEVICES];
    private VacPlutoDevice vacPlutoDevc;
    private IonPumpDevice ionPumpDevc;
    private VacTurboDevice cryoTurboDevc, hxTurboDevc;
    private final VacSysState vacState = new VacSysState();
    private final Map<String, Boolean> activeAlarmMap = new HashMap<>();
    private long cryoIonOverStartTime = 0, hxIonOverStartTime = 0;
    private long cryoVacCCOverStartTime = 0, hxVacCCOverStartTime = 0;
    private long cryoTurboCCOverStartTime = 0, hxTurboCCOverStartTime = 0;
    private long vacBadTime = 0;
    private SensorData cryoFlinePrs, cryoMainPrs, cryoTurboPrs, cryoTurboSpd, cryoTurboTmp, cryoTurboPwr,
                       hxFlinePrs, hxMainPrs, hxTurboPrs, hxTurboSpd, hxTurboTmp, hxTurboPwr, cmpAirPrs;
    private Map<Integer, Long> gateTransitStart = new HashMap<>();
    private boolean gotCommand = false;


    /**
     *  Constructor
     */
    public VacuumMain() {
        super("vacuum", AgentInfo.AgentType.WORKER);
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, MonitorControl.NODE_NAME);

        // Create and schedule an AgentPeriodicTask to update the vacuum state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("vacuum-state", () -> updateVacuumState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);

        // Register vacuum states
        stateService.registerState(CryoVacuumState.class, "Cryo vacuum state", this);
        stateService.updateAgentState(CryoVacuumState.UNKNOWN);
        stateService.registerState(HxVacuumState.class, "HX vacuum state", this);
        stateService.updateAgentState(HxVacuumState.UNKNOWN);

        // Initialize active alert map
        for (VacuumAlert alert : VacuumAlert.values()) {
            activeAlarmMap.put(alert.getId(), false);
        }
    }

    /**
     * Init phase.
     * We register here all the Alerts raised by this Subsystem.
     */
    @Override
    public void init() {
        for ( VacuumAlert alert : VacuumAlert.values() ) {
            alertService.registerAlert(alert.getAlert());
        }
    }

    /**
     *  Post initialization phase
     */
    @Override
    public void postInit() {

        //Set a property to define that this Agent is a vacuum subsystem.
        propertiesService.setAgentProperty(VacuumAgentProperties.VACUUM_TYPE, VacuumMain.class.getCanonicalName());

        // Add alert listener
        alertService.addListener(this);

        for (SwitchDevice device : switchDevcs) {
            int id = device.getSwitchDevice();
            if (id < Devices.DEVC_NO_SWITCH || id >= Devices.NUM_DEVICES) {
                throw new RuntimeException("Device of class " + device.getClass().getSimpleName() + " has invalid id: " + id);
            }
            if (id != Devices.DEVC_NO_SWITCH) {
                switchDevices[id] = device;
            }
        }
        for (int id = 0; id < Devices.NUM_DEVICES; id++) {
            if (switchDevices[id] == null) {
                throw new RuntimeException(Devices.getDescription(id) + " device not defined");
            }
        }
        vacPlutoDevc = (VacPlutoDevice)switchDevices[Devices.DEVC_PLUTO];
        ionPumpDevc = (IonPumpDevice)switchDevices[Devices.DEVC_ION_PUMP];
        cryoTurboDevc = (VacTurboDevice)switchDevices[Devices.DEVC_CRYO_TURBO_PUMP];
        hxTurboDevc = (VacTurboDevice)switchDevices[Devices.DEVC_HX_TURBO_PUMP];

        if (cryoFlinePressure == null) {
            throw new RuntimeException("Cryo foreline pressure channel (cryoFlinePressure) not specified");
        }
        cryoFlinePrs = new SensorData(cryoFlinePressure, MAX_PRESS_ERRORS);
        if (cryoVacPressure == null) {
            throw new RuntimeException("Cryo vacuum pressure channel (cryoVacPressure) not specified");
        }
        cryoMainPrs = new SensorData(cryoVacPressure, MAX_PRESS_ERRORS);
        ionPumpDevc.setCryoPressureChannel(cryoVacPressure);
        if (cryoTurboPressure == null) {
            throw new RuntimeException("Cryo turbo pump pressure channel (cryoTurboPressure) not specified");
        }
        cryoTurboPrs = new SensorData(cryoTurboPressure, MAX_PRESS_ERRORS);
        if (cryoTurboSpeed == null) {
            throw new RuntimeException("Cryo turbo pump speed channel (cryoTurboSpeed) not specified");
        }
        cryoTurboSpd = new SensorData(cryoTurboSpeed, MAX_SPEED_ERRORS);
        if (cryoTurboTemp == null) {
            throw new RuntimeException("Cryo turbo pump temperature channel (cryoTurboTemp) not specified");
        }
        cryoTurboTmp = new SensorData(cryoTurboTemp, MAX_TEMP_ERRORS, true);
        if (cryoTurboPower == null) {
            throw new RuntimeException("Cryo turbo pump power channel (cryoTurboPower) not specified");
        }
        cryoTurboPwr = new SensorData(cryoTurboPower, MAX_POWER_ERRORS, true);
        if (hxFlinePressure == null) {
            throw new RuntimeException("HX foreline pressure channel (hxFlinePressure) not specified");
        }
        hxFlinePrs = new SensorData(hxFlinePressure, MAX_PRESS_ERRORS);
        if (hxVacPressure == null) {
            throw new RuntimeException("HX vacuum pressure channel (hxVacPressure) not specified");
        }
        hxMainPrs = new SensorData(hxVacPressure, MAX_PRESS_ERRORS);
        ionPumpDevc.setHxPressureChannel(hxVacPressure);
        if (hxTurboPressure == null) {
            throw new RuntimeException("HX turbo pump pressure channel (hxTurboPressure) not specified");
        }
        hxTurboPrs = new SensorData(hxTurboPressure, MAX_PRESS_ERRORS);
        if (hxTurboSpeed == null) {
            throw new RuntimeException("HX turbo pump speed channel (hxTurboSpeed) not specified");
        }
        hxTurboSpd = new SensorData(hxTurboSpeed, MAX_SPEED_ERRORS);
        if (hxTurboTemp == null) {
            throw new RuntimeException("HX turbo pump temperature channel (hxTurboTemp) not specified");
        }
        hxTurboTmp = new SensorData(hxTurboTemp, MAX_TEMP_ERRORS);
        if (hxTurboPower == null) {
            throw new RuntimeException("HX turbo pump power channel (hxTurboPower) not specified");
        }
        hxTurboPwr = new SensorData(hxTurboPower, MAX_POWER_ERRORS);
        if (airPressure == null) {
            throw new RuntimeException("Compressed air pressure channel (airPressure) not specified");
        }
        cmpAirPrs = new SensorData(airPressure, MAX_AIR_ERRORS, true);
        speedCryoTurboLow = speedFractTurboLow * SPEED_CRYO_TURBO_MAX;
        speedHxTurboLow = speedFractTurboLow * SPEED_HX_TURBO_MAX;
    }


    /**
     *  Post start
     */
    @Override
    public void postStart() {
        LOG.info("Vacuum subsystem started");
    }


    /**
     *  Gets the state of the Vacuum system.
     *
     *  @return  The vacuum state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the vacuum system state", level=0)
    public VacSysState getVacuumState() {
        vacState.setTickMillis(monitorControl.getFastPeriod());
        return vacState;
    }    


    /**
     *  Gets the list of switch names.
     *
     *  @return  The switch names.
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.QUERY, description="Get switch names", level=0)
    public List<String> getSwitchNames() throws VacuumException
    {
        return Switches.getNames();
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  name  The switch name.
     *  @param  on    Whether to turn on or off
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a named switch")
    public void setSwitchOn(@Argument(description="The switch name") String name,
                            @Argument(description="Whether to turn on") boolean on) throws VacuumException
    {
        gotCommand = true;
        setSwitch(Switches.getId(name), on);
    }


    /**
     *  Returns enable and state of switch
     *
     *  @param  name  The switch name.
     *  @throws  VacuumException
     *  @return  The enabled and switch states, as a string
     */
    @Command(type=Command.CommandType.QUERY, description="Print enable and state for switch")
    public String checkSwitch(@Argument(description="The switch name") String name) throws VacuumException
    {
        int sw = Switches.getId(name);
        SwitchEnable enable = vacState.getSwitchEnable(sw);
        SwitchState state = vacState.getSwitchState(sw);
        return "enable="+ enable.name() +", state="+state.name();
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  Whether to turn on or off
     *  @throws  VacuumException
     */
    private void setSwitch(int sw, boolean on) throws VacuumException
    {
        SwitchState state = vacState.getSwitchState(sw);
        if (state == SwitchState.OFFLINE) return;
        SwitchEnable enable = vacState.getSwitchEnable(sw);
        if (on && enable != SwitchEnable.ON) return;
        try {
            switchDevices[Switches.getDevice(sw)].setSwitch(SwitchInfo.getSwitch(sw), on);
        }
        catch (DriverException e) {
            throw new VacuumException("Error setting switch for " + Devices.getDescription(Switches.getDevice(sw)) + " device: " + e);
        }
    }


    /**
     *  Gets whether a switch is on.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     *  @throws  VacuumException
     */
    private Boolean isSwitchOn(int sw)
    {
        return switchDevices[Switches.getDevice(sw)].isSwitchOn(SwitchInfo.getSwitch(sw));
    }


    /**
     *  Clears a latched condition.
     *
     *  @param  cond  The condition number.
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Clear a condition")
    public void clearLatch(@Argument(description="The condition number") int cond) throws VacuumException
    {
        gotCommand = true;
        vacPlutoDevc.clearLatch(cond);
    }


    /**
     *  Updates the vacuum system state periodically.
     *
     *  The vacuum state consists mainly of the state of the switches (lines) being
     *  controlled, along with whether they can be turned on.
     *
     *  Pressures and pump speeds are read to determine which switches are enabled
     *  and/or need to be turned off.  If the state changes it is published.
     */
    private void updateVacuumState()
    {
        boolean changed = monitorControl.hasPeriodChanged();
        changed |= updatePlcState();
        changed |= updateLatchState();
        changed |= updateCondState();
        if (readSensors()) {
            changed |= updateSwitchState();
            changed |= updateCryoState();
            changed |= updateHxState();
            checkRefrigVacuum();
        }
        if (gotCommand) {
            gotCommand = false;
            changed = true;
        }
        if (changed) {
            publishState();
        }
    }


    /**
     *  Updates the PLC state
     * 
     *  @return  Whether the state changed
     */
    private boolean updatePlcState() {
        boolean changed = false;
        Boolean plcActive = vacPlutoDevc.isPLCActive();
        if (plcActive == null) {
            if (vacState.getPlcState() != PLCState.OFFLINE) {
                raiseAlarm(VacuumAlert.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is offline", null);
                vacState.setPlcState(PLCState.OFFLINE);
                changed = true;
            }
        }
        else if (!plcActive) {
            if (vacState.getPlcState() != PLCState.DEAD) {
                raiseAlarm(VacuumAlert.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC has died", null);
                vacState.setPlcState(PLCState.DEAD);
                changed = true;
            }
        }
        else {
            if (vacState.getPlcState() != PLCState.ALIVE) {
                lowerAlarm(VacuumAlert.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is alive", null);
                vacState.setPlcState(PLCState.ALIVE);
                changed = true;
            }
        }
        return changed;
    }


    /**
     *  Updates the state of the PLC latches.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateLatchState() {
        boolean changed = false;
        for (int cond = 0; cond < Latches.NUM_LATCHES; cond++) {
            Boolean active = vacPlutoDevc.isLatchActive(cond);
            Boolean latched = vacPlutoDevc.isLatchLatched(cond);
            LatchState state = active == null || latched == null ? LatchState.OFFLINE :
                               latched ? LatchState.LATCHED :
                               active ? LatchState.ACTIVE : LatchState.CLEAR;
            LatchState oldState = vacState.getLatch(cond); 
            if (state != oldState) {
                vacState.setLatch(cond, state);
                VacuumAlert alert = latchAlertMap.get(cond);
                if (state == LatchState.ACTIVE) {
                    raiseAlarm(alert, "Vacuum PLC error condition set", null);
                }
                else if (state != LatchState.OFFLINE && oldState != LatchState.OFFLINE) {
                    if (state == LatchState.LATCHED && oldState == LatchState.CLEAR) {
                        raiseAlarm(alert, "Vacuum PLC error condition set", null);
                    }
                    if (state == LatchState.LATCHED || oldState == LatchState.ACTIVE) {
                        lowerAlarm(alert, "Vacuum PLC error condition cleared", null);
                    }
                }
                changed = true;
            }
        }
        return changed;
    }


    /**
     *  Updates the state of the PLC conditions.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCondState() {
        boolean changed = false;
        for (int cond = 0; cond < Conditions.NUM_CONDITIONS; cond++) {
            Boolean active = vacPlutoDevc.isConditionActive(cond);
            ConditionState state = active == null ? ConditionState.OFF :
                                   active ? ConditionState.YES : ConditionState.NO;
            if (state != vacState.getCondition(cond)) {
                vacState.setCondition(cond, state);
                changed = true;
            }
        }
        return changed;
    }


    /**
     *  Reads the gauges and the state of the turbo pumps.
     * 
     *  @return  Whether valid values were available for all devices
     */
    private boolean readSensors() {
        boolean valid = true;
        valid &= readSensor(cryoFlinePrs);
        valid &= readSensor(cryoMainPrs);
        valid &= readSensor(cryoTurboPrs);
        valid &= readSensor(cryoTurboSpd);
        valid &= readSensor(cryoTurboTmp);
        valid &= readSensor(cryoTurboPwr);
        valid &= readSensor(hxFlinePrs);
        valid &= readSensor(hxMainPrs);
        valid &= readSensor(hxTurboPrs);
        valid &= readSensor(hxTurboSpd);
        valid &= readSensor(hxTurboTmp);
        valid &= readSensor(hxTurboPwr);
        valid &= readSensor(cmpAirPrs);
        return valid;
    }


    /**
     *  Reads a sensor, allowing for possible temporary errors.
     * 
     *  @param  data  The sensor data object
     *  @return  Whether valid value was available 
     */
    private boolean readSensor(SensorData data) {
        Measurement meas = data.channel.getLastMeasurement();
        if (meas.getCCSTimestamp() == null) return false;
        data.value = meas.getValue();
        if (!Double.isNaN(data.value)) {
            data.goodValue = data.value;
            data.numErrors = 0;
        }
        else if (data.numErrors++ < data.maxErrors) {
            data.value = data.goodValue;
        }
        if (data.getLimits) {
            data.lowLimit = data.channel.getLimitLo();
            data.highLimit = data.channel.getLimitHi();
        }
        return true;
    }


    /**
     *  Updates the state of the switches.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateSwitchState() {
        boolean changed = false;
        changed |= updateCryoGateValve();
        changed |= updateCryoTurboPump();
        changed |= updateCryoIonPumps();
        changed |= updateHxGateValve();
        changed |= updateHxTurboPump();
        changed |= updateHxIonPumps();
        changed |= updateUtSwitches();
        changed |= updateCryoVacCC();
        changed |= updateCryoTurboCC();
        changed |= updateHxVacCC();
        changed |= updateHxTurboCC();
        return changed;
    }


    /**
     *  Updates the state of the cryo gate valve.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCryoGateValve()
    {
        boolean enable = false;
        boolean cryoTurbAvail = !Double.isNaN(cryoMainPrs.value) && !Double.isNaN(cryoTurboPrs.value) && !Double.isNaN(cryoTurboSpd.value);
        if (cryoTurbAvail) {
            double prDiff = Math.abs(cryoTurboPrs.value - cryoMainPrs.value);
            enable = ((prDiff <= pressDiffHigh && cryoTurboSpd.value <= speedCryoTurboLow)
                        || (prDiff <= pressDiffLow && cryoTurboSpd.value > speedCryoTurboLow))
                       && cmpAirPrs.value > cmpAirPrs.lowLimit && cmpAirPrs.value <= cmpAirPrs.highLimit;
        }
        boolean enable2 = cryoTurbAvail && (cryoFlinePrs.value < pressForelineLow || cryoTurboSpd.value <= speedCryoTurboLow)
                            && cryoTurboTmp.value < cryoTurboTmp.highLimit && cryoTurboPwr.value < cryoTurboPwr.highLimit;
        enable &= enable2;
        enable &= ((cryoMainPrs.value + 5.0e-5 + 0.026 * cryoTurboPrs.value) > cryoTurboPrs.value);  // p_in > p_out unless very low
        return updateSwitch(Switches.SW_CRYO_GATE_VALVE, enable, !enable2,
                            vacPlutoDevc.getSwitchState(SwitchInfo.getSwitch(Switches.SW_CRYO_GATE_VALVE)), VacuumAlert.CRYO_GATE_CLOSED);
    }


    /**
     *  Updates the state of the cryo turbo pump.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCryoTurboPump()
    {
        boolean enable = false;
        if (!Double.isNaN(cryoTurboPrs.value) && !Double.isNaN(cryoFlinePrs.value) && !Double.isNaN(cryoTurboSpd.value)) {
            enable = cryoTurboPrs.value < pressTurboLow
                       && (cryoFlinePrs.value < pressForelineLow || cryoTurboSpd.value < speedCryoTurboLow);
        }
        return updateSwitch(Switches.SW_CRYO_TURBO_PUMP, enable, !enable, cryoTurboDevc.getDeviceState(), VacuumAlert.CRYO_TURBO_PUMP_STOPPED);
    }


    /**
     *  Updates the state of the cryo ion pumps.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCryoIonPumps()
    {
        boolean changed = false, enable = false, turnOff = false;
        if (!Double.isNaN(cryoMainPrs.value)) {
            enable = cryoMainPrs.value < pressIonEnable;
            if (cryoMainPrs.value >= pressIonOff) {
                if (cryoIonOverStartTime == 0) {
                    cryoIonOverStartTime = System.currentTimeMillis();
                }
                else {
                    turnOff = System.currentTimeMillis() - cryoIonOverStartTime >= DELAY_ION_OFF;
                }
            }
            else {
                cryoIonOverStartTime = 0;
            }
        }
        else {
            turnOff = true;
        }
        for (int sw : cryoIonPumps) {
            changed |= updateSwitch(sw, enable, turnOff, null, VacuumAlert.CRYO_ION_PUMPS_STOPPED);
        }
        return changed;
    }


    /**
     *  Updates the state of the HX gate valve.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateHxGateValve()
    {
        boolean enable = false;
        boolean hxTurbAvail = !Double.isNaN(hxMainPrs.value) && !Double.isNaN(hxTurboPrs.value) && !Double.isNaN(hxTurboSpd.value);
        if (hxTurbAvail) {
            double prDiff = Math.abs(hxTurboPrs.value - hxMainPrs.value);
            enable = ((prDiff <= pressDiffHigh && hxTurboSpd.value <= speedHxTurboLow)
                        || (prDiff <= pressDiffLow && hxTurboSpd.value > speedHxTurboLow))
                       && cmpAirPrs.value > cmpAirPrs.lowLimit && cmpAirPrs.value <= cmpAirPrs.highLimit;
        }
        boolean enable2 = hxTurbAvail && (hxFlinePrs.value < pressForelineLow || hxTurboSpd.value <= speedHxTurboLow)
                            && hxTurboTmp.value < hxTurboTmp.highLimit && hxTurboPwr.value < hxTurboPwr.highLimit;
        enable &= enable2;
        enable &= ((hxMainPrs.value + 5.0e-5 + 0.026 * hxTurboPrs.value) > hxTurboPrs.value);  // p_in > p_out unless very low
        return updateSwitch(Switches.SW_HX_GATE_VALVE, enable, !enable2,
                            vacPlutoDevc.getSwitchState(SwitchInfo.getSwitch(Switches.SW_HX_GATE_VALVE)), VacuumAlert.HX_GATE_CLOSED);
    }


    /**
     *  Updates the state of the HX turbo pump.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateHxTurboPump()
    {
        boolean enable = false;
        if (!Double.isNaN(hxTurboPrs.value) && !Double.isNaN(hxFlinePrs.value) && !Double.isNaN(hxTurboSpd.value)) {
            enable = hxTurboPrs.value < pressTurboLow
                       && (hxFlinePrs.value < pressForelineLow || hxTurboSpd.value < speedHxTurboLow);
        }
        return updateSwitch(Switches.SW_HX_TURBO_PUMP, enable, !enable, hxTurboDevc.getDeviceState(), VacuumAlert.HX_TURBO_PUMP_STOPPED);
    }


    /**
     *  Updates the state of the HX ion pumps.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateHxIonPumps()
    {
        boolean changed = false, enable = false, turnOff = false;
        if (!Double.isNaN(hxMainPrs.value)) {
            enable = hxMainPrs.value < pressIonEnable;
            if (hxMainPrs.value >= pressIonOff) {
                if (hxIonOverStartTime == 0) {
                    hxIonOverStartTime = System.currentTimeMillis();
                }
                else {
                    turnOff = System.currentTimeMillis() - hxIonOverStartTime >= DELAY_ION_OFF;
                }
            }
            else {
                hxIonOverStartTime = 0;
            }
        }
        else {
            turnOff = true;
        }
        for (int sw : hxIonPumps) {
            changed |= updateSwitch(sw, enable, turnOff, null, VacuumAlert.HX_ION_PUMPS_STOPPED);
        }
        return changed;
    }


    /*
     *  Updates the state of the CryoVacGauge Cold Cathode
     * 
     *  @return  Whether the state changed
    */
    private boolean updateCryoVacCC()
    {
        boolean changed = false, enable = false, turnOff = false;
        if (!Double.isNaN(cryoMainPrs.value)) {
            enable = cryoMainPrs.value <= pressCCEnable;
            if (cryoMainPrs.value >= pressCCOff) {
                if (cryoVacCCOverStartTime == 0) {
                    cryoVacCCOverStartTime = System.currentTimeMillis();
                }
                else {
                    turnOff = System.currentTimeMillis() - cryoVacCCOverStartTime >= delayCCOff;
                }
            }
            else {
                cryoVacCCOverStartTime = 0;
            }
        }
        else {
            turnOff = true;
        }
        changed |= updateSwitch(Switches.SW_CRYO_VAC_CC, enable, turnOff, null, VacuumAlert.CRYO_VAC_CC_STOPPED);
        return changed;
    }


    /*
     *  Updates the state of the TurboVacGauge Cold Cathode
     * 
     *  @return  Whether the state changed
    */
    private boolean updateCryoTurboCC()
    {
        boolean changed = false, enable = false, turnOff = false;
        if (!Double.isNaN(cryoTurboPrs.value)) {
            enable = cryoTurboPrs.value < pressCCEnable;
            if (cryoTurboPrs.value >= pressCCOff) {
                if (cryoTurboCCOverStartTime == 0) {
                    cryoTurboCCOverStartTime = System.currentTimeMillis();
                }
                else {
                    turnOff = System.currentTimeMillis() - cryoTurboCCOverStartTime >= delayCCOff;
                }
            }
            else {
                cryoTurboCCOverStartTime = 0;
            }
        }
        else {
            turnOff = true;
        }
        changed |= updateSwitch(Switches.SW_CRYO_TURBO_CC, enable, turnOff, null, VacuumAlert.CRYO_TURBO_CC_STOPPED);
        return changed;
    }


    /*
     *  Updates the state of the HxVacGauge Cold Cathode
     * 
     *  @return  Whether the state changed
    */
    private boolean updateHxVacCC()
    {
        boolean changed = false, enable = false, turnOff = false;
        if (!Double.isNaN(hxMainPrs.value)) {
            enable = hxMainPrs.value < pressCCEnable;
            if (hxMainPrs.value >= pressCCOff) {
                if (hxVacCCOverStartTime == 0) {
                    hxVacCCOverStartTime = System.currentTimeMillis();
                }
                else {
                    turnOff = System.currentTimeMillis() - hxVacCCOverStartTime >= delayCCOff;
                }
            }
            else {
                hxVacCCOverStartTime = 0;
            }
        }
        else {
            turnOff = true;
        }
        changed |= updateSwitch(Switches.SW_HX_VAC_CC, enable, turnOff, null, VacuumAlert.HX_VAC_CC_STOPPED);
        return changed;
    }


    /*
     *  Updates the state of the TurboVacGauge Cold Cathode
     * 
     *  @return  Whether the state changed
    */
    private boolean updateHxTurboCC()
    {
        boolean changed = false, enable = false, turnOff = false;
        if (!Double.isNaN(hxTurboPrs.value)) {
            enable = hxTurboPrs.value < pressCCEnable;
            if (hxTurboPrs.value >= pressCCOff) {
                if (hxTurboCCOverStartTime == 0) {
                    hxTurboCCOverStartTime = System.currentTimeMillis();
                }
                else {
                    turnOff = System.currentTimeMillis() - hxTurboCCOverStartTime >= delayCCOff;
                }
            }
            else {
                hxTurboCCOverStartTime = 0;
            }
        }
        else {
            turnOff = true;
        }
        changed |= updateSwitch(Switches.SW_HX_TURBO_CC, enable, turnOff, null, VacuumAlert.HX_TURBO_CC_STOPPED);
        return changed;
    }


    /**
     *  Updates the state of the UT switches (scroll pumps & interstitial valves).
     * 
     *  @return  Whether the state changed
     */
    private boolean updateUtSwitches()
    {
        boolean changed = false;
        for (int sw : utSwitches) {
            changed |= updateSwitch(sw, true, false, null, null);
        }
        return changed;
    }


    /**
     *  Updates the state of a switch.
     * 
     *  @param sw        The switch ID
     *  @param enable    Whether to enable it
     *  @param turnOff   Whether to turn it off
     *  @param devState  The state of the switch's device (turbo pump only)
     *  @param alert     The alert to raise/lower if on/off state change was forced
     *  @return  Whether the state changed
     */
    private boolean updateSwitch(int sw, boolean enable, boolean turnOff, DeviceState devState, VacuumAlert alert)
    {
        boolean changed = false;
        SwitchState oldState = vacState.getSwitchState(sw);
        if (turnOff && oldState == SwitchState.ON) {
            try {
                setSwitch(sw, false);
                raiseAlarm(alert, "Switch was forced to open", null);
            }
            catch (VacuumException e) {
                LOG.log(Level.SEVERE, "Error setting switch: {0}", e);
            }
        }
        Boolean isOn = isSwitchOn(sw);
        SwitchState state = isOn != null ? isOn ? SwitchState.ON : SwitchState.OFF : SwitchState.OFFLINE;
        SwitchEnable enabled = enable ? SwitchEnable.ON : SwitchEnable.OFF;
        if (state != oldState || enabled != vacState.getSwitchEnable(sw)) {
            vacState.setSwitchState(sw, state);
            vacState.setSwitchEnable(sw, enabled);
            changed = true;
            if (enable && alert != null && isAlarmRaised(alert)) {
                lowerAlarm(alert, "Switch has become re-enabled", null);
            }
        }
        DeviceState oldDevState = vacState.getDeviceState(sw);
        if (devState != oldDevState) {
            if (oldDevState == DeviceState.ERROR) {
                lowerAlarm(sw == Switches.SW_CRYO_GATE_VALVE ? VacuumAlert.CRYO_GATE_POSN_ERROR : VacuumAlert.HX_GATE_POSN_ERROR,
                           "Gate valve no longer reading both open and closed", null);
            }
            else if (oldDevState == DeviceState.TRANSIT) {
                if (gateTransitStart.get(sw) == null) {
                    lowerAlarm(sw == Switches.SW_CRYO_GATE_VALVE ? VacuumAlert.CRYO_GATE_IN_TRANSIT : VacuumAlert.HX_GATE_IN_TRANSIT,
                               "Gate valve no longer in too-long transition", null);
                }
            }
            if (devState == DeviceState.ERROR) {
                raiseAlarm(sw == Switches.SW_CRYO_GATE_VALVE ? VacuumAlert.CRYO_GATE_POSN_ERROR : VacuumAlert.HX_GATE_POSN_ERROR,
                           "Gate valve reading both open and closed", null);
            }
            else if (devState == DeviceState.TRANSIT) {
                gateTransitStart.put(sw, System.currentTimeMillis());
            }
            vacState.setDeviceState(sw, devState);
            changed = true;
        }
        if (devState == DeviceState.TRANSIT) {
            Long transitTime = gateTransitStart.get(sw);
            if (transitTime != null
                  && (System.currentTimeMillis() - transitTime) > (sw == Switches.SW_CRYO_GATE_VALVE ? tranTimeCryoGate : tranTimeHxGate)) {
                raiseAlarm(sw == Switches.SW_CRYO_GATE_VALVE ? VacuumAlert.CRYO_GATE_IN_TRANSIT : VacuumAlert.HX_GATE_IN_TRANSIT,
                           "Gate valve in transition for too long", null);
                gateTransitStart.remove(sw);
            }
        }
        return changed;
    }


    /**
     *  Checks whether the vacuum is good enough to allow refrigeration.
     */
    private void checkRefrigVacuum() {
        boolean haveAlarm = isAlarmRaised(VacuumAlert.REFRIG_NOT_PERMITTED);
        if (cryoMainPrs.value < pressRefrigOk && hxMainPrs.value < pressRefrigOk) {  // Fails if either is NaN
            if (haveAlarm) {
                lowerAlarm(VacuumAlert.REFRIG_NOT_PERMITTED, "Cryo & HX vacuums are good (< " + pressRefrigOk + ")",
                           MpmAction.Action.BLOCK_REFRIG);
            }
            vacBadTime = 0;
        }
        else {
            long time = System.currentTimeMillis();
            if (vacBadTime == 0) {
                vacBadTime = time;
            }
            else {
                if (time - vacBadTime >= DELAY_VACUUM_BAD) {
                    if (!haveAlarm) {
                        raiseAlarm(VacuumAlert.REFRIG_NOT_PERMITTED, "Cryo or HX vacuum is bad (>= " + pressRefrigOk + ")",
                                   MpmAction.Action.BLOCK_REFRIG);
                    }
                }
            }
        }
    }


    /**
     *  Updates the summary cryo vacuum state.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateCryoState() {
        boolean changed = false;
        CryoVacuumState cvState;
        if (Double.isNaN(cryoMainPrs.value)) {
            cvState = CryoVacuumState.UNKNOWN;
        }
        else if (cryoMainPrs.value <= pressVacuum) {
            cvState = CryoVacuumState.VACUUM;
        }
        else if (vacState.getSwitchState(Switches.SW_CRYO_ION_PUMP1) == SwitchState.ON) {
            cvState = CryoVacuumState.ION_ON;
        }
        else if (vacState.getSwitchEnable(Switches.SW_CRYO_ION_PUMP1) == SwitchEnable.ON) {
            cvState = CryoVacuumState.ION_OFF;
        }
        else if (vacState.getSwitchState(Switches.SW_CRYO_TURBO_PUMP) == SwitchState.ON) {
            cvState = CryoVacuumState.TURBO_ON;
        }
        else if (vacState.getSwitchEnable(Switches.SW_CRYO_TURBO_PUMP) == SwitchEnable.ON) {
            cvState = CryoVacuumState.TURBO_OFF;
        }
        else if (cryoMainPrs.value < PRESS_ATMOS) {
            cvState = CryoVacuumState.FORELINE;
        }
        else {
            cvState = CryoVacuumState.OFF;
        }
        if (cvState != vacState.getCryoVacuumState()) {
            vacState.setCryoVacuumState(cvState);
            stateService.updateAgentState(cvState);
            changed = true;
        }
        return changed;
    }


    /**
     *  Updates the summary HX vacuum state.
     * 
     *  @return  Whether the state changed
     */
    private boolean updateHxState() {
        boolean changed = false;
        HxVacuumState hvState;
        if (Double.isNaN(hxMainPrs.value)) {
            hvState = HxVacuumState.UNKNOWN;
        }
        else if (hxMainPrs.value <= pressVacuum) {
            hvState = HxVacuumState.VACUUM;
        }
        else if (vacState.getSwitchState(Switches.SW_HX_ION_PUMP1) == SwitchState.ON) {
            hvState = HxVacuumState.ION_ON;
        }
        else if (vacState.getSwitchEnable(Switches.SW_HX_ION_PUMP1) == SwitchEnable.ON) {
            hvState = HxVacuumState.ION_OFF;
        }
        else if (vacState.getSwitchState(Switches.SW_HX_TURBO_PUMP) == SwitchState.ON) {
            hvState = HxVacuumState.TURBO_ON;
        }
        else if (vacState.getSwitchEnable(Switches.SW_HX_TURBO_PUMP) == SwitchEnable.ON) {
            hvState = HxVacuumState.TURBO_OFF;
        }
        else if (hxMainPrs.value < PRESS_ATMOS) {
            hvState = HxVacuumState.FORELINE;
        }
        else {
            hvState = HxVacuumState.OFF;
        }
        if (hvState != vacState.getHxVacuumState()) {
            vacState.setHxVacuumState(hvState);
            stateService.updateAgentState(hvState);
            changed = true;
        }
        return changed;
    }


    /**
     *  Raises an alarm.
     *
     *  @param  alert  The vacuum alert to raise to alarm state
     *  @param  cond   The alert condition
     *  @param  action  The MPM action to perform, or null if none
     */
    private void raiseAlarm(VacuumAlert alert, String cond, MpmAction.Action action)
    {
        Alert al = alert.getAlert();
        if (action != null) {
            MpmAction.addData(al, action);
        }
        alertService.raiseAlert(al, AlertState.ALARM, cond);
        activeAlarmMap.put(alert.getId(), true);
    }


    /**
     *  Lowers an alarm.
     *
     *  @param  alert  The vacuum alert to lower to normal state
     *  @param  cond   The alert condition
     *  @param  action  The MPM action to perform, or null if none
     */
    private void lowerAlarm(VacuumAlert alert, String cond, MpmAction.Action action)
    {
        Alert al = alert.getAlert();
        if (action != null) {
            MpmAction.addData(al, action);
        }
        alertService.raiseAlert(al, AlertState.NOMINAL, cond);
        activeAlarmMap.put(alert.getId(), false);
    }


     /**
     *  Checks whether an alert is in alarm state.
     *
     *  @param  alert  The vacuum alert to check
     *  @return  Whether it has been raised
     */
    private boolean isAlarmRaised(VacuumAlert alert)
    {
        return activeAlarmMap.get(alert.getId()) == Boolean.TRUE;
    }


    /**
     *  Alert event handler.
     *
     *  Resets PLC latch when corresponding alert is cleared.
     *
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event)
    {
        if (event.getType() != AlertEvent.AlertEventType.ALERT_CLEARED) return;
        for (String id : event.getClearedIds()) {
            Integer cond = revLatchAlertMap.get(id);
            if (cond != null) {
                try {
                    clearLatch(cond);
                }
                catch (VacuumException e) {
                    LOG.log(Level.SEVERE, "Error clearing latched PLC condition ({0}): {1}", new Object[]{cond, e});
                }
            }
        }
    }


    /**
     *  Publishes the state of the vacuum system.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        publishSubsystemDataOnStatusBus(new KeyValueData(VacSysState.KEY, getVacuumState()));
    }

}
